package com.digiwin.dap.nest.infrastructure.middleware.nacos;

import com.digiwin.dap.nest.kernel.core.config.DwEnvProperty;
import com.digiwin.dap.nest.kernel.core.config.DwPrePropertyHandler;
import com.digiwin.dap.nest.kernel.core.config.DwProperty;
import com.digiwin.dap.nest.kernel.core.config.processor.DwPropertiesProcessor;
import com.digiwin.dap.nest.kernel.core.dapper.log.DwLog;
import com.digiwin.dap.nest.kernel.core.util.DwStringUtil;
import com.digiwin.dap.nest.kernel.core.util.datastructure.DwCollectionUtil;
import com.digiwin.dap.nest.kernel.core.util.reflect.clazz.JaClassUtil;

import java.util.Properties;

import static com.digiwin.dap.nest.infrastructure.middleware.nacos.config.meta.DwNacosConstants.KeyArray;
import static com.digiwin.dap.nest.infrastructure.middleware.nacos.config.meta.DwNacosConstants.KeyPassword;
import static com.digiwin.dap.nest.infrastructure.middleware.nacos.config.meta.DwNacosConstants.KeyServerAddr;
import static com.digiwin.dap.nest.infrastructure.middleware.nacos.config.meta.DwNacosConstants.KeyUsername;

/**
 * 加载 nacos 所需要的配置
 *
 * @author chenjian
 * @since 2024年06月14日 17:42:12
 */
public class DwNacosPrePropertyHandler implements DwPrePropertyHandler {
    @Override
    public void addAndCover() {
        try {
            /**
             * {@link com.alibaba.nacos.client.utils.AppNameUtils.PARAM_MARKING_PROJECT}
             */
            System.setProperty("project.name", DwEnvProperty.getApplicationName());
            loadPropertiesByEnv();
            loadPropertiesByLocal();
            wrapDiscovery();
        } catch (Throwable e) {
            DwLog.error("nacos config init error", e);
        }
    }

    @Override
    public int order() {
        return -100;
    }

    /**
     * nacos地址配置在防腐层的resources中
     */
    public static void loadPropertiesByLocal(String... env) {
        if (DwStringUtil.isEmpty(DwProperty.get(KeyServerAddr))) {
            String path = String.format("nacos/%s.properties", DwCollectionUtil.isEmpty(env) ? DwEnvProperty.getEnv() : env[0]);
            Properties properties = DwPropertiesProcessor.readByResources(path);
            if (null != properties) {
                properties.forEach((o, o2) -> DwProperty.getPropertyMap().put(o.toString(), o2));
            } else {
                DwLog.warn("nacos loadPropertiesByLocal empty");
            }
        }
    }

    /**
     * nacos地址配置在 os env中
     */
    public static void loadPropertiesByEnv() {
        if (DwStringUtil.isSafeNotEmpty(System.getenv(KeyServerAddr))) {
            for (String key : KeyArray) {
                DwProperty.getPropertyMap().put(key, System.getenv(key));
            }
        }
    }

    private void wrapDiscovery() {
        if (JaClassUtil.hashClass("com.alibaba.nacos.api.naming.NamingService")) {
            if (DwStringUtil.isSafeNotEmpty(DwProperty.get(KeyServerAddr))) {
                addAndCover("spring.cloud.nacos.server-addr", DwProperty.get(KeyServerAddr));
                addAndCover("spring.cloud.nacos.username", DwProperty.get(KeyUsername));
                addAndCover("spring.cloud.nacos.password", DwProperty.get(KeyPassword));
            }
        }
    }
}
