package com.digiwin.dap.nest.infrastructure.middleware.rabbitmq.spring;

import com.digiwin.dap.nest.kernel.core.context.DwMapContextChain;
import com.digiwin.dap.nest.kernel.core.context.biz.DwCoreContext;
import com.digiwin.dap.nest.kernel.core.dapper.JaDapper;
import com.digiwin.dap.nest.kernel.core.dapper.log.DwLog;
import com.digiwin.dap.nest.kernel.core.dapper.meta.NodeKind;
import com.digiwin.dap.nest.kernel.core.dapper.meta.NodeSpan;
import com.digiwin.dap.nest.kernel.core.util.bytecode.aop.JaAspectUtil;
import com.digiwin.dap.nest.kernel.core.util.concurrent.DwThreadLocal;
import com.digiwin.dap.nest.kernel.core.util.datastructure.JaCollectionUtil;
import com.digiwin.dap.nest.kernel.core.util.datastructure.JaMapUtil;
import com.digiwin.dap.nest.kernel.core.util.io.serialize.json.DwJson;
import com.digiwin.dap.nest.kernel.core.util.reflect.JaReflectUtil;
import com.digiwin.dap.nest.kernel.core.util.unsafe.JaUnsafe;
import com.digiwin.dap.nest.infrastructure.spring.util.DwSpringAopUtil;
import com.digiwin.dap.nest.infrastructure.spring.util.DwSpringBeanUtil;
import com.rabbitmq.client.Channel;
import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.springframework.amqp.core.Message;
import org.springframework.amqp.core.MessageProperties;
import org.springframework.amqp.rabbit.listener.AbstractMessageListenerContainer;
import org.springframework.amqp.rabbit.listener.MessageListenerContainer;
import org.springframework.amqp.rabbit.listener.RabbitListenerEndpointRegistry;
import org.springframework.amqp.rabbit.listener.SimpleMessageListenerContainer;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.lang.Nullable;

import java.lang.reflect.Field;
import java.util.Map;

@Order(10)
@Configuration
@ConditionalOnClass(name = {"org.springframework.amqp.rabbit.annotation.EnableRabbit"})
public class DwNodeSpanRabbitListenerConsumer
//        implements ApplicationListener<ContextRefreshedEvent>
{

    public static final DwThreadLocal<Message> MessageThreadLocal = new DwThreadLocal<>();
    public static final DwThreadLocal<Channel> ChannelThreadLocal = new DwThreadLocal<>();

    //    @Override
    public void onApplicationEvent(@Nullable ContextRefreshedEvent event) {
        RabbitListenerEndpointRegistry bean = DwSpringBeanUtil.getBean(RabbitListenerEndpointRegistry.class);
        if (null == bean) {
            return;
        }
        Map<String, MessageListenerContainer> listenerContainers = JaReflectUtil.getFieldValue(bean, "listenerContainers");
        if (JaCollectionUtil.isEmpty(listenerContainers)) {
            return;
        }
        JaMessagingMessageListenerAdapterInterceptor methodInterceptor = new JaMessagingMessageListenerAdapterInterceptor();
        listenerContainers.forEach((k, v) -> {
            if (v instanceof SimpleMessageListenerContainer) {
                Object messageListener = ((SimpleMessageListenerContainer) v).getMessageListener();
                Object messageListenerProxy = DwSpringAopUtil.getProxy(messageListener, methodInterceptor);
                Field messageListenerField = JaReflectUtil.getField(AbstractMessageListenerContainer.class, "messageListener");
                JaUnsafe.setObjectFieldValue(v, messageListenerProxy, messageListenerField);
            }
        });
    }

    @Bean
    public JaRabbitListenerAspect jaRabbitListenerAspect() {
        return new JaRabbitListenerAspect();
    }

    /**
     * @author chenjian
     * @see MessagingMessageListenerAdapter#onMessage(org.springframework.amqp.core.Message, com.rabbitmq.client.Channel)
     * @since 2024年06月03日 20:36:53
     */
    static class JaMessagingMessageListenerAdapterInterceptor implements MethodInterceptor {

        private static final String onMessageMethodName = "onMessage";

        @Override
        public Object invoke(MethodInvocation invocation) throws Throwable {
            if (invocation.getMethod().getName().equals(onMessageMethodName)) {
                try {
                    MessageThreadLocal.set((Message) invocation.getArguments()[0]);
                    ChannelThreadLocal.set((Channel) invocation.getArguments()[1]);
                    return invocation.proceed();
                } finally {
                    MessageThreadLocal.remove();
                    ChannelThreadLocal.remove();
                }
            } else {
                return invocation.proceed();
            }
        }
    }

    @Aspect
    @Order(Ordered.HIGHEST_PRECEDENCE + 100)
    static class JaRabbitListenerAspect {
        private static final NodeKind rabbitMQConsume = new NodeKind("rabbitmq-consume", true);
        private static final String logFormatEntry = "queue:{},context:{}";

        public JaRabbitListenerAspect() {
            DwMapContextChain instance = DwMapContextChain.getInstance();
            DwLog.info("JaRabbitListenerAspect init:{}", instance);
        }


        @Pointcut("@annotation(org.springframework.amqp.rabbit.annotation.RabbitListener)")
        public void pointcut() {
        }

        @Around("pointcut()")
        public Object aroundAdvice(ProceedingJoinPoint pjp) throws Throwable {
            Message message = MessageThreadLocal.get();
            if (null == message) {
                DwLog.warn("rabbitmq consumer not fount dapper node:{}", JaAspectUtil.getFullName(pjp));
                return pjp.proceed();
            }
            try {
                MessageProperties messageProperties = message.getMessageProperties();
                DwMapContextChain.getInstance().inherit(JaMapUtil.convertToString(messageProperties.getHeaders()));
                NodeSpan nodeSpan = NodeSpan.builder()
                        .nodeId(JaAspectUtil.getSimpleName(pjp))
                        .nodeKind(rabbitMQConsume)
                        .build();
                DwLog.info(logFormatEntry, messageProperties.getConsumerQueue(), DwJson.toString(DwCoreContext.getInstance().get()));
                return JaDapper.dapperAspect(nodeSpan, new Object[]{message.getBody()}, pjp);
            } finally {
                DwMapContextChain.getInstance().remove();
            }
        }
    }
}
