package com.jugg.agile.middleware.db.dapper;

import com.jugg.agile.framework.core.config.JaProperty;
import com.jugg.agile.framework.core.dapper.aspect.JaDapperAspectPointcut;
import com.jugg.agile.framework.core.dapper.aspect.JaNodeSpanResolver;
import com.jugg.agile.framework.core.dapper.log.JaLog;
import com.jugg.agile.framework.core.dapper.meta.NodeKind;
import com.jugg.agile.framework.core.dapper.meta.NodeSpan;
import com.jugg.agile.framework.core.util.JaStringUtil;
import com.jugg.agile.framework.core.util.datastructure.JaCollectionUtil;
import com.jugg.agile.spring.util.JaSpringContextUtil;
import org.aopalliance.intercept.MethodInvocation;
import org.mybatis.spring.annotation.MapperScan;
import org.mybatis.spring.annotation.MapperScans;
import org.springframework.aop.framework.ReflectiveMethodInvocation;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.stereotype.Component;
import org.springframework.util.ClassUtils;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * TODO 后续迁移到db
 *
 * @author chenjian
 * @since 2024年07月08日 13:38:02
 */
@Component
@ConditionalOnClass(name = {"org.mybatis.spring.annotation.MapperScan", JaNodeSpanMybatis.ProxyClassName})
public class JaNodeSpanMybatis implements JaNodeSpanResolver, JaDapperAspectPointcut {

    public static final String ProxyClassName = "org.apache.ibatis.binding.MapperProxy";

    @Override
    public NodeSpan getNodeSpan(MethodInvocation invocation) {
        if (invocation instanceof ReflectiveMethodInvocation) {
            Object proxy = ((ReflectiveMethodInvocation) invocation).getProxy();
            if (proxy.toString().startsWith(ProxyClassName)) {
                return get(NodeKind.Constant.Db);
            }
        }
        return null;
    }

    @Override
    public String getExpression() {
        String expression = JaProperty.get("ja.dapper.aspect.pointcut.mybatis");
        if (JaStringUtil.isNotEmpty(expression)) {
            return expression + " || @within(org.apache.ibatis.annotations.Mapper)";
        }

        List<String> basePackages = new ArrayList<>(addBasePackages(JaSpringContextUtil.getMainApplicationClass()));
//        String[] beanDefinitionNames = JaSpringContextUtil.getApplicationContext().getBeanDefinitionNames();
//        for (String beanDefinitionName : beanDefinitionNames) {
//            basePackages.addAll(addBasePackages(Objects.requireNonNull(JaSpringBeanUtil.getBean(beanDefinitionName)).getClass()));
//        }
        Set<String> basePackageSet = new HashSet<>();
        basePackages.forEach(s -> basePackageSet.addAll(Arrays.asList(s.split(","))));
        if (basePackageSet.size() > 1) {
            JaLog.get().warn("multiple mybatis mapper scan package");
        }

        StringBuilder expressionBuilder = new StringBuilder();
        for (String s : basePackageSet) {
            expressionBuilder.append(String.format("execution(* %s..*.*(..)) || ", s));
        }
        return expressionBuilder + "@within(org.apache.ibatis.annotations.Mapper)";
    }

    private List<String> addBasePackages(Class<?> importingClass) {
        List<String> basePackages = new ArrayList<>();
        MapperScans mapperScans = importingClass.getAnnotation(MapperScans.class);
        if (null != mapperScans) {
            basePackages.addAll(addBasePackages(mapperScans, importingClass));
        } else {
            MapperScan mapperScan = importingClass.getAnnotation(MapperScan.class);
            if (null != mapperScan) {
                basePackages.addAll(addBasePackages(mapperScan, importingClass));
            }
        }
        return basePackages;
    }

    private List<String> addBasePackages(MapperScans mapperScans, Class<?> importingClass) {
        List<String> basePackages = new ArrayList<>();
        if (JaCollectionUtil.isEmpty(mapperScans.value())) {
            basePackages.add(ClassUtils.getPackageName(importingClass.getName()));
        } else {
            for (MapperScan mapperScan : mapperScans.value()) {
                basePackages.addAll(addBasePackages(mapperScan, importingClass));
            }
        }
        return basePackages;
    }


    private List<String> addBasePackages(MapperScan annotation, Class<?> importingClass) {
        List<String> basePackages = new ArrayList<>();
        basePackages.addAll(
                Arrays.stream(annotation.value()).filter(StringUtils::hasText).collect(Collectors.toList()));
        basePackages.addAll(
                Arrays.stream(annotation.basePackages()).filter(StringUtils::hasText).collect(Collectors.toList()));
        basePackages.addAll(
                Arrays.stream(annotation.basePackageClasses()).map(ClassUtils::getPackageName).collect(Collectors.toList()));
        if (JaCollectionUtil.isEmpty(basePackages)) {
            basePackages.add(ClassUtils.getPackageName(importingClass));
        }
        return basePackages;
    }

}
