package com.digiwin.dap.nest.infrastructure.middleware.rdb.mybatis.generator;

import com.digiwin.dap.nest.kernel.core.util.DwStringUtil;
import com.digiwin.dap.nest.infrastructure.middleware.rdb.jdbc.DwJdbcConnectionProcessor;
import com.digiwin.dap.nest.infrastructure.middleware.rdb.jdbc.DwJdbcUrlProcessor;
import com.digiwin.dap.nest.infrastructure.middleware.rdb.jdbc.mysql.DwMysqlDescProcessor;
import com.digiwin.dap.nest.infrastructure.middleware.rdb.jdbc.mysql.DwMysqlShowProcessor;
import lombok.SneakyThrows;
import org.dom4j.Document;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.dom4j.Node;
import org.dom4j.io.OutputFormat;
import org.dom4j.io.SAXReader;
import org.dom4j.io.XMLWriter;

import java.io.File;
import java.io.FileWriter;
import java.sql.Connection;
import java.util.List;

/**
 *
 */
public class JaMybatisGeneratorXmlDom4j {

    private static final String targetPackage = "targetPackage";
    private static final String targetProject = "targetProject";

    /**
     * 添加 table 节点
     */
    @SneakyThrows
    public static void initGeneratorXml(JaMybatisGeneratorXmlParams xmlParams) {
        SAXReader reader = new SAXReader();
        Document doc = reader.read(new File(xmlParams.getXmlPath()));
        Element root = doc.getRootElement();

        Element classPathEntry = root.element("classPathEntry");
        classPathEntry.attribute("location").setValue(DwMybatisGenerator.getMysqlJarPath());

        Element context = root.element("context");

        Element jdbcConnection = context.element("jdbcConnection");
        String userId = xmlParams.getUserId();
        jdbcConnection.attribute("userId").setValue(userId);
        String password = xmlParams.getPassword();
        jdbcConnection.attribute("password").setValue(password);
        String connectionURL = DwJdbcUrlProcessor.getSimple(xmlParams.getIp(), xmlParams.getPort(), xmlParams.getDatabase());
        jdbcConnection.attribute("connectionURL").setValue(connectionURL);
        try (Connection connection = DwJdbcConnectionProcessor.get(connectionURL, userId, password)) {
            List<String> tableNameList = DwMysqlShowProcessor.showTables(connection, xmlParams.getDatabase());

            Element javaModelGenerator = context.element("javaModelGenerator");
            javaModelGenerator.attribute(targetPackage).setValue(xmlParams.getPojoPackage());
            javaModelGenerator.attribute(targetProject).setValue(xmlParams.getPojoTargetProject());

            Element sqlMapGenerator = context.element("sqlMapGenerator");
            sqlMapGenerator.attribute(targetPackage).setValue(xmlParams.getMapperXmlPackage());
            sqlMapGenerator.attribute(targetProject).setValue(xmlParams.getMapperXmlTargetProject());

            Element javaClientGenerator = context.element("javaClientGenerator");
            javaClientGenerator.attribute(targetPackage).setValue(xmlParams.getMapperPackage());
            javaClientGenerator.attribute(targetProject).setValue(xmlParams.getMapperTargetProject());

            List<Node> tableNodes = context.selectNodes("./table");

            for (Node tableNode : tableNodes) {
                context.remove(tableNode);
            }

            for (String tableName : tableNameList) {
                Element newTableElement = DocumentHelper.createElement("table");
                String toCamelCase = DwStringUtil.upperFirstCase(DwStringUtil.underscoreToCamelCase(tableName));

                /*
                 | 数据库            | catalog            | schema                  | 实际意义                 |
                 | -------------- | ------------------ | ----------------------- | -------------------- |
                 | **MySQL**      | ✅ 数据库名（Database）   | ❌ 通常固定为 null 或空         | MySQL 没有 schema 概念   |
                 | **PostgreSQL** | ✅ 实际库名             | ✅ schema 概念存在（public 等） | 一般 schema = “public” |
                 | **SQL Server** | ✅ 库名               | ✅ schema 名（dbo 等）       | 类似 PostgreSQL        |
                 | **Oracle**     | ❌ catalog 一般为 null | ✅ schema = 用户名          | Oracle 只有 schema 概念  |
                 */

                /*
                在 MBG 的源码中（org.mybatis.generator.internal.db.DatabaseIntrospector）：
                catalog 和 schema 都会被传入 JDBC 的 DatabaseMetaData.getTables(catalog, schema, tableName, null)
                MySQL 驱动会只识别 catalog 参数。
                如果你设置 schema 而不设置 catalog，MySQL 下会查不到表。
                如果你同时设置了两者，MySQL 一般会使用 catalog，忽略 schema。
                 */
//                newTableElement.addAttribute("schema", xmlParams.getDatabase());
                newTableElement.addAttribute("catalog", xmlParams.getDatabase());
                newTableElement.addAttribute("tableName", tableName);
                newTableElement.addAttribute("domainObjectName", toCamelCase + xmlParams.getPojoSuffix());
                newTableElement.addAttribute("mapperName", toCamelCase + xmlParams.getMapperSuffix());
                newTableElement.addAttribute("delimitIdentifiers", "true");

                // 添加 property 元素
                Element propertyElement = newTableElement.addElement("property");
                propertyElement.addAttribute("name", "ignoreQualifiersAtRuntime");
                propertyElement.addAttribute("value", "true");


                String autoIncrementColumn = DwMysqlDescProcessor.getAutoIncrementColumn(connection, tableName);
                if (null != autoIncrementColumn) {
                    Element generatedKeyElement = newTableElement.addElement("generatedKey");
                    generatedKeyElement.addAttribute("column", autoIncrementColumn);
                    generatedKeyElement.addAttribute("sqlStatement", "MySql");
                    generatedKeyElement.addAttribute("identity", "true");
                }

                context.add(newTableElement);
            }

            OutputFormat format = OutputFormat.createPrettyPrint();
            XMLWriter writer = new XMLWriter(new FileWriter(xmlParams.getXmlPath()), format);
            writer.write(doc);
            writer.close();
        }
    }

}
