package com.digiwin.dap.nest.infrastructure.middleware.rdb.datasource.aspect;

import com.digiwin.dap.nest.kernel.core.config.DwConfig;
import com.digiwin.dap.nest.kernel.core.config.DwProperty;
import com.digiwin.dap.nest.kernel.core.config.DwPropertyListener;
import com.digiwin.dap.nest.kernel.core.dapper.alarm.JaAlarm;
import com.digiwin.dap.nest.kernel.core.dapper.log.DwLog;
import com.digiwin.dap.nest.kernel.core.util.DwStringUtil;
import com.digiwin.dap.nest.kernel.core.util.datastructure.DwCollectionUtil;
import com.digiwin.dap.nest.infrastructure.middleware.rdb.datasource.DwDataSourceFactory;
import com.digiwin.dap.nest.infrastructure.middleware.rdb.datasource.DwDataSourceRoute;
import com.digiwin.dap.nest.infrastructure.middleware.rdb.datasource.meta.DwDataSourceAnnotation;
import com.digiwin.dap.nest.infrastructure.middleware.rdb.transaction.DwTransactionProcessor;
import com.zaxxer.hikari.HikariDataSource;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Pointcut;
import org.springframework.transaction.support.TransactionSynchronizationManager;

import javax.sql.DataSource;
import java.util.Map;

/**
 * 适配已经使用自建数据源的多数据源模式
 *
 * @author chenjian
 * @since 2023年06月13日 10:17:49
 */
@Aspect
public class DwConnectionAspect {

    private static long timeoutDbConn;
    private static final String keyTimeoutDbConn = DwConfig.KeyPrefix + "dapper.common.timeout.db-conn";

    static {
        DwPropertyListener.addAndRunCommonListener(() -> {
            timeoutDbConn = DwProperty.getLong(keyTimeoutDbConn, 200L);
        });
    }

    @Pointcut("execution(* javax.sql.DataSource.getConnection())")
    public void pointcut() {
    }

    @Around("pointcut()")
    public Object aroundAdvice(ProceedingJoinPoint pjp) throws Throwable {
//        long start = System.currentTimeMillis();
        try {
            return getConnection(pjp);
        } finally {
//            alarm(start, pjp);
        }

    }

    private Object getConnection(ProceedingJoinPoint pjp) throws Throwable {
        DwDataSourceAnnotation route = DwDataSourceRoute.getRoute().get();
        if (null == route) {
            return pjp.proceed();
        }
        DataSource dataSource = null;
        Map<String, DataSource> dataSourcePool = DwDataSourceFactory.get();
        if (DwCollectionUtil.isNotEmpty(dataSourcePool)) {
            if (route.isSlaver() && (TransactionSynchronizationManager.isActualTransactionActive() || DwTransactionProcessor.isBeginSpringTx())) {
                String master = route.getMaster();
                if (DwStringUtil.isNotEmpty(master)) {
                    dataSource = dataSourcePool.get(master);
                } else {
                    DwLog.get().debug("datasource route slave by : master");
                    return pjp.proceed();
                }
            } else {
                DwLog.get().debug("datasource route by : {}", route.getName());
                dataSource = dataSourcePool.get(route.getName());
            }
        }
        if (null == dataSource) {
            throw new RuntimeException("switch datasource error, datasource is null : " + route.getName());
        }
        DwLog.get().debug("datasource route by : {}", route.getName());
        return dataSource.getConnection();
    }

    private void alarm(long start, ProceedingJoinPoint pjp) {
        long end = System.currentTimeMillis() - start;
        JaAlarm.alarmAndLog(end > timeoutDbConn
                , () -> {
                    Object target = pjp.getTarget();
                    String poolName = "";
                    if (target instanceof HikariDataSource) {
                        poolName = ((HikariDataSource) target).getPoolName();
                    }

                    return String.format("javax.sql.DataSource.getConnection poolName:%s timeout: %s > %s", poolName + "", end, timeoutDbConn);
                }
                , () -> "javax.sql.DataSource.getConnection timeout");
    }
}
