package com.jugg.agile.middleware.redis;

import com.jugg.agile.framework.core.dapper.log.JaLog;
import com.jugg.agile.framework.core.lock.distributed.JaDistributedLockAdapter;
import com.jugg.agile.framework.core.lock.distributed.JaDistributedLockEntity;
import com.jugg.agile.framework.core.lock.distributed.JaDistributedLockAspect;
import com.jugg.agile.framework.core.util.JaStringUtil;
import com.jugg.agile.framework.core.util.JaValidateUtil;
import com.jugg.agile.framework.core.util.algorithm.id.JaUUID;
import com.jugg.agile.framework.meta.adapter.JaI18nAdapter;
import com.jugg.agile.framework.meta.function.JaFunctionTR;
import lombok.SneakyThrows;
import org.springframework.transaction.support.TransactionSynchronizationAdapter;
import org.springframework.transaction.support.TransactionSynchronizationManager;

import java.util.Set;

/**
 * redis 分布式锁
 *
 * @author chenjian
 * @since 2020年09月07日 10:00:27
 */
public class JaRedisLock {

    private JaRedisLock() {
    }


    /**
     * 使用demo 可参考{@link JaDistributedLockAspect}
     *
     * @author chenjian
     * @since 2020年10月13日 12:04:29
     */
    @SneakyThrows
    public static <T> T lock(JaDistributedLockEntity entity, JaFunctionTR<T> callable) {
        String lockKey = entity.getLockKey();
        String apiName = entity.getApiName();
        if (JaStringUtil.isEmpty(lockKey)) {
            JaLog.get().warn("lockKey is null,{}", apiName);
            return callable.apply();
        }
        // 判断是否可重入
        Set<String> reentrantLockKeySet = JaDistributedLockAdapter.getReentrantThreadLocal().getNotNull();
        if (reentrantLockKeySet.contains(lockKey)) {
            return callable.apply();
        }
        JaLog.biz("lockKey:{}", lockKey);
        // TODO 应对误删, 其实防不住, 待思考
        String requestId = lockKey + ":" + JaUUID.UUID19();
        boolean lock = JaDistributedLockAdapter.spin(entity, () -> JaRedis.setNx(lockKey, requestId, entity.getTimeout()));
        JaValidateUtil.isTrue(!lock, () -> JaI18nAdapter.getMessage(entity.getMessage()));
        try {
            reentrantLockKeySet.add(lockKey);
            return callable.apply();
        } finally {
            unlock(lockKey, requestId, entity.isTx());
        }
    }

    /**
     * 释放锁
     *
     * @author chenjian
     * @since 2020年08月11日 11:43:45
     */
    private static void unlock(String lockKey, String requestId, boolean tx) {
        if (TransactionSynchronizationManager.isSynchronizationActive() && tx) {
            try {
                TransactionSynchronizationManager.registerSynchronization(new TransactionSynchronizationAdapter() {
                    @Override
                    public void afterCompletion(int status) {
                        unlock(lockKey, requestId);
                    }
                });
            } catch (Throwable e) {
                JaLog.get().error("Transaction afterCompletion unLock lock failed", e);
                unlock(lockKey, requestId);
            }
        } else {
            unlock(lockKey, requestId);
        }
    }

    /**
     * TODO 暂用del
     * 是否要使用unlink
     * org.springframework.data.redis.core.RedisTemplate#unlink(java.lang.Object)
     */
    private static void unlock(String lockKey, String requestId) {
        try {
            String currentValue = JaRedis.getString(lockKey);
            if (!requestId.equals(currentValue)) {
                JaLog.get().warn("redis unlock warn {}:[currentValue:{}, requestId:{}", lockKey, currentValue, requestId);
            }
            if (JaRedis.del(lockKey) < 1) {
                JaLog.get().error("redis unlock fail:{}", lockKey);
            }
        } catch (Throwable e) {
            JaLog.get().error("redis unlock error:{}", lockKey);
        } finally {
            JaDistributedLockAdapter.getReentrantThreadLocal().get().remove(lockKey);
        }
    }
}
