package com.digiwin.dap.nest.infrastructure.middleware.redis.adapter.spring.command;

import com.digiwin.dap.nest.kernel.core.util.datastructure.DwCollectionUtil;
import com.digiwin.dap.nest.infrastructure.middleware.redis.DwRedisCommand;
import com.digiwin.dap.nest.infrastructure.middleware.redis.adapter.spring.JaRedisTemplateProcessor;
import com.digiwin.dap.nest.infrastructure.middleware.redis.meta.JaRedisConstants;
import lombok.SneakyThrows;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.script.DefaultRedisScript;
import org.springframework.data.redis.core.script.RedisScript;

import java.time.Duration;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

@SuppressWarnings({"unchecked", "rawtypes"})
public class JaRedisTemplateCommand implements DwRedisCommand {

    private final RedisTemplate redisTemplate;

    public JaRedisTemplateCommand() {
        redisTemplate = JaRedisTemplateProcessor.getRedisTemplate();
    }

    public JaRedisTemplateCommand(RedisTemplate redisTemplate) {
        this.redisTemplate = redisTemplate;
    }

    @Override
    public Boolean set(String key, String value, Long... seconds) {
        if (DwCollectionUtil.isEmpty(seconds)) {
            redisTemplate.opsForValue().set(key, value);
        } else {
            redisTemplate.opsForValue().set(key, value, seconds[0], TimeUnit.SECONDS);
        }
        return true;
    }

    private static final RedisScript<Long> scriptSetNx = new DefaultRedisScript<>(JaRedisConstants.LuaSetNxScript, Long.class);

    @Override
    public Boolean setNx(String key, String value, Long... seconds) {
        if (DwCollectionUtil.isEmpty(seconds)) {
            return redisTemplate.opsForValue().setIfAbsent(key, value);
        } else {
            return JaRedisConstants.LuaSuccess == redisTemplate.execute(scriptSetNx, Collections.singletonList(key), value, seconds[0].toString());
        }
    }

    @Override
    public Long del(String key) {
        return Boolean.TRUE.equals(redisTemplate.delete(key)) ? 1L : 0L;
    }

    @Override
    public Boolean exists(String key) {
        return Boolean.TRUE.equals(redisTemplate.hasKey(key));
    }

    @Override
    @SneakyThrows
    public String get(String key) {
        Object o = redisTemplate.opsForValue().get(key);
        return null == o ? null : o.toString();
    }

    @Override
    public Long expire(String key, Long seconds) {
        return Boolean.TRUE.equals(redisTemplate.expire(key, Duration.ofSeconds(seconds))) ? 1L : 0;
    }

    @Override
    public Long incr(String key, Long... increment) {
        if (DwCollectionUtil.isEmpty(increment)) {
            return redisTemplate.opsForValue().increment(key);
        }
        return redisTemplate.opsForValue().increment(key, increment[0]);
    }

    @Override
    public Long decr(String key, Long... decrement) {
        if (DwCollectionUtil.isEmpty(decrement)) {
            return redisTemplate.opsForValue().decrement(key);
        }
        return redisTemplate.opsForValue().decrement(key, decrement[0]);
    }

    @Override
    public Object eval(String script, List<String> keys, Object... args) {
        return redisTemplate.execute(new DefaultRedisScript<>(script, Object.class), keys, args);
    }

    @Override
    public Long lpush(String key, String... args) {
        if (args == null || args.length == 0) {
            return 0L;
        }
        return redisTemplate.opsForList().leftPushAll(key, args);
    }

    @Override
    public Long rpush(String key, String... args) {
        if (args == null || args.length == 0) {
            return 0L;
        }
        return redisTemplate.opsForList().rightPushAll(key, args);
    }

    @Override
    public List<String> lrange(String key, long start, long end) {
        List<Object> objects = redisTemplate.opsForList().range(key, start, end);
        if (objects == null) {
            return Collections.emptyList();
        }
        return objects.stream()
                .map(obj -> obj != null ? obj.toString() : null)
                .collect(Collectors.toList());
    }

    @Override
    public String lindex(String key, long index) {
        Object value = redisTemplate.opsForList().index(key, index);
        return value != null ? value.toString() : null;
    }

    @Override
    public Long lrem(String key, long count, String value) {
        return redisTemplate.opsForList().remove(key, count, value);
    }
}
