package com.digiwin.dap.nest.infrastructure.spring.boot.webmvc.util;

import com.alibaba.fastjson.JSON;
import com.digiwin.dap.nest.kernel.core.config.DwProperty;
import com.digiwin.dap.nest.kernel.core.dapper.log.DwMDC;
import com.digiwin.dap.nest.kernel.core.util.DwStringUtil;
import com.digiwin.dap.nest.kernel.core.util.io.net.http.meta.JaContentTypeEnum;
import com.digiwin.dap.nest.kernel.core.util.io.net.http.meta.JaHttpHeaders;
import com.digiwin.dap.nest.kernel.core.util.io.net.http.meta.JaHttpNginxHeaders;
import com.digiwin.dap.nest.kernel.core.util.io.serialize.json.DwJson;
import jakarta.servlet.ServletOutputStream;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import lombok.SneakyThrows;

import java.nio.charset.StandardCharsets;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;

public class DwServletUtil {

    private DwServletUtil() {
    }

    public static final String MozillaKey = "Mozilla";

    /**
     * @return /contextPath/
     */
    public static String getContextPath() {
        String contextPath = DwProperty.get("server.servlet.context-path");
        if (null == contextPath) {
            return DwStringUtil.SeparatorBackslash;
        }
        if (!contextPath.startsWith(DwStringUtil.SeparatorBackslash)) {
            contextPath = DwStringUtil.SeparatorBackslash + contextPath;
        }
        if (!contextPath.endsWith(DwStringUtil.SeparatorBackslash)) {
            contextPath = contextPath + DwStringUtil.SeparatorBackslash;
        }
        return contextPath;
    }

    public static String getPort() {
        return DwProperty.get("server.port");
    }

    /**
     * 暂不考虑遍历全部
     */
    private static final String[] headersToCheck = {
            "X-Forwarded-For",        // Standard header for proxy servers
            "Proxy-Client-IP",        // Older headers for proxy servers
            "WL-Proxy-Client-IP",     // WebLogic server proxy
            "HTTP_X_FORWARDED_FOR",   // Commonly used by Apache
            "HTTP_X_FORWARDED",       // Commonly used by Apache
            "HTTP_X_CLUSTER_CLIENT_IP", // Commonly used by some proxies
            "HTTP_CLIENT_IP",         // Commonly used by some proxies
            "HTTP_FORWARDED_FOR",     // Commonly used by some proxies
            "HTTP_FORWARDED"          // Commonly used by some proxies
    };


    public static String getUserAgent(HttpServletRequest request) {
        if (null == request) {
            return null;
        }
        return request.getHeader(JaHttpHeaders.USER_AGENT);
    }

    /**
     * servlet 打印 response
     *
     * @author chenjian
     * @since 2020年09月29日 10:21:13
     */
    @SneakyThrows
    public static void printResponse(HttpServletRequest request, HttpServletResponse response, Object object) {
        ServletOutputStream out = response.getOutputStream();
        response.setContentType(JaContentTypeEnum.Json.getContentType(StandardCharsets.UTF_8));
        try {
            out.write(JSON.toJSONString(object).getBytes());
        } catch (Throwable e) {
            out.write(DwJson.toString(String.format("response print error:[%s:%s:%s]"
                    , request.getRequestURL()
                    , object.getClass().getSimpleName()
                    , DwMDC.get())
                    ).getBytes()
            );
        }
        out.flush();
    }

    public static String getRemoteAddr(HttpServletRequest request) {
        String ip = request.getHeader(headersToCheck[0]);
        if (DwStringUtil.isNotEmpty(ip) && !DwStringUtil.UNKNOWN.equalsIgnoreCase(ip)) {
            // 多次反向代理后会有多个ip值，第一个ip才是真实ip
            int index = ip.indexOf(",");
            return index != -1 ? ip.substring(0, index) : ip;
        }

        ip = request.getHeader(JaHttpNginxHeaders.X_Real_IP);
        if (DwStringUtil.isNotEmpty(ip) && !DwStringUtil.UNKNOWN.equalsIgnoreCase(ip)) {
            return ip;
        }
        return request.getRemoteAddr();
    }

    public static Map<String, String> getAllHeaders(HttpServletRequest request) {
        Map<String, String> headers = new HashMap<>();
        Enumeration<String> headerNames = request.getHeaderNames();
        while (headerNames.hasMoreElements()) {
            String headerName = headerNames.nextElement();
            String headerValue = request.getHeader(headerName);
            headers.put(headerName, headerValue);
        }
        return headers;
    }

    /**
     * 判断请求是否来自web端
     */
    public static boolean isFromWeb(HttpServletRequest request) {
        String header = request.getHeader(JaHttpNginxHeaders.X_Forwarded_For);
        if (DwStringUtil.isNotEmpty(header)) {
            return true;
        }
        header = request.getHeader(JaHttpNginxHeaders.X_Real_IP);
        if (DwStringUtil.isNotEmpty(header)) {
            return true;
        }
        header = request.getHeader(JaHttpHeaders.USER_AGENT);
        return header.contains(MozillaKey);
    }
}
