package com.digiwin.dap.nest.infrastructure.spring.boot.util;

import com.digiwin.dap.nest.kernel.core.dapper.log.DwLog;
import com.digiwin.dap.nest.kernel.core.util.DwStringUtil;
import com.digiwin.dap.nest.kernel.core.util.concurrent.DwThreadLocal;
import com.digiwin.dap.nest.kernel.meta.adapter.JaI18nAdapter;
import com.digiwin.dap.nest.infrastructure.spring.util.DwSpringBeanUtil;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.context.MessageSource;
import org.springframework.context.MessageSourceAware;
import org.springframework.context.NoSuchMessageException;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.context.support.MessageSourceAccessor;

import java.util.Locale;

/**
 * 国际化工具类
 *
 * @author chenjian﻿
 * @since 2023年03月09日 15:57:28
 */
public class DwSpringI18nUtil {

    private DwSpringI18nUtil() {
    }

    public static final String Header = "accept-language";
    public static final String HeaderUp = "Accept-Language";

    private static MessageSourceAccessor accessor;
    public static final Locale DefaultLocale = Locale.SIMPLIFIED_CHINESE;

    public static void setAccessor(MessageSourceAccessor accessor) {
        DwSpringI18nUtil.accessor = accessor;
    }

    public static MessageSourceAccessor getAccessor() {
        return accessor;
    }

    public static final DwThreadLocal<Locale> LocaleThreadLocal = new DwThreadLocal<>(o -> LocaleContextHolder.setLocale((Locale) o));

    public static String getMessage(String code, Object... args) {
        return getMessage(code, LocaleThreadLocal.get(), args);
    }

    public static String getMessage(String code, Locale locale, Object... args) {
        if (null == accessor) {
            initAccessor();
            if (null == accessor) {
                return code;
            }
        }
        if (null == locale) {
            locale = DefaultLocale;
        }
        try {
            return accessor.getMessage(code, args, locale);
        } catch (NoSuchMessageException e) {
            return "NoSuch:" + code;
        } catch (Throwable e) {
            return "ParseError:" + code;
        }
    }

    private static synchronized void initAccessor() {
        accessor = DwSpringBeanUtil.getBean(MessageSourceAccessor.class);
    }

    /**
     * 获取字符串AcceptLanguage构建Locale, 容错tomcat复用请求时不做解析的bug
     */
    public static Locale parseLocales(String acceptLanguage) {
        try {
            if (DwStringUtil.isEmpty(acceptLanguage)) {
                return DefaultLocale;
            }

            String[] language = acceptLanguage.split(";")[0].split(",")[0].split("-");
            if (language.length == 2) {
                return new Locale(language[0], language[1]);
            } else if (language.length == 1) {
                return new Locale(language[0]);
            }
            return DefaultLocale;
        } catch (Throwable e) {
            DwLog.error("parseLocales error:{}", acceptLanguage, e);
            return DefaultLocale;
        }
    }

    public static String getLanguage() {
        Locale locale = LocaleThreadLocal.get();
        return null == locale ? DefaultLocale.toLanguageTag() : locale.toLanguageTag();
    }

    /**
     * 构建MessageSource, 以便后续出现后端国际化需求
     *
     * @author chenjian
     * @see org.springframework.boot.autoconfigure.context.MessageSourceAutoConfiguration
     * @see org.springframework.context.support.ResourceBundleMessageSource#ResourceBundleMessageSource()
     * @since 2023年03月09日 19:38:19
     */
    @Configuration
    public static class JaI18nConfiguration {

        public JaI18nConfiguration() {
            JaI18nAdapter.setI18nHandler(DwSpringI18nUtil::getMessage);
            JaI18nAdapter.setLocaleHandler(DwSpringI18nUtil::getLanguage);
        }

        @Configuration
        @ConditionalOnBean({MessageSource.class})
        public static class MessageSourceAwareConfiguration implements MessageSourceAware {
            @Override
            @SuppressWarnings("all")
            public void setMessageSource(MessageSource messageSource) {
                DwSpringI18nUtil.accessor = new MessageSourceAccessor(messageSource);
            }
        }
    }
}
