package com.digiwin.dap.nest.infrastructure.spring.cloud.openfeign;

import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.digiwin.dap.nest.kernel.core.config.DwConfig;
import com.digiwin.dap.nest.kernel.core.dapper.log.DwLog;
import com.digiwin.dap.nest.kernel.core.util.io.serialize.json.adapter.jackson.JaJacksonProcessor;
import com.digiwin.dap.nest.kernel.meta.dto.DwResultDTO;
import com.digiwin.dap.nest.kernel.meta.exception.DwException;
import com.digiwin.dap.nest.kernel.meta.handler.JaResultParseHandler;
import feign.FeignException;
import feign.Response;
import feign.codec.Decoder;
import feign.codec.Encoder;
import org.springframework.beans.factory.ObjectFactory;
import org.springframework.boot.autoconfigure.http.HttpMessageConverters;
import org.springframework.cloud.openfeign.support.ResponseEntityDecoder;
import org.springframework.cloud.openfeign.support.SpringDecoder;
import org.springframework.cloud.openfeign.support.SpringEncoder;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.converter.json.MappingJackson2HttpMessageConverter;
import org.springframework.util.StreamUtils;

import java.io.IOException;
import java.lang.reflect.Type;
import java.nio.charset.StandardCharsets;

@Configuration
public class JaFeignConfiguration {
    private static final ObjectMapper objectMapper = JaJacksonProcessor.createObjectMapper();

    @Bean
    public Decoder feignDecoder() {
        if (Boolean.TRUE.equals(DwConfig.getBoolean("feign.unwrapDecoder", false))) {
            return new JaResultUnwrapDecoder();
        } else {
            return new ResponseEntityDecoder(new SpringDecoder(createHttpMessageConvertersObjectFactory()));
        }
    }

    @Bean
    public Encoder feignEncoder() {
        return new SpringEncoder(createHttpMessageConvertersObjectFactory());
    }

    private ObjectFactory<HttpMessageConverters> createHttpMessageConvertersObjectFactory() {
        HttpMessageConverter<Object> jacksonConverter = new MappingJackson2HttpMessageConverter(objectMapper);
        HttpMessageConverters httpMessageConverters = new HttpMessageConverters(jacksonConverter);
        return () -> httpMessageConverters;
    }

    /**
     * 自定义解码器,自动拆包实现了 JaResultParseHandler 接口的包装类
     * <p>
     * 使用方式:
     * 1. 默认使用 JaResultDTO,无需额外配置
     * 2. 自定义包装类,创建配置类并在 @FeignClient 中指定 configuration
     */
    public static class JaResultUnwrapDecoder implements Decoder {

        @Override
        public Object decode(Response response, Type type) throws IOException, FeignException {
            if (response.body() == null) {
                return null;
            }
            JavaType wrappedType = objectMapper.getTypeFactory().constructParametricType(DwResultDTO.class, objectMapper.constructType(type));

            // 直接从 InputStream 反序列化,避免中间转 String 的开销
            JaResultParseHandler<?> result;
            try {
                result = objectMapper.readValue(response.body().asInputStream(), wrappedType);
            } catch (Throwable e) {
                // 如果反序列化失败,读取原始内容用于错误信息
                String body = StreamUtils.copyToString(response.body().asInputStream(), StandardCharsets.UTF_8);
                DwLog.error("[{}] feign decode error[{}:{}]", response.status(), body.getBytes(StandardCharsets.UTF_8));
                throw new DwException(response.status() + "").buildCodeMsg(response.reason());
            }

            if (result == null) {
                return null;
            }
            if (Boolean.FALSE.equals(result.success())) {
                throw new DwException(result.code()).buildCodeMsg(result.message());
            }
            return result.data();
        }
    }
}
