package com.digiwin.cross.app.command;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.digiwin.athena.esp.sdk.util.StringUtil;
import com.digiwin.cross.app.dto.EaiMappingDTO;
import com.digiwin.cross.app.dto.request.QueryErrorConnectionRequest;
import com.digiwin.cross.app.dto.response.QueryEaiListResponse;
import com.digiwin.cross.domain.common.HeaderNamesConstant;
import com.digiwin.cross.domain.enums.EaiTypeEnum;
import com.digiwin.cross.domain.parameter.ApplicationSystemParameter;
import com.digiwin.cross.domain.utils.JsonUtil;
import com.digiwin.cross.infrastructure.database.entity.*;
import com.digiwin.cross.infrastructure.database.mapper.EaiTenantMappingMapper;
import com.digiwin.cross.infrastructure.database.mapper.UnionEaiMapper;
import com.digiwin.http.client.DWHttpClient;
import com.digiwin.http.client.DWRequestOption;
import com.digiwin.http.client.exception.DWHttpFailedException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.apachecommons.CommonsLog;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.text.MessageFormat;
import java.util.*;

/**
 * @author 宋卫奇
 * @ClassName QueryEaiListCmd
 * @description: 查询地中台清单
 * add by songwq 20250715
 * 针对集成中心开放接口 story-view-48716
 * @date 2025年07月11日
 * @version: 1.0
 */
@CommonsLog
@Component
public class QueryEaiListCmd {

    // 远端中台Mapper
    private final UnionEaiMapper unionEaiMapper;

    // 租户与中台映射Mapper
    private final EaiTenantMappingMapper eaiTenantMappingMapper;

    /**
     * dwHttpClient 请求http客户端
     */
    private final DWHttpClient dwHttpClient;

    @Autowired
    public QueryEaiListCmd(UnionEaiMapper unionEaiMapper, EaiTenantMappingMapper eaiTenantMappingMapper, DWHttpClient dwHttpClient) {
        this.unionEaiMapper = unionEaiMapper;
        this.eaiTenantMappingMapper = eaiTenantMappingMapper;
        this.dwHttpClient = dwHttpClient;
    }

    /**
     * 执行
     * @param token    认证
     * @param tenantId 租户ID
     * @param versionList  版本：s、m、c
     * @param page 分页对象
     * @return
     */
    public QueryEaiListResponse execute(String token, String tenantId, List<String> versionList, Page page) {
        List<EaiTenantMappingPO> eaiTenantMappingPOList = null;
        // 获取匹配的租户对应UID列表
        LambdaQueryWrapper<EaiTenantMappingPO> queryWrapper = Wrappers.lambdaQuery(EaiTenantMappingPO.class);
        if (StringUtil.isNotEmpty(tenantId)) {
            queryWrapper.eq(EaiTenantMappingPO::getTenantId, tenantId);

            // 先根据租户查询UID
            eaiTenantMappingPOList = eaiTenantMappingMapper.selectList(queryWrapper);

            // 根据UID查询一台中台上所有的租户（中台上可以绑定多个租户）
            queryWrapper.or().eq(EaiTenantMappingPO::getEaiUid, eaiTenantMappingPOList.get(0).getEaiUid());
        }
        eaiTenantMappingPOList = eaiTenantMappingMapper.selectList(queryWrapper);

        // 拼接满足条件的UID列表作为查询条件（eai需要去重，多个租户可能对应一个中台）
        Set<String> eaiIdSet = new HashSet<>();
        List<String> tenantIdList = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(eaiTenantMappingPOList)) {
            for (EaiTenantMappingPO eaiTenantMappingPO : eaiTenantMappingPOList) {
                eaiIdSet.add(eaiTenantMappingPO.getEaiUid());
                tenantIdList.add(eaiTenantMappingPO.getTenantId());
            }
        }

        // 查询租户是否连接异常
        List<String> errorConnectTenantList = queryErrorConnect(token, tenantIdList);

        // 调用iam获取租户对应的客户名称(要考虑分批查询，一次100个租户，最多循环查20次，也就是这个list元素不要超过2000，如果超过2000，那么只反馈前2000个租户信息)
        Map<String, String> tenantIdMappingNode = callIamGetTenantName(token, tenantIdList);

        // 查询中台清单
        LambdaQueryWrapper<UnionEAIPO> wrapper = Wrappers.lambdaQuery(UnionEAIPO.class);
        if (CollectionUtils.isNotEmpty(eaiIdSet)) {
            wrapper.in(UnionEAIPO::getUid, eaiIdSet);
        }
        if (CollectionUtils.isNotEmpty(versionList)) {
            wrapper.and(wq -> {
                for(String cond : versionList){
                    wq.or(sub -> sub.like(UnionEAIPO::getRemark, cond));
                }
            });
        }

        Page<UnionEAIPO> unionEAIPOPage = unionEaiMapper.selectPage(page, wrapper);

        // 拼接响应
        QueryEaiListResponse response = new QueryEaiListResponse();
        response.setEaiDTOs(appendEaiMappingDTO(unionEAIPOPage.getRecords(), eaiTenantMappingPOList, tenantIdMappingNode, errorConnectTenantList));
        response.setCode(200);
        response.setMessage("success");
        response.setTotal(unionEAIPOPage.getTotal());
        return response;
    }

    /**
     * 拼接中台响应对象
     * @param unionEAIPOList 地中台清单列表
     * @param eaiTenantMappingPOList 租户与中台映射列表
     * @param tenantIdMappingNode 租户信息
     * @param errorConnectTenantList 连接异常租户信息
     * @return
     */
    private List<EaiMappingDTO> appendEaiMappingDTO(List<UnionEAIPO> unionEAIPOList,
                                                    List<EaiTenantMappingPO> eaiTenantMappingPOList,
                                                    Map<String, String> tenantIdMappingNode,
                                                    List<String> errorConnectTenantList) {
        // 创建拼接响应对象
        List<EaiMappingDTO> eaiDTOs = new ArrayList<>();

        // 迭代拼接中台信息
        for (UnionEAIPO unionEAIPO : unionEAIPOList) {
            EaiMappingDTO eaiMappingDTO = new EaiMappingDTO();
            eaiMappingDTO.setUid(unionEAIPO.getUid());

            // 租户连接状态标识
            String tenantStatus = "1";

            // 拼接租户信息
            for (EaiTenantMappingPO eaiTenantMappingPO : eaiTenantMappingPOList) {
                if (unionEAIPO.getUid().equals(eaiTenantMappingPO.getEaiUid())) {
                    String tid = eaiTenantMappingPO.getTenantId();
                    eaiMappingDTO.addTenant(tid, tenantIdMappingNode.get(tid));

                    // 存在租户链接异常说明当前中台已经掉线
                    if (errorConnectTenantList.contains(tid)) {
                        tenantStatus = "0";
                    }
                }
            }

            // 根据remark进行配置类型
            eaiMappingDTO.setRemark(unionEAIPO.getRemark());
            String remark = unionEAIPO.getRemark();
            if (remark.startsWith(EaiTypeEnum.EAI_TYPE_SHUHUANTONG.getType())) {
                eaiMappingDTO.setType(EaiTypeEnum.EAI_TYPE_SHUHUANTONG.getCode());
            } else if (remark.startsWith(EaiTypeEnum.EAI_TYPE_MILIAN.getType())) {
                eaiMappingDTO.setType(EaiTypeEnum.EAI_TYPE_MILIAN.getCode());
            } else {
                // 不是m或者s，那就是旧中台
                eaiMappingDTO.setType(EaiTypeEnum.EAI_TYPE_CROSS.getCode());

                // 一般中台反馈的remark是固定拼接格式
                // 如：_2.5.0 Build 0157 所以直接通过_来截取
                int i = remark.lastIndexOf("_");
                if (i > 0) {
                    eaiMappingDTO.setRemark(remark.substring(i+1));
                } else {
                    // 如果没匹配到，直接写cross
                    eaiMappingDTO.setRemark(EaiTypeEnum.EAI_TYPE_CROSS.getType());
                }
            }
            eaiMappingDTO.setIp(unionEAIPO.getIpAddress());
            eaiMappingDTO.setGateway_id(unionEAIPO.getGatewayId());
            eaiMappingDTO.setGateway_name(unionEAIPO.getGatewayName());
            eaiMappingDTO.setBuildTime(unionEAIPO.getBuildTime());
            eaiMappingDTO.setLastUpdateTime(unionEAIPO.getLastUpdateTime());
            eaiMappingDTO.setId(unionEAIPO.getId());
            eaiMappingDTO.setStatus(tenantStatus);
            eaiDTOs.add(eaiMappingDTO);
        }
        return eaiDTOs;
    }


    /**
     * json处理mapper
     */
    ObjectMapper mapper = new ObjectMapper();

    // 一百个一批，避免查询过多，最多支持20批，
    private int chunkSize = 100;
    private int maxSize = 20;

    /**
     * 调用iam 获取租户名称
     * @param token
     * @param tIdList 请求对象
     */
    private Map<String, String> callIamGetTenantName(String token, List<String> tIdList) {
        Map<String, String> resMap = new HashMap<>();
        String reqJsonStr = null;

        // 为了避免一次请求查询过多的租户，需要进行拆分，一百个一批，避免查询过多，最多支持20批，也就是一次最多查询2000个租户，多余的租户不进行查询
        List<List<String>> tenantIdLists = splitList(tIdList, chunkSize, maxSize);
        for (List<String> tenantIdList : tenantIdLists) {
            try {
                HttpPost tHttpPost = new HttpPost(ApplicationSystemParameter.SDK_IAM + "/api/iam/v2/tenant/simples");
                reqJsonStr = JsonUtil.toJson(tenantIdList);
                StringEntity tEntity = new StringEntity(reqJsonStr, "utf-8");
                tEntity.setContentEncoding("UTF-8");
                tEntity.setContentType("application/json");
                tHttpPost.setEntity(tEntity);
                tHttpPost.setHeader(HeaderNamesConstant.APP_AUTH, ApplicationSystemParameter.APPTOKEN);
                tHttpPost.setHeader(HeaderNamesConstant.AUTH_USER, token);

                String tResponse = dwHttpClient.execute(tHttpPost, String.class, new DWRequestOption(false));
                JsonNode resJson = mapper.readTree(tResponse);
                Iterator<JsonNode> resJsonIt = resJson.iterator();
                while (resJsonIt.hasNext()) {
                    JsonNode iamJson = resJsonIt.next();
                    resMap.put(iamJson.get("id").asText(), iamJson.get("name").asText());
                }
            } catch (DWHttpFailedException dwHttpFailedException) {
                log.error(MessageFormat.format("调用iam失敗; request message body: {0};HTTP狀態碼: {1}; MDC回應的訊息: {2}",
                        reqJsonStr, dwHttpFailedException.getStatusCode(), dwHttpFailedException.getEntity(String.class)));
            } catch (Exception e) {
                log.error("call iam err, tenantIdList =" + reqJsonStr, e);
            }
        }
        return resMap;
    }

    /**
     * 数组拆分
     * @param originalList 需要拆分的数组
     * @param chunkSize 拆分数
     * @param maxSize 最大拆分数
     * @return
     */
    private List<List<String>> splitList(List<String> originalList, int chunkSize, int maxSize) {
        List<List<String>> chunks = new ArrayList<>();
        for (int i = 0; i < originalList.size(); i += chunkSize) {
            // 如果过大就不进行拼接了 如果一次100个元素。最大拆10批，那么拆到地1001个元素，就直接返回
            int maxNum = chunkSize * maxSize;
            if (i > maxNum) {
                log.info(MessageFormat.format("The list of tenants to be queried is too large. Currently, only {0} tenants can be queried", String.valueOf(maxNum)));
                break;
            }
            int end = Math.min(originalList.size(), i + chunkSize);
            chunks.add(new ArrayList<>(originalList.subList(i, end)));
        }
        return chunks;
    }

    /**
     * 查询租户异常连接
     * @param token
     * @param tenantIds
     */
    private List<String> queryErrorConnect(String token, List<String> tenantIds) {
        List<String> errTenantIdList = new ArrayList<>();
        String reqJsonStr = null;

        // 为了避免一次请求查询过多的租户，需要进行拆分，一百个一批，避免查询过多，最多支持20批，也就是一次最多查询2000个租户，多余的租户不进行查询
        List<List<String>> tenantIdLists = splitList(tenantIds,  chunkSize, maxSize);
        for (List<String> tenantIdList : tenantIdLists) {
            try {
                HttpPost tHttpPost = new HttpPost(ApplicationSystemParameter.MON_URL + "/restful/standard/esp_mon/tool/config/connect/queryError");
                reqJsonStr = JsonUtil.toJson(new QueryErrorConnectionRequest(tenantIdList));
                StringEntity tEntity = new StringEntity(reqJsonStr, "utf-8");
                tEntity.setContentEncoding("UTF-8");
                tEntity.setContentType("application/json");
                tHttpPost.setEntity(tEntity);
                tHttpPost.setHeader(HeaderNamesConstant.APP_AUTH, ApplicationSystemParameter.APPTOKEN);
                tHttpPost.setHeader(HeaderNamesConstant.AUTH_USER, token);

                String tResponse = dwHttpClient.execute(tHttpPost, String.class, new DWRequestOption(false));
                JsonNode resJson = mapper.readTree(tResponse).get("response");
                Iterator<JsonNode> resJsonIt = resJson.iterator();
                while (resJsonIt.hasNext()) {
                    errTenantIdList.add(resJsonIt.next().get("eai_tenantid").asText());
                }
            } catch (Exception e) {
                log.error("call espmoin err, tenantIdList =" + reqJsonStr, e);
            }
        }
        return errTenantIdList;
    }

}
