package com.digiwin.cross.app.command;

import com.digiwin.cross.app.convertor.EaiBOConvertor;
import com.digiwin.cross.app.convertor.ProductBOConvertor;
import com.digiwin.cross.app.dto.EaiDTO;
import com.digiwin.cross.app.dto.EaiProdDTO;
import com.digiwin.cross.app.dto.request.EaiRegisterRequest;
import com.digiwin.cross.app.dto.response.EaiRegisterResponse;
import com.digiwin.cross.domain.bo.EaiBO;
import com.digiwin.cross.domain.bo.entity.EaiEntity;
import com.digiwin.cross.domain.bo.entity.EaiTenantEntity;
import com.digiwin.cross.domain.bo.entity.ProductEntity;
import com.digiwin.cross.domain.common.LockNamespaceEnum;
import com.digiwin.cross.domain.exception.BaseException;
import com.digiwin.cross.domain.exception.UnionEAINotFoundException;
import com.digiwin.cross.domain.gateway.IEaiGateway;
import com.digiwin.cross.domain.gateway.ILockGateway;
import com.digiwin.cross.domain.gateway.IProductQueryGateWay;
import com.digiwin.cross.domain.gateway.ISyncMdcGateway;
import com.digiwin.cross.domain.parameter.ApplicationSystemParameter;
import com.digiwin.cross.domain.parameter.SaasParameter;
import com.digiwin.cross.domain.service.IProductService;
import com.digiwin.cross.domain.state.StateEnum;
import com.digiwin.cross.domain.utils.JsonUtil;
import lombok.extern.apachecommons.CommonsLog;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @description:
 * @author: liunansheng
 * @date: 2023/5/26 15:33
 */
@CommonsLog
@Component
public class EaiRegisterCmd {

    private final static String ACTION_REG = "reg";
    private final static String ACTION_UPDATE = "update";
    private final static String ACTION_UNREG = "unreg";

    private final IProductService productService;

    private final IEaiGateway eaiGateway;

    private final IProductQueryGateWay productQueryGateWay;

    private final ISyncMdcGateway syncMdcGateway;

    private EaiDTO espEai;

    private final ILockGateway lockGateway;

    public EaiRegisterCmd(IProductService productService, IEaiGateway eaiGateway, IProductQueryGateWay productQueryGateWay, ISyncMdcGateway syncMdcGateway, ILockGateway lockGateway) {
        this.productService = productService;
        this.eaiGateway = eaiGateway;
        this.productQueryGateWay = productQueryGateWay;
        this.syncMdcGateway = syncMdcGateway;
        this.lockGateway = lockGateway;
        espEai = new EaiDTO();
        espEai.setIp(ApplicationSystemParameter.SYS_EXT_IP);
        espEai.setUid(ApplicationSystemParameter.UNION_SYS_UID);
        espEai.setTenant_id(SaasParameter.SAAS_TENANT_ID);
        espEai.setGateway_id(SaasParameter.SAAS_GATEWAY_ID);
        espEai.setGateway_name(SaasParameter.SAAS_GATEWAY_NAME);
        espEai.setRemark(null == ApplicationSystemParameter.UNION_SYS_REMARK ? "" : ApplicationSystemParameter.UNION_SYS_REMARK);
    }

    public EaiRegisterResponse registerEai(EaiRegisterRequest request) throws BaseException {
        lockGateway.tryLock(LockNamespaceEnum.REG_EAI, request.getEai().getUid(), 100);
        try {
            EaiEntity eaiEntity = eaiGateway.queryEai(request.getEai().getUid());
            if (null != eaiEntity) {
                return updateEai(request, eaiEntity);
            }
            List<EaiProdDTO> repeatProducts = findRepeatProducts(request.getProd());
            if (CollectionUtils.isNotEmpty(repeatProducts)) {
                return new EaiRegisterResponse(repeatProducts, StateEnum.UNION_PROD_EXIST);
            }
            EaiBO eaiBONew = EaiBOConvertor.convertToBO(request);
            List<String> tenants = eaiBONew.getTenants().stream().map(EaiTenantEntity::getTenantId).collect(Collectors.toList());
            List<EaiTenantEntity> existsEaiTenants = eaiGateway.queryOtherEaiTenantByTenantIds(tenants, request.getEai().getUid());
            if (CollectionUtils.isNotEmpty(existsEaiTenants)) {
                log.error("tenant is bound:" + JsonUtil.toJson(existsEaiTenants));
                return new EaiRegisterResponse(StateEnum.UNION_TENANT_EXISTS, existsEaiTenants);
            }
            eaiGateway.addEai(eaiBONew);
            //同步mdc
            syncMdcGateway.syncEai(eaiBONew.getEai(), false);
            return new EaiRegisterResponse(ACTION_REG, espEai, findAllProducts(), StateEnum.REGEAI_REGED);
        } finally {
            lockGateway.unlock(LockNamespaceEnum.REG_EAI, request.getEai().getUid());
        }

    }

    public EaiRegisterResponse unRegisterEai(EaiRegisterRequest request) throws BaseException {
        EaiEntity eaiEntity = eaiGateway.queryEai(request.getEai().getUid());
        if (null == eaiEntity) {
            log.info("Can't find union EAI.");
        } else {
            eaiGateway.deleteEai(eaiEntity);
            //同步mdc
            syncMdcGateway.syncEai(eaiEntity, true);
        }
        return new EaiRegisterResponse(ACTION_UNREG, espEai, null, StateEnum.REGEAI_DELED);
    }


    public EaiRegisterResponse updateEai(EaiRegisterRequest request, EaiEntity... eaiEntities) throws BaseException {
        boolean isFromReg = eaiEntities.length > 0;
        EaiEntity eai = isFromReg ? eaiEntities[0] : eaiGateway.queryEai(request.getEai().getUid());
        if (null == eai) {
            throw new UnionEAINotFoundException();
        }
        List<EaiProdDTO> repeatProducts = findRepeatProducts(request.getProd());
        if (CollectionUtils.isNotEmpty(repeatProducts)) {
            eaiGateway.updateEaiStatus(eai.getId(), StateEnum.UNION_PROD_EXIST);
            return new EaiRegisterResponse(repeatProducts, StateEnum.UNION_PROD_EXIST);
        }
        EaiBO eaiBONew = EaiBOConvertor.convertToBO(request);
        List<String> tenants = eaiBONew.getTenants().stream().map(EaiTenantEntity::getTenantId).collect(Collectors.toList());
        List<EaiTenantEntity> existsEaiTenants = eaiGateway.queryOtherEaiTenantByTenantIds(tenants, eai.getUid());
        if (CollectionUtils.isNotEmpty(existsEaiTenants)) {
            log.error("tenant is bound:" + JsonUtil.toJson(existsEaiTenants));
            return new EaiRegisterResponse(StateEnum.UNION_TENANT_EXISTS, existsEaiTenants);
        }
        eaiBONew.getEai().setId(eai.getId());
        eaiGateway.updateEai(eaiBONew);
        //同步mdc
        syncMdcGateway.syncEai(eai, false);
        String action = isFromReg ? ACTION_REG : ACTION_UPDATE;
        StateEnum stateEnum = isFromReg ? StateEnum.REGEAI_REGED : StateEnum.REGEAI_UPDED;
        return new EaiRegisterResponse(action, espEai, findAllProducts(), stateEnum);
    }

    private List<EaiProdDTO> findRepeatProducts(List<EaiProdDTO> products) {
        List<EaiProdDTO> repeatProducts = new ArrayList<>();
        for (EaiProdDTO product : products) {
            boolean existsProduct = productService.validExistsProduct(product.getUid(), product.getName(), product.getIp(), product.getId());
            if (existsProduct) {
                repeatProducts.add(product.cloneWithoutSrv());
            }
        }
        return repeatProducts;
    }

    private List<EaiRegisterResponse.EspProduct> findAllProducts() {
        List<ProductEntity> espProducts = productQueryGateWay.queryAllNotClientOnlyProduct();
        return ProductBOConvertor.convertToEaiRegProductDTO(espProducts);
    }
}
