package com.digiwin.cross.app.command;

import com.digiwin.cross.app.convertor.ProductBOConvertor;
import com.digiwin.cross.app.convertor.ServiceBOConvertor;
import com.digiwin.cross.app.dto.request.ProductRegisterRequest;
import com.digiwin.cross.app.dto.request.ProductSrvRegisterRequest;
import com.digiwin.cross.app.dto.response.ProductRegisterResponse;
import com.digiwin.cross.app.dto.response.ProductSrvRegisterResponse;
import com.digiwin.cross.app.dto.response.SrvRegisterResponse;
import com.digiwin.cross.domain.bo.ProductBO;
import com.digiwin.cross.domain.bo.entity.ProductEntity;
import com.digiwin.cross.domain.bo.entity.ServiceEntity;
import com.digiwin.cross.domain.common.LockNamespaceEnum;
import com.digiwin.cross.domain.exception.*;
import com.digiwin.cross.domain.gateway.ILockGateway;
import com.digiwin.cross.domain.gateway.ISyncMdcGateway;
import com.digiwin.cross.domain.service.IProductService;
import com.digiwin.cross.domain.service.ISrvService;
import com.digiwin.cross.domain.state.StateEnum;
import com.digiwin.cross.infrastructure.http.HttpUtil;
import lombok.extern.apachecommons.CommonsLog;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;

/**
 * @description:
 * @author: liunansheng
 * @date: 2023/5/26 15:33
 */
@Component
@CommonsLog
public class ProductSrvRegisterCmd {

    private static final String SRV_NAME_REGEX = "^[a-zA-Z0-9_\\-,\\./]{1,100}($|;([a-zA-Z0-9._-]{1,20}$)|;(|[a-zA-Z0-9._-]{1,20});([\\-\u4e00-\u9fa5_a-zA-Z0-9]{1,100}$))";

    private static final String SRV_NAME_ERR_FORMAT = "Service register info is invalid. [%s]";


    private final IProductService productService;

    private final ISrvService srvService;

    private final HttpUtil httpUtil;

    private final ILockGateway lockGateway;

    private final ISyncMdcGateway syncMdcGateway;


    public ProductSrvRegisterCmd(IProductService productService, ISrvService srvService, HttpUtil httpUtil, ILockGateway lockGateway, ISyncMdcGateway syncMdcGateway) {
        this.productService = productService;
        this.srvService = srvService;
        this.httpUtil = httpUtil;
        this.lockGateway = lockGateway;
        this.syncMdcGateway = syncMdcGateway;
    }

    public ProductSrvRegisterResponse register(ProductSrvRegisterRequest request) throws BaseException {
        lockGateway.tryLock(LockNamespaceEnum.REG_PROD, request.getHost().getProd(), 100);
        try {
            ProductSrvRegisterRequest.Host host = request.getHost();
            boolean existsUnionProduct = productService.validExistsUnionProduct(host.getUid(), host.getProd(), host.getIp(), host.getId());
            if (existsUnionProduct) {
                throw new UnionProductExistException();
            }
            boolean checkUrlRs = httpUtil.checkRestURL(host.getResturl());
            if (!checkUrlRs) {
                throw new RESTParseException();
            }
            ProductEntity productBO = ProductBOConvertor.convertToProductBO(request);
            ProductEntity existsProduct = productService.findExistsProduct(host.getUid(), host.getProd(), host.getIp(), host.getId());
            if (existsProduct == null) {
                productService.saveProductBO(productBO);
            } else {
                productBO.setId(existsProduct.getId());
                productBO.setUid(existsProduct.getUid());
                productService.updateProductBO(productBO);
            }
            String uid = productBO.getUid();

            List<String> regFailSrvs = new ArrayList<>();
            List<String> successSrvs = new ArrayList<>();
            List<String> srvs = new ArrayList<>();
            //排除不合法的服务
            request.getService().getSrvname().forEach(srvName -> {
                if (!srvName.matches(SRV_NAME_REGEX)) {
                    log.error(String.format(SRV_NAME_ERR_FORMAT, srvName));
                    // 不合法的服務註冊字串
                    regFailSrvs.add(srvName);
                } else {
                    srvs.add(srvName);
                }
            });
            //开始注册服务
            lockGateway.tryLock(LockNamespaceEnum.REG_SERVICE, productBO.getId().toString(), 100);
            List<ServiceEntity> successSrvBOS = new ArrayList<>();
            try {
                srvs.forEach(one -> {
                    ServiceEntity srvBO = ServiceBOConvertor.convertToServiceBO(productBO, one);
                    boolean success = srvService.regService(productBO, srvBO);
                    if (success) {
                        successSrvs.add(one);
                        successSrvBOS.add(srvBO);
                    } else {
                        regFailSrvs.add(one);
                    }
                });
            } finally {
                lockGateway.unlock(LockNamespaceEnum.REG_SERVICE, productBO.getId().toString());
            }

            if (!successSrvBOS.isEmpty()) {
                ProductBO tProductBO = new ProductBO(productBO, successSrvBOS);
                syncMdcGateway.addProduct(tProductBO);
                srvService.checkDuplicate(successSrvBOS);

            }
            if (CollectionUtils.isNotEmpty(regFailSrvs)) {
                return new ProductSrvRegisterResponse(uid, successSrvs, regFailSrvs,StateEnum.REG_SRV_ONE_MORE_FAILED);
            } else {
                return new ProductSrvRegisterResponse(uid, successSrvs, regFailSrvs, StateEnum.REGPROD_REGED);
            }
        } finally {
            lockGateway.unlock(LockNamespaceEnum.REG_PROD, request.getHost().getProd());
        }
    }
}
