package com.digiwin.cross.app.command;

import com.digiwin.cross.app.convertor.QueryEaiLogsConverter;
import com.digiwin.cross.app.dto.QueryEaiLogsResultDTO;
import com.digiwin.cross.app.dto.request.QueryEaiLogsRequest;
import com.digiwin.cross.app.dto.response.QueryEaiLogsResponse;
import com.digiwin.cross.app.exception.*;
import com.digiwin.cross.domain.bo.entity.EaiEntity;
import com.digiwin.cross.domain.common.HeaderNamesConstant;
import com.digiwin.cross.domain.gateway.IEaiGateway;
import com.digiwin.cross.domain.utils.Base64Util;
import com.digiwin.cross.domain.utils.JsonUtil;
import com.digiwin.cross.infrastructure.mq.po.*;
import com.digiwin.queue.saas.SaasClient;
import lombok.extern.apachecommons.CommonsLog;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Component;

import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author clay
 * on 2023/12/7
 */
@Component
@CommonsLog
public class QueryEaiLogsCmd {

    private SaasClient mqClient;
    private IEaiGateway eaiGateway;

    public QueryEaiLogsCmd(SaasClient maClient, IEaiGateway eaiGateway) {
        this.mqClient = maClient;
        this.eaiGateway = eaiGateway;
    }

    public QueryEaiLogsResponse queryEaiLogs(QueryEaiLogsRequest request) throws Exception {
        try {

            EaiEntity eaiEntity = eaiGateway.getEaiByTenantId(request.getTenantId());

            if(eaiEntity == null) {
                return QueryEaiLogsResponse.builder().code(9).description("TenantId not found").build();
            }

            MqMessage tMqMessage = QueryEaiLogsConverter.convertToMqMessage(request.getReqId());

            final BlockingQueue<Boolean> tResultQueue = new ArrayBlockingQueue<Boolean>(1);
            final BlockingQueue<String> tResponseQueue = new ArrayBlockingQueue<String>(1);

            SaasClient.CompleteCallback tCompleteCallback = pMessage -> {
                log.info("Eai logs MQ return");
                MqResult tMqResult = JsonUtil.toJavaObj(pMessage, MqResult.class);
                EaiResult tEaiResult = tMqResult.getDetail().get(0);
                tResultQueue.offer(tEaiResult.getResult());
                tResponseQueue.offer(tEaiResult.getResponse());
            };
            SaasClient.TimeoutCallback tTimeoutCallback = pMessage -> {
                log.info("Eai logs MQ timeout");
                tResultQueue.offer(false);
                tResponseQueue.offer("timeout");
            };

            mqClient.onSend(eaiEntity.getLongConnId(), eaiEntity.getGatewayId(), JsonUtil.toJson(tMqMessage), tCompleteCallback, tTimeoutCallback, 20, TimeUnit.SECONDS);

            Boolean tResult = tResultQueue.take();
            String tMqResponse = tResponseQueue.take();

            RestResponse tEaiRestResponse;
            if (tResult) {
                EaiResponse tEaiResponse = JsonUtil.toJavaObj(tMqResponse, EaiResponse.class);
                String tRestMessage = Base64Util.decodeString(tEaiResponse.getMessage(), StandardCharsets.UTF_8.name());
                tEaiRestResponse = JsonUtil.toJavaObj(tRestMessage, RestResponse.class);
            } else {
                if ("timeout".equals(tMqResponse)) {
                    throw new EaiNotRespondException();
                } else {
                    throw new EaiErrorException(tMqResponse);
                }
            }

            String tDigiCode = tEaiRestResponse.getHeaders().get(HeaderNamesConstant.DIGI_CODE);
            if ("101".equals(tDigiCode)) {
                throw new EaiVersionNotSupportException();
            }

            QueryEaiLogsResultDTO tQueryResult = JsonUtil.toJavaObj(tEaiRestResponse.getMessagebody(), QueryEaiLogsResultDTO.class);
            if (!tQueryResult.getResult()) {
                throw new EaiErrorException(tQueryResult.getMessage());
            }

            if (CollectionUtils.isEmpty(tQueryResult.getLogs())) {
                throw new ReqIdNotFoundInEaiException();
            }
            return QueryEaiLogsResponse.builder().code(0).logs(tQueryResult.getLogs().stream().map(QueryEaiLogsConverter::convertToEaiLogDTO).collect(Collectors.toList())).build();
        } catch (QueryEaiLogsException e) {
            log.error(e.getMessage(), e);
            return QueryEaiLogsResponse.builder().code(e.getCode()).description(e.getMessage()).build();
        }
    }

}
