package com.digiwin.cross.app.command;

import com.digiwin.cross.app.convertor.ServiceBOConvertor;
import com.digiwin.cross.app.dto.request.SrvRegisterRequest;
import com.digiwin.cross.app.dto.response.SrvRegisterResponse;
import com.digiwin.cross.domain.bo.ProductBO;
import com.digiwin.cross.domain.bo.entity.ProductEntity;
import com.digiwin.cross.domain.bo.entity.ServiceEntity;
import com.digiwin.cross.domain.common.LockNamespaceEnum;
import com.digiwin.cross.domain.exception.BaseException;
import com.digiwin.cross.domain.exception.ClientOnlyProductCannotBeModifiedException;
import com.digiwin.cross.domain.exception.ProductNotFoundException;
import com.digiwin.cross.domain.gateway.ILockGateway;
import com.digiwin.cross.domain.gateway.ISyncMdcGateway;
import com.digiwin.cross.domain.service.IProductService;
import com.digiwin.cross.domain.service.ISrvService;
import com.digiwin.cross.domain.state.StateEnum;
import lombok.extern.apachecommons.CommonsLog;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @description:
 * @author: liunansheng
 * @date: 2023/5/26 15:33
 */
@CommonsLog
@Component
public class SrvRegisterCmd {

    private String SRV_NAME_REGEX = "^[a-zA-Z0-9_\\-,\\./]{1,100}($|;([a-zA-Z0-9._-]{1,20}$)|;(|[a-zA-Z0-9._-]{1,20});([\\-\u4e00-\u9fa5_a-zA-Z0-9]{1,100}$))";

    private String SRV_NAME_ERR_FORMAT = "Service register info is invalid. [%s]";

    private final IProductService productService;

    private final ISrvService srvService;

    private final ISyncMdcGateway syncMdcGateway;

    private final ILockGateway lockGateway;

    public SrvRegisterCmd(IProductService productService, ISrvService srvService, ISyncMdcGateway syncMdcGateway, ILockGateway lockGateway) {
        this.productService = productService;
        this.srvService = srvService;
        this.syncMdcGateway = syncMdcGateway;
        this.lockGateway = lockGateway;
    }

    public SrvRegisterResponse registerSrv(SrvRegisterRequest request) throws BaseException {
        SrvRegisterRequest.Host host = request.getHost();

        ProductEntity product = productService.findProduct(host.getUid(), host.getProd(), host.getIp(), host.getId());
        if (null == product) {
            throw new ProductNotFoundException();
        }
        if (BooleanUtils.isTrue(product.getClientOnly())) {
            throw new ClientOnlyProductCannotBeModifiedException();
        }
        List<String> regFailSrvs = new ArrayList<>();
        List<String> successSrvs = new ArrayList<>();
        List<String> srvs = new ArrayList<>();
        //排除不合法的服务
        request.getService().getSrvname().forEach(srvName -> {
            if (!srvName.matches(SRV_NAME_REGEX)) {
                log.error(String.format(SRV_NAME_ERR_FORMAT, srvName));
                // 不合法的服務註冊字串
                regFailSrvs.add(srvName);
            } else {
                srvs.add(srvName);
            }
        });
        if (CollectionUtils.isEmpty(srvs)) {
            return new SrvRegisterResponse(new SrvRegisterResponse.Result(null, regFailSrvs), StateEnum.REG_SRV_ONE_MORE_FAILED);
        }
        log.info("开始执行注册服务");
        //开始注册服务
        lockGateway.tryLock(LockNamespaceEnum.REG_SERVICE, product.getId().toString(), 100);
        log.info("开始执行注册服务-加锁成功");
        List<ServiceEntity> successSrvBOS = new ArrayList<>();
        try {
            srvs.forEach(one -> {
                ServiceEntity srvBO = ServiceBOConvertor.convertToServiceBO(product, one);
                boolean success = srvService.regService(product, srvBO);
                if (success) {
                    successSrvs.add(one);
                    successSrvBOS.add(srvBO);
                } else {
                    regFailSrvs.add(one);
                }
            });
        } finally {
            lockGateway.unlock(LockNamespaceEnum.REG_SERVICE, product.getId().toString());
        }

        if (!successSrvBOS.isEmpty()) {
            ProductBO tProductBO = new ProductBO(product, successSrvBOS);
            log.info("开始执行注册服务-同步MDC");
            syncMdcGateway.addProduct(tProductBO);
            log.info("开始执行注册服务-检查重复");
            srvService.checkDuplicate(successSrvBOS);
            log.info("开始执行注册服务-检查重复结束");
        }
        if (CollectionUtils.isNotEmpty(regFailSrvs)) {
            return new SrvRegisterResponse(new SrvRegisterResponse.Result(successSrvs, regFailSrvs), StateEnum.REG_SRV_ONE_MORE_FAILED);
        } else {
            return new SrvRegisterResponse(new SrvRegisterResponse.Result(successSrvs, null), StateEnum.REGSRV_REGED);
        }
    }

    public SrvRegisterResponse unRegisterSrv(SrvRegisterRequest request) throws BaseException {
        SrvRegisterRequest.Host host = request.getHost();
        ProductEntity product = productService.findProduct(host.getUid(), host.getProd(), host.getIp(), host.getId());
        if (null == product) {
            throw new ProductNotFoundException();
        }
        if (BooleanUtils.isTrue(product.getClientOnly())) {
            throw new ClientOnlyProductCannotBeModifiedException();
        }
        List<String> regFailSrvs = new ArrayList<>();
        List<String> successSrvs = new ArrayList<>();
        try {
            lockGateway.tryLock(LockNamespaceEnum.REG_SERVICE, product.getId().toString(), 100);
            request.getService().getSrvname().forEach(srvName -> {
                boolean success = srvService.unRegisterSrv(product, srvName);
                if (success) {
                    successSrvs.add(srvName);
                } else {
                    regFailSrvs.add(srvName);
                }
            });
            if (CollectionUtils.isNotEmpty(successSrvs)) {
                //添加标记，等全量同步
                syncMdcGateway.addFailFlag();
            }
            if (CollectionUtils.isNotEmpty(regFailSrvs)) {
                return new SrvRegisterResponse(new SrvRegisterResponse.Result(successSrvs, regFailSrvs), StateEnum.REG_SRV_ONE_MORE_FAILED);
            } else {
                return new SrvRegisterResponse(new SrvRegisterResponse.Result(successSrvs, null), StateEnum.REGSRV_REGED);
            }
        } finally {
            lockGateway.unlock(LockNamespaceEnum.REG_SERVICE, product.getId().toString());
        }
    }

}
