package com.digiwin.cross.app.command;

import com.digiwin.cross.domain.bo.FullAsyncRequestBO;
import com.digiwin.cross.domain.common.LockNamespaceEnum;
import com.digiwin.cross.domain.gateway.IFullAsyncRequestGateway;
import com.digiwin.cross.domain.gateway.ILockGateway;
import com.digiwin.cross.domain.utils.SpringContextUtil;
import lombok.extern.apachecommons.CommonsLog;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.time.DateUtils;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.io.Serializable;
import java.util.Date;
import java.util.List;

/**
 * @description:
 * @author: liunansheng
 * @date: 2023/9/22 15:21
 */
@CommonsLog
@Component
public class CallBackTimeOutCheckCmd {

    public final static String TASK_NAME = "esp:callback_check2";

    private final IFullAsyncRequestGateway fullAsyncRequestGateway;

    private final ILockGateway lockGateway;

    public CallBackTimeOutCheckCmd(IFullAsyncRequestGateway fullAsyncRequestGateway, ILockGateway lockGateway) {
        this.fullAsyncRequestGateway = fullAsyncRequestGateway;
        this.lockGateway = lockGateway;
    }

    @Scheduled(fixedDelay = 60 * 1000)
    public void doTask() {
        if (!lockGateway.tryLock(LockNamespaceEnum.TASK_LOCK, TASK_NAME, 10, 120)) {
            log.info("task is running");
            return;
        }
        try {
            log.info("callback time out check task start");
            Date end = new Date();
            //只检查1天以内的数据
            Date start = DateUtils.addDays(end, -1);
            List<FullAsyncRequestBO> callBackList = fullAsyncRequestGateway.getTimeOutFullAsyncRequestInfo(start, end);
            if (CollectionUtils.isEmpty(callBackList)) {
                log.info("callback time out check is no data");
                return;
            }
            callBackList.forEach(one -> {
                try {
                    fullAsyncRequestGateway.alarmCallbackTimeOut(one);
                    fullAsyncRequestGateway.saveTimeOutCheckTag(one.getReqId());
                } catch (Exception e) {
                    log.error("send callback timeout info error，reqId: " + one.getReqId(), e);
                }
            });
        } catch (Exception e) {
            log.error("callback check err", e);
        } finally {
            lockGateway.unlock(LockNamespaceEnum.TASK_LOCK, TASK_NAME);
        }
    }

    public static class CallBackCheckRunable implements Runnable, Serializable {
        @Override
        public void run() {
            CallBackTimeOutCheckCmd cmd = SpringContextUtil.getBean(CallBackTimeOutCheckCmd.class);
            if (cmd != null) {
                cmd.doTask();
            }
        }
    }
}
