package com.digiwin.cross.app.command;

import com.digiwin.cross.app.convertor.ProductBOConvertor;
import com.digiwin.cross.app.dto.request.ProductRegisterRequest;
import com.digiwin.cross.app.dto.response.ProductRegisterResponse;
import com.digiwin.cross.domain.bo.entity.ProductEntity;
import com.digiwin.cross.domain.common.LockNamespaceEnum;
import com.digiwin.cross.domain.exception.BaseException;
import com.digiwin.cross.domain.exception.ClientOnlyProductCannotBeModifiedException;
import com.digiwin.cross.domain.exception.InvokeRestFormatValidationException;
import com.digiwin.cross.domain.exception.ProductExistException;
import com.digiwin.cross.domain.exception.ProductNotFoundException;
import com.digiwin.cross.domain.exception.RESTParseException;
import com.digiwin.cross.domain.exception.UnionProductExistException;
import com.digiwin.cross.domain.gateway.ILockGateway;
import com.digiwin.cross.domain.gateway.ISyncMdcGateway;
import com.digiwin.cross.domain.service.IProductService;
import com.digiwin.cross.domain.state.StateEnum;
import com.digiwin.cross.infrastructure.http.HttpUtil;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

/**
 * @description:
 * @author: liunansheng
 * @date: 2023/5/26 15:33
 */
@Component
public class ProductRegisterCmd {

    private final IProductService productService;

    private final HttpUtil httpUtil;

    private final ILockGateway lockGateway;

    private final ISyncMdcGateway syncMdcGateway;


    public ProductRegisterCmd(IProductService productService, HttpUtil httpUtil, ILockGateway lockGateway, ISyncMdcGateway syncMdcGateway) {
        this.productService = productService;
        this.httpUtil = httpUtil;
        this.lockGateway = lockGateway;
        this.syncMdcGateway = syncMdcGateway;
    }

    public ProductRegisterResponse registerProduct(ProductRegisterRequest request) throws BaseException {
        lockGateway.tryLock(LockNamespaceEnum.REG_PROD, request.getHost().getProd(), 100);
        try {
            ProductRegisterRequest.Host host = request.getHost();
            boolean existsProduct = productService.validExistsProduct(host.getUid(), host.getProd(), host.getIp(), host.getId());
            if (existsProduct) {
                throw new ProductExistException();
            }
            existsProduct = productService.validExistsUnionProduct(host.getUid(), host.getProd(), host.getIp(), host.getId());
            if (existsProduct) {
                throw new UnionProductExistException();
            }
            if (BooleanUtils.isNotTrue(host.getClientonly())) {
                boolean checkUrlRs = httpUtil.checkRestURL(host.getResturl());
                if (!checkUrlRs) {
                    throw new RESTParseException();
                }
            }
            ProductEntity productBO = ProductBOConvertor.convertToProductBO(request);
            productService.saveProductBO(productBO);
            return new ProductRegisterResponse(productBO.getUid(), StateEnum.REGPROD_REGED);
        } finally {
            lockGateway.unlock(LockNamespaceEnum.REG_PROD, request.getHost().getProd());
        }
    }

    public ProductRegisterResponse updateProduct(ProductRegisterRequest request) throws BaseException {
        lockGateway.tryLock(LockNamespaceEnum.REG_PROD, request.getHost().getProd(), 100);
        try {
            ProductRegisterRequest.Host host = request.getHost();
            if (StringUtils.isBlank(host.getUid())) {
                throw new InvokeRestFormatValidationException("uid is null");
            }
            ProductEntity productEntity = productService.findProduct(host.getUid(), null, null, null);
            if (null == productEntity) {
                throw new ProductNotFoundException();
            }
            if (BooleanUtils.isTrue(productEntity.getClientOnly()) || BooleanUtils.isTrue(host.getClientonly())) {
                throw new ClientOnlyProductCannotBeModifiedException();
            }
            boolean existsProduct = productService.validExistsOtherProduct(host.getProd(), host.getIp(), host.getId(), productEntity.getId());
            if (existsProduct) {
                throw new ProductExistException();
            }
            existsProduct = productService.validExistsUnionProduct(host.getUid(), host.getProd(), host.getIp(), host.getId());
            if (existsProduct) {
                throw new UnionProductExistException();
            }
            if (BooleanUtils.isNotTrue(host.getClientonly())) {
                boolean checkUrlRs = httpUtil.checkRestURL(host.getResturl());
                if (!checkUrlRs) {
                    throw new RESTParseException();
                }
            }
            ProductEntity productBO = ProductBOConvertor.convertToProductBO(request);
            productBO.setId(productEntity.getId());
            if(productBO.getAppId() == null) {
                productBO.setAppId(productEntity.getAppId());
            }
            if(productBO.getSettingEocFlag() == null) {
                productBO.setSettingEocFlag(productEntity.getSettingEocFlag());
            }
            ProductEntity newProduct = productService.updateProductBO(productBO);
            //暂时通过排程全量同步
            syncMdcGateway.addFailFlag();
            return new ProductRegisterResponse(productBO.getUid(), StateEnum.REGPROD_UPDED);
        } finally {
            lockGateway.unlock(LockNamespaceEnum.REG_PROD, request.getHost().getProd());
        }
    }

    public ProductRegisterResponse unRegisterProduct(ProductRegisterRequest request) throws BaseException {
        lockGateway.tryLock(LockNamespaceEnum.REG_PROD, request.getHost().getProd(), 100);
        try {
            ProductRegisterRequest.Host host = request.getHost();
            if (StringUtils.isBlank(host.getUid())) {
                throw new InvokeRestFormatValidationException("uid is null");
            }
            ProductEntity productEntity = productService.deleteProduct(host.getUid());
            if (null == productEntity) {
                return new ProductRegisterResponse(null, StateEnum.REGPROD_DELED);
            }
            //暂时通过排程全量同步
            syncMdcGateway.addFailFlag();
            return new ProductRegisterResponse(productEntity.getUid(), StateEnum.REGPROD_DELED);
        } finally {
            lockGateway.unlock(LockNamespaceEnum.REG_PROD, request.getHost().getProd());
        }
    }
}
