package com.digiwin.cross.app.command;

import com.digiwin.cross.app.convertor.CallbackConvertor;
import com.digiwin.cross.app.dto.request.CallbackRequest;
import com.digiwin.cross.app.dto.response.CallBackResponse;
import com.digiwin.cross.domain.bo.CallResponseBO;
import com.digiwin.cross.domain.bo.FullAsyncRequestBO;
import com.digiwin.cross.domain.bo.ServiceResponseBO;
import com.digiwin.cross.domain.common.HeaderNamesConstant;
import com.digiwin.cross.domain.common.LockNamespaceEnum;
import com.digiwin.cross.domain.convertor.InvokeConvertor;
import com.digiwin.cross.domain.exception.BaseException;
import com.digiwin.cross.domain.exception.ServiceCallbackReqidNotFoundException;
import com.digiwin.cross.domain.exception.ServiceCallbackReqidProcessedException;
import com.digiwin.cross.domain.gateway.IFullAsyncRequestGateway;
import com.digiwin.cross.domain.gateway.ILockGateway;
import com.digiwin.cross.domain.service.impl.CallBackService;
import com.digiwin.cross.domain.utils.CallBackLogUtil;
import lombok.extern.apachecommons.CommonsLog;
import org.springframework.stereotype.Component;

import java.util.concurrent.TimeUnit;

/**
 * @description:
 * @author: liunansheng
 * @date: 2023/6/21 16:44
 */
@CommonsLog
@Component
public class CallbackCmd {

    private final CallBackService callBackService;

    private final IFullAsyncRequestGateway fullAsyncRequestGateway;

    private final ILockGateway lockGateway;

    public CallbackCmd (CallBackService callBackService, IFullAsyncRequestGateway fullAsyncRequestGateway, ILockGateway lockGateway) {
        this.callBackService = callBackService;
        this.fullAsyncRequestGateway = fullAsyncRequestGateway;
        this.lockGateway = lockGateway;
    }

    public CallBackResponse execute (CallbackRequest request) throws BaseException {
        String reqId = request.getHeaders().get(HeaderNamesConstant.DIGI_REQID);
        boolean locked = false;
        try {
            //等待全異步請求寫入DB解鎖
            //FIXME 華為測試區的redis鎖有BUG，無法正常運作，暫時改為用重試取得
//            boolean asked = lockGateway.tryLock(LockNamespaceEnum.EAI_FULLASYNC_ASKING, reqId, 30, 5);
//            try {
//                if (!asked) {
//                    log.error(String.format("Service asking ReqId: %s is locked, callback failed", reqId));
//                    throw new ServiceCallbackReqidNotFoundException();
//                }
//            } finally {
//                if (asked) {
//                    lockGateway.unlock(LockNamespaceEnum.EAI_FULLASYNC_ASKING, reqId);
//                }
//            }

            //鎖定全異步callback執行中
            if (lockGateway.tryLock(LockNamespaceEnum.EAI_FULLASYNC_REQID, reqId, 0, 130)) {
                log.info(String.format("Service callback ReqId: %s is locked", reqId));
            } else {
                throw new ServiceCallbackReqidProcessedException();
            }
            locked = true;
            FullAsyncRequestBO context = fullAsyncRequestGateway.getFullAsyncRequestInfo(reqId, 6, TimeUnit.SECONDS.toMillis(5));
            if (null == context) {
                throw new ServiceCallbackReqidNotFoundException();
            }

            // 修改表中已经存在回调的标志
            fullAsyncRequestGateway.setCallBackTag(reqId);
            CallBackLogUtil.beforeCallBackRequester(request.getRestUrl(), context, request.getHeaders(), request.getBody());
            ServiceResponseBO serviceResponseBO = InvokeConvertor.convertToServiceResponseBO(request);
            CallResponseBO rs = callBackService.call(context, serviceResponseBO);
            if (rs.isBusinessSuccess()) {
                fullAsyncRequestGateway.removeFullAsyncRequestInfo(serviceResponseBO.getReqId());
            }
            return CallbackConvertor.convertRs(rs);
        } finally {
            if (locked) {
                log.info(String.format("Service callback ReqId: %s is unlocked", reqId));
                lockGateway.unlock(LockNamespaceEnum.EAI_FULLASYNC_REQID, reqId);
            }
        }
    }
}
