package com.digiwin.cross.app.command;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.digiwin.athena.esp.sdk.util.StringUtil;
import com.digiwin.cross.app.convertor.ProductBOConvertor;
import com.digiwin.cross.app.dto.request.GetProdRequest;
import com.digiwin.cross.app.dto.response.*;
import com.digiwin.cross.domain.bo.entity.ProductEntity;
import com.digiwin.cross.domain.exception.AmbiguousProductException;
import com.digiwin.cross.domain.exception.BaseException;
import com.digiwin.cross.domain.exception.ProductNotFoundException;
import com.digiwin.cross.domain.gateway.IProductQueryGateWay;
import com.digiwin.cross.domain.gateway.IServiceQueryGateway;
import com.digiwin.cross.domain.state.StateEnum;
import com.digiwin.cross.infrastructure.database.entity.EaiTenantMappingPO;
import com.digiwin.cross.infrastructure.database.entity.UnionEAIPO;
import com.digiwin.cross.infrastructure.database.entity.UnionServicePO;
import com.digiwin.cross.infrastructure.database.mapper.EaiTenantMappingMapper;
import com.digiwin.cross.infrastructure.database.mapper.UnionEaiMapper;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Component;

import java.util.List;

/**
 * @description:
 * @author: liunansheng
 * @date: 2023/7/19 9:50
 */
@Component
public class RegInfoQueryCmd {

    private final IProductQueryGateWay productQueryGateWay;

    private final IServiceQueryGateway serviceQueryGateway;

    // 租户与中台映射Mapper
    private final EaiTenantMappingMapper eaiTenantMappingMapper;

    // 远端中台Mapper
    private final UnionEaiMapper unionEaiMapper;

    public RegInfoQueryCmd(IProductQueryGateWay productQueryGateWay, IServiceQueryGateway serviceQueryGateway,
                           EaiTenantMappingMapper eaiTenantMappingMapper, UnionEaiMapper unionEaiMapper) {
        this.productQueryGateWay = productQueryGateWay;
        this.serviceQueryGateway = serviceQueryGateway;
        this.eaiTenantMappingMapper = eaiTenantMappingMapper;
        this.unionEaiMapper = unionEaiMapper;
    }

    public GetProdListResponse getAllProd() {
        List<ProductEntity> productBOs = productQueryGateWay.queryAllProduct();
        return new GetProdListResponse(StateEnum.INFO_PRODLIST, ProductBOConvertor.convertToProductDto(productBOs));
    }

    /**
     * 获取地端产品清单
     * add by songwq 20250715
     * 针对集成中心开放接口 story-view-48716
     * @param eaiId 地端ID
     * @param tenantId 租户
     * @return
     */
    public GetUnionProdListResponse getUnionProdList(Long eaiId, String tenantId) {
        // 如果传了租户ID，就会以租户ID为准仅查询
        if (StringUtil.isNotEmpty(tenantId)) {
            EaiTenantMappingPO eaiTenantMappingPO = eaiTenantMappingMapper.selectOne(
                    Wrappers.<EaiTenantMappingPO>lambdaQuery()
                            .eq(EaiTenantMappingPO::getTenantId, tenantId));

            // 根据eaiUid查询中台的ID记录
            UnionEAIPO unionEAIPO = null;
            if (null != eaiTenantMappingPO && StringUtil.isNotEmpty(eaiTenantMappingPO.getEaiUid())) {
                unionEAIPO = unionEaiMapper.selectOne(
                        Wrappers.<UnionEAIPO>lambdaQuery()
                                .eq(UnionEAIPO::getUid, eaiTenantMappingPO.getEaiUid()));
            }
            if (null != unionEAIPO && null != unionEAIPO.getId()) {
                eaiId = unionEAIPO.getId();
            }
        }

        // 查询产品清单
        List<ProductEntity> productBOs = productQueryGateWay.queryUnionProduct(eaiId);
        return new GetUnionProdListResponse(StateEnum.INFO_PRODLIST, ProductBOConvertor.convertToUnionProductDto(productBOs));
    }

    public GetProdResponse getProd(GetProdRequest request) {
        ProductEntity condition = ProductBOConvertor.convertToProductBO(request);
        List<ProductEntity> productBOs = productQueryGateWay.queryProductByCondition(condition);
        return new GetProdResponse(StateEnum.INFO_PROD, ProductBOConvertor.convertToGetProdResInfo(productBOs));
    }

    public GetSrvResponse getSrv(GetProdRequest request) throws BaseException {
        ProductEntity condition = ProductBOConvertor.convertToProductBO(request);
        List<ProductEntity> productBOs = productQueryGateWay.queryProductByCondition(condition);
        if (CollectionUtils.isEmpty(productBOs)) {
            throw new ProductNotFoundException();
        }
        if (productBOs.size() > 1) {
            throw new AmbiguousProductException();
        }
        List<String> srvs = serviceQueryGateway.getSrvByProdId(productBOs.get(0).getId());
        return new GetSrvResponse(StateEnum.INFO_SRVLIST, srvs);
    }

    /**
     * 获取地端产品下的服务清单
     * add by songwq 20250715
     * 针对集成中心开放接口 story-view-48716
     * @param pid 产品ID
     * @param name 服务名称
     * @param page 分页对象
     * @return
     * @throws BaseException
     */
    public GetUnionSrvResponse getUnionSrvList(Long pid, String name, Page page) throws BaseException {
        Page<UnionServicePO> srvs = serviceQueryGateway.getUnionServiceLis(pid, name, page);
        return new GetUnionSrvResponse(StateEnum.INFO_SRVLIST, srvs.getRecords(), 200,"success", srvs.getTotal());
    }

}
