package com.digiwin.cross.domain.service.impl;

import com.digiwin.cross.domain.bo.entity.ProductEntity;
import com.digiwin.cross.domain.bo.entity.ServiceEntity;
import com.digiwin.cross.domain.bo.entity.ServiceProdListEntity;
import com.digiwin.cross.domain.gateway.IServiceEditGateway;
import com.digiwin.cross.domain.gateway.IServiceQueryGateway;
import com.digiwin.cross.domain.parameter.ApplicationSystemParameter;
import com.digiwin.cross.domain.service.ISrvService;
import lombok.extern.apachecommons.CommonsLog;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * @description:
 * @author: liunansheng
 * @date: 2023/6/8 16:58
 */
@CommonsLog
@Service
public class SrvService implements ISrvService {

    private final IServiceQueryGateway serviceQueryGateway;

    private final IServiceEditGateway serviceEditGateway;

    private final List<String> ignoreDuplicateServiceNameList = new ArrayList<>();

    public SrvService(IServiceQueryGateway serviceQueryGateway, IServiceEditGateway serviceEditGateway) {
        this.serviceQueryGateway = serviceQueryGateway;
        this.serviceEditGateway = serviceEditGateway;
        if (StringUtils.isNotBlank(ApplicationSystemParameter.IGNORE_DUPLICATE_SERVICE_NAME)) {
            this.ignoreDuplicateServiceNameList.addAll(Arrays.asList(ApplicationSystemParameter.IGNORE_DUPLICATE_SERVICE_NAME.split(",")));
        }
    }

    @Override
    public boolean regService(ProductEntity productEntity, ServiceEntity srvBO) {
        try {
            ServiceEntity oldSrv = serviceQueryGateway.getSrvByName(srvBO.getProductId(), srvBO.getServiceName(), srvBO.getVersion() != null);
            if (null != oldSrv) {
                srvBO.setId(oldSrv.getId());
                if (oldSrv.getVersion() != null && srvBO.getVersion() != null
                        && StringUtils.equals(oldSrv.getVersion().getVersion(), srvBO.getVersion().getVersion())) {
                    //存在一致的版本号，无需新增
                    srvBO.setVersion(null);
                } else if (oldSrv.getVersion() != null && srvBO.getVersion() != null) {
                    srvBO.getVersion().setId(oldSrv.getVersion().getId());
                }
                if (srvBO.getVersion() != null) {
                    srvBO.getVersion().setSrvId(oldSrv.getId());
                }
            }
            serviceEditGateway.saveOrUpdateSrv(productEntity, srvBO);
            return true;
        } catch (Exception e) {
            log.error(e);
            return false;
        }
    }

    @Override
    public boolean unRegisterSrv(ProductEntity product, String serviceName) {
        try {
            ServiceEntity oldSrv = serviceQueryGateway.getSrvByName(product.getId(), serviceName, false);
            if (null == oldSrv) {
                log.error(String.format("unreg srv [%s], srv not exists", serviceName));
                return false;
            }
            serviceEditGateway.deleteSrv(product, oldSrv);
            return true;
        } catch (Exception e) {
            log.error(e);
            return false;
        }
    }

    @Override
    public void checkDuplicate(List<ServiceEntity> serviceEntities) {
        if (CollectionUtils.isNotEmpty(serviceEntities)) {
            List<Pattern> ignoreDuplicateServiceNamePatterns = ignoreDuplicateServiceNameList.stream()
                    .map(Pattern::compile)
                    .collect(Collectors.toList());

            List<String> tSrvNames = serviceEntities.stream()
                    .map(ServiceEntity::getServiceName)
                    .filter(srvName -> ignoreDuplicateServiceNamePatterns.stream()
                            .noneMatch(pattern -> pattern.matcher(srvName).matches()))
                    .collect(Collectors.toList());

            if (CollectionUtils.isNotEmpty(tSrvNames)) {
                List<ServiceProdListEntity> tServiceProdListEntityList = serviceQueryGateway.queryServiceWithProductNames(tSrvNames);
                String tRegProd = serviceEntities.get(0).getProdName();
                List<ServiceProdListEntity> tDuplicateEntityList = tServiceProdListEntityList.stream().filter(entity -> {
                    if (entity.getProductNames().stream().anyMatch(prodName -> !prodName.equals(tRegProd))) {
                        entity.getProductNames().remove(tRegProd);
                        return true;
                    } else {
                        return false;
                    }
                }).collect(Collectors.toList());
                if (CollectionUtils.isNotEmpty(tDuplicateEntityList)) {
                    serviceQueryGateway.alarmDuplicateService(tDuplicateEntityList, tRegProd);
                }
            }
        }
    }
}
