package com.digiwin.cross.domain.service.validator;

import com.digiwin.cross.domain.exception.JsonValidatorException;
import com.fasterxml.jackson.core.JsonParser.Feature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.networknt.schema.JsonSchema;
import com.networknt.schema.JsonSchemaFactory;
import com.networknt.schema.SpecVersion.VersionFlag;
import com.networknt.schema.ValidationMessage;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.Set;

public abstract class JsonValidator {

	protected Logger log;
	private JsonSchema schema;
	private ObjectMapper objectMapper = new ObjectMapper();

	public JsonValidator(Class<? extends JsonValidator> pClass, String pSchemaFileName) throws Exception {
		log = LoggerFactory.getLogger(pClass);
		schema = JsonSchemaFactory.getInstance(VersionFlag.V4).getSchema(objectMapper.readTree(ClassLoader.getSystemResourceAsStream("jsonSchema/" + pSchemaFileName)));
		objectMapper.configure(Feature.ALLOW_SINGLE_QUOTES, true);
	}

	public void validate(String pJsonStr, String pTargetName) throws JsonValidatorException {
		if (StringUtils.isBlank(pJsonStr)) {
			String tMessage = String.format("'%s' content is empty", pTargetName);
			throw new JsonValidatorException(tMessage);
		}
		try {
			Set<ValidationMessage> tValidationMessages = schema.validate(objectMapper.readTree(pJsonStr));
			if (tValidationMessages.size() > 0) {
				JsonObject tErrorMessage = new JsonObject();
				JsonArray tDetialArray = new JsonArray();
				tErrorMessage.add(pTargetName, tDetialArray);
				for (ValidationMessage tValidationMessage : tValidationMessages) {
					tDetialArray.add(tValidationMessage.getMessage());
				}
				throw new JsonValidatorException(tErrorMessage.toString());
			}
		} catch (IOException e) {
			log.error(e.getMessage(), e);
			String tMessage = String.format("'%s' cant't parse: %s", pTargetName, e.getMessage());
			throw new JsonValidatorException(tMessage);
		}
	}
}
