package com.digiwin.cross.domain.service.validator;

import com.digiwin.cross.domain.bo.EaiBO;
import com.digiwin.cross.domain.bo.ProductBO;
import com.digiwin.cross.domain.bo.ProviderInfoBO;
import com.digiwin.cross.domain.bo.RequesterInfoBO;
import com.digiwin.cross.domain.bo.entity.EaiEntity;
import com.digiwin.cross.domain.bo.entity.ProductEntity;
import com.digiwin.cross.domain.bo.entity.ServiceEntity;
import com.digiwin.cross.domain.exception.BaseException;
import com.digiwin.cross.domain.exception.DuplicateProductException;
import com.digiwin.cross.domain.gateway.IEaiGateway;
import com.digiwin.cross.domain.gateway.IProductQueryGateWay;
import com.digiwin.cross.domain.gateway.IServiceQueryGateway;
import com.digiwin.cross.domain.service.validator.bo.ProviderBO;
import com.digiwin.cross.domain.service.validator.bo.RequesterBO;
import com.digiwin.cross.domain.service.validator.exception.CrossRequesterNotFoundException;
import com.digiwin.cross.domain.service.validator.exception.RequesterProductNotFoundException;
import com.digiwin.cross.domain.service.validator.exception.ServiceNameNotFoundException;
import com.digiwin.cross.domain.service.validator.exception.ServiceProviderNotFoundException;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.stream.Collectors;

/**
 * @description:
 * @author: clay
 * @date: 2023/6/12
 */
@Component
public class ProductValidator {

    private IProductQueryGateWay productQueryGateWay;

    private IServiceQueryGateway serviceQueryGateway;

    private IEaiGateway eaiGateway;

    public ProductValidator(IProductQueryGateWay productQueryGateWay, IServiceQueryGateway serviceQueryGateway, IEaiGateway eaiGateway) {
        this.productQueryGateWay = productQueryGateWay;
        this.serviceQueryGateway = serviceQueryGateway;
        this.eaiGateway = eaiGateway;
    }

    public RequesterBO validateRequester(RequesterInfoBO pRequesterInfoBO) throws BaseException {
        RequesterBO tRequesterBO = new RequesterBO();

        if (pRequesterInfoBO.getEaiUid() == null) {
            //驗證敏態發起端
            ProductEntity tProductEntity = validateRequesterProduct(pRequesterInfoBO);
            if (tProductEntity == null) {
                throw new RequesterProductNotFoundException();
            }
            tRequesterBO.setProduct(tProductEntity);
        } else {
            //驗證穩態發起端中台
            EaiEntity tEaiEntity = validateRequesterEai(pRequesterInfoBO.getEaiUid());
            if (tEaiEntity == null) {
                throw new CrossRequesterNotFoundException();
            }
            tRequesterBO.setEai(tEaiEntity);
        }
        return tRequesterBO;
    }

    private ProductEntity validateRequesterProduct(RequesterInfoBO pRequesterInfo) {
        //cache
        ProductEntity tSelectedProduct = productQueryGateWay.findProductInCache(pRequesterInfo.getProd(), pRequesterInfo.getIp(), pRequesterInfo.getId(), pRequesterInfo.getUid());
        //database
        if (tSelectedProduct == null) {
            tSelectedProduct = productQueryGateWay.findProductInDB(pRequesterInfo.getProd(), pRequesterInfo.getIp(), pRequesterInfo.getId(), pRequesterInfo.getUid());
            //寫回cache
            if (tSelectedProduct != null) {
                productQueryGateWay.saveProductCache(tSelectedProduct);
            }
        }

        //client only cache
        if (tSelectedProduct == null) {
            List<ProductEntity> tProductEntityList = productQueryGateWay.findProductInCache(pRequesterInfo.getProd());
            if (CollectionUtils.isNotEmpty(tProductEntityList)) {
                List<ProductEntity> tClientOnlyProductEntityList = tProductEntityList.stream().filter(ProductEntity::getClientOnly).collect(Collectors.toList());
                if (CollectionUtils.isNotEmpty(tClientOnlyProductEntityList)) {
                    tSelectedProduct = tClientOnlyProductEntityList.get(0);
                }
            }
        }
        //client only DB
        if (tSelectedProduct == null) {
            List<ProductEntity> tProductEntityList = productQueryGateWay.findProductInDB(pRequesterInfo.getProd());
            if (CollectionUtils.isNotEmpty(tProductEntityList)) {
                List<ProductEntity> tClientOnlyProductEntityList = tProductEntityList.stream().filter(ProductEntity::getClientOnly).collect(Collectors.toList());
                if (CollectionUtils.isNotEmpty(tClientOnlyProductEntityList)) {
                    tSelectedProduct = tClientOnlyProductEntityList.get(0);
                }
            }
            //寫回cache
            if (tSelectedProduct != null) {
                productQueryGateWay.saveProductCache(tSelectedProduct);
            }
        }


        return tSelectedProduct;
    }


    private EaiEntity validateRequesterEai(String pEaiUid) {
        EaiEntity tSelectedEai = eaiGateway.getEaiInCache(pEaiUid);
        if (tSelectedEai == null) {
            tSelectedEai = eaiGateway.queryEai(pEaiUid);
            //寫回Cache
            if (tSelectedEai != null) {
                eaiGateway.saveEaiCache(tSelectedEai);
            }
        }

        return tSelectedEai;
    }

    public ProviderBO validateProvider(ProviderInfoBO pProviderInfoBO) throws BaseException {
        ProviderBO tProviderBO = new ProviderBO();

        boolean hasUnionInfo = StringUtils.isNotBlank(pProviderInfoBO.getTenantId()) && StringUtils.isNotBlank(pProviderInfoBO.getUid());


        boolean foundProvider = false;

        //找產品cache
        if (hasUnionInfo) {
            //有tenantId及uid先找穩態Cache

            //找EAI
            EaiEntity eaiEntity = queryProviderEaiInCache(pProviderInfoBO.getTenantId());
            if (eaiEntity != null) {
                //找產品
                ProductEntity productEntity = queryProviderUnionProductInCache(pProviderInfoBO, eaiEntity.getUid());
                if (productEntity != null) {
                    tProviderBO.setProduct(productEntity);
                    tProviderBO.setEai(eaiEntity);
                    foundProvider = true;
                }
            }
        }

        if (!foundProvider) {
            //找敏態Cache
            ProductEntity productEntity = queryProviderProductInCache(pProviderInfoBO);
            if (productEntity != null) {
                tProviderBO.setProduct(productEntity);
                foundProvider = true;
            }
        }

        //cacche找不到從DB找
        if (!foundProvider) {
            if (hasUnionInfo) {
                //有tenantId及uid先找穩態DB
                EaiBO tEaiBO = queryEaiBOWithUnionProductsInDB(pProviderInfoBO.getTenantId());
                if (tEaiBO != null) {
                    ProductBO productBO = queryProviderUnionProductInEai(pProviderInfoBO, tEaiBO);
                    if (productBO != null) {
                        tProviderBO.setProduct(productBO.getProduct());
                        tProviderBO.setEai(tEaiBO.getEai());
                        foundProvider = true;
                        //寫回Cache
                        eaiGateway.saveEaiCache(tEaiBO, pProviderInfoBO.getTenantId());
                    }
                }
            }
        }

        if (!foundProvider) {
            //找敏態DB
            ProductEntity productEntity = queryProviderProductInDB(pProviderInfoBO);
            if (productEntity != null) {
                tProviderBO.setProduct(productEntity);
                productQueryGateWay.saveProductCache(productEntity);
                foundProvider = true;
            }
        }


        if (foundProvider) {
            //驗證服務
            ServiceEntity serviceEntity = findService(tProviderBO, pProviderInfoBO.getName());

            if (serviceEntity == null) {
                throw new ServiceNameNotFoundException();
            }
            tProviderBO.setService(serviceEntity);
        }

        if (!foundProvider) {
            throw new ServiceProviderNotFoundException();
        }

        return tProviderBO;
    }

    private EaiEntity queryProviderEaiInCache(String pTenantId) {
        return eaiGateway.getEaiInCacheByTenantId(pTenantId);
    }

    private ProductEntity queryProviderUnionProductInCache(ProviderInfoBO pProviderInfo, String pEaiUid) {
        return productQueryGateWay.findUnionProductInCache(pEaiUid, pProviderInfo.getUid(), pProviderInfo.getProd());
    }

    private EaiBO queryEaiBOWithUnionProductsInDB(String pTenantId) {
        return eaiGateway.queryEaiWithUnionProductByTenantId(pTenantId);
    }

    private ProductBO queryProviderUnionProductInEai(ProviderInfoBO pProviderInfo, EaiBO pEaiBO) {
        return pEaiBO.getProducts().stream()
                .filter(productBO ->
                        (pProviderInfo.getUid().equals(productBO.getProduct().getUid()) && pProviderInfo.getProd().equals(productBO.getProduct().getName())))
                .findFirst().orElse(null);
    }

    private ProductEntity queryProviderProductInCache(ProviderInfoBO pProviderInfo) throws BaseException {
        ProductEntity tProductBO = null;

        String tName = pProviderInfo.getProd();
        String tIp = pProviderInfo.getIp();
        String tApId = pProviderInfo.getId();
        String tUid = pProviderInfo.getUid();

        List<ProductEntity> tProductBOList = productQueryGateWay.findProductInCache(tName);

        if (CollectionUtils.isNotEmpty(tProductBOList)) {
            //去除clientOnly
            tProductBOList = tProductBOList.stream().filter(productBO -> !productBO.getClientOnly()).collect(Collectors.toList());

            List<ProductEntity> tSelectedProductBOList = filterProductBOList(tProductBOList, tIp, tApId, tUid);

            switch (tSelectedProductBOList.size()) {
                case 0:
                    break;
                case 1:
                    tProductBO = tSelectedProductBOList.get(0);
                    break;
                default:
                    //有多台時尋找預設主機
                    tProductBO = tProductBOList.stream().filter(ProductEntity::getIsDefault).findFirst().orElse(null);
                    if (tProductBO == null) {
                        throw new DuplicateProductException();
                    }
                    break;
            }
        }
        return tProductBO;
    }

    private ProductEntity queryProviderProductInDB(ProviderInfoBO pProviderInfo) throws BaseException {
        ProductEntity tProductBO = null;

        String tName = pProviderInfo.getProd();
        String tIp = pProviderInfo.getIp();
        String tApId = pProviderInfo.getId();
        String tUid = pProviderInfo.getUid();

        List<ProductEntity> tProductBOList = productQueryGateWay.queryNotClientOnlyProductByName(tName);

        if (CollectionUtils.isNotEmpty(tProductBOList)) {
            List<ProductEntity> tSelectedProductBOList = filterProductBOList(tProductBOList, tIp, tApId, tUid);

            switch (tSelectedProductBOList.size()) {
                case 0:
                    break;
                case 1:
                    tProductBO = tSelectedProductBOList.get(0);
                    break;
                default:
                    //有多台時尋找預設主機
                    tProductBO = tProductBOList.stream().filter(ProductEntity::getIsDefault).findFirst().orElse(null);
                    if (tProductBO == null) {
                        throw new DuplicateProductException();
                    }
                    break;
            }
        }
        return tProductBO;
    }

    private List<ProductEntity> filterProductBOList(List<ProductEntity> productBOList, String ip, String apId, String uid) {
        List<ProductEntity> tFilterProductBOList;

        if (StringUtils.isNotBlank(uid)) {
            //有uid優先找uid
            tFilterProductBOList = productBOList.stream().filter(productBO -> uid.equals(productBO.getUid())).collect(Collectors.toList());
        } else if (StringUtils.isNoneBlank(ip, apId)) {
            //沒有uid ip id都有給 怎精準比對
            tFilterProductBOList = productBOList.stream().filter(productBO -> ip.equals(productBO.getIpAddress()) && apId.equals(productBO.getApid())).collect(Collectors.toList());
        } else if (StringUtils.isNotBlank(ip)) {
            //找ip符合的
            tFilterProductBOList = productBOList.stream().filter(productBO -> ip.equals(productBO.getIpAddress())).collect(Collectors.toList());
        } else if (StringUtils.isNotBlank(apId)) {
            //找apId符合的
            tFilterProductBOList = productBOList.stream().filter(productBO -> apId.equals(productBO.getApid())).collect(Collectors.toList());
        } else {
            tFilterProductBOList = productBOList;
        }

        return tFilterProductBOList;
    }

    private ServiceEntity findService(ProviderBO provider, String srvName) {
        switch (provider.getProductType()) {
            case AGILE:
                return serviceQueryGateway.findService(provider.getProduct().getUid(), srvName);
            case STANDARD:
                return serviceQueryGateway.findUnionService(provider.getEai().getUid(), provider.getProduct().getUid(), srvName);
        }
        return null;
    }
}
