package com.digiwin.cross.domain.utils;

import com.digiwin.cross.domain.exception.InvokeRestFormatValidationException;
import com.digiwin.cross.domain.exception.MessageNotXMLorJSONValidationException;
import com.digiwin.cross.domain.exception.ValidatorException;
import com.fasterxml.jackson.databind.JsonNode;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.networknt.schema.JsonSchema;
import com.networknt.schema.JsonSchemaFactory;
import com.networknt.schema.SpecVersion;
import com.networknt.schema.ValidationMessage;
import lombok.extern.apachecommons.CommonsLog;
import org.apache.commons.lang3.StringUtils;

import java.io.IOException;
import java.io.InputStream;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;

/**
 * @description:
 * @author: liunansheng
 * @date: 2023/7/10 18:12
 */
@CommonsLog
public class JsonSchemaUtil {

    private static Map<String, JsonSchema> schemaMap = new ConcurrentHashMap<>();

    private static JsonSchema getSchema(String pSchemaFileName) {
        Function<String, JsonSchema> createSchema = key -> {
            InputStream jsonFileStream = null;
            try {
                jsonFileStream = ClassLoader.getSystemResourceAsStream(key);
                return JsonSchemaFactory.getInstance(SpecVersion.VersionFlag.V4).getSchema(JsonUtil.readJsonNode(jsonFileStream));
            } finally {
                try {
                    if (jsonFileStream != null) {
                        jsonFileStream.close();
                    }
                } catch (IOException e) {
                    log.error("关闭文件" + pSchemaFileName + "IO异常", e);
                }
            }
        };
        return schemaMap.computeIfAbsent(pSchemaFileName, createSchema);
    }

    public static void validate(String schemaFileUrl, String pJsonStr, String pTargetName) throws ValidatorException {
        if (StringUtils.isBlank(pJsonStr)) {
            String tMessage = String.format("'%s' content is empty", pTargetName);
            throw new InvokeRestFormatValidationException(tMessage);
        }
        try {
            JsonSchema schema = getSchema(schemaFileUrl);
            JsonNode node =JsonUtil.readJsonNode(pJsonStr);
            Set<ValidationMessage> tValidationMessages = schema.validate(node);
            if (tValidationMessages.size() > 0) {
                JsonObject tErrorMessage = new JsonObject();
                JsonArray tDetialArray = new JsonArray();
                tErrorMessage.add(pTargetName, tDetialArray);
                for (ValidationMessage tValidationMessage : tValidationMessages) {
                    tDetialArray.add(tValidationMessage.getMessage());
                }
                throw new InvokeRestFormatValidationException(tErrorMessage.toString());
            }
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            String tMessage = String.format("'%s' cant't parse: %s", pTargetName, e.getMessage());
            if (e instanceof InvokeRestFormatValidationException) {
                throw e;
            } else {
                throw new MessageNotXMLorJSONValidationException(tMessage);
            }
        }
    }
}
