package com.digiwin.cross.domain.utils;

import com.auth0.jwt.JWT;
import com.auth0.jwt.interfaces.DecodedJWT;
import com.digiwin.cross.domain.bo.LogBO;
import com.digiwin.cross.domain.bo.LogContent;
import com.digiwin.cross.domain.common.HeaderNamesConstant;
import com.digiwin.cross.domain.gateway.ILogGateway;
import com.digiwin.cross.domain.parameter.ApplicationSystemParameter;
import lombok.extern.apachecommons.CommonsLog;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.ThreadContext;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.zip.GZIPOutputStream;

/**
 * @description:
 * @author: clay
 * @date: 2023/6/30
 */
@CommonsLog
public class LogUtil {

    protected static Map<String, LogContent> logContentMap = Collections.synchronizedMap(new HashMap<>());
    //修正callback因不同執行序， finalizeLog後會導致log寫入異常，將全異部的logCache分開
    protected static Map<String, LogContent> callbackLogContentMap = Collections.synchronizedMap(new HashMap<>());

    protected static ExecutorService executor;

    private static ILogGateway logGateway;

    public static void setDependencies (ExecutorService executorService, ILogGateway logGateway) {
        LogUtil.executor = executorService;
        LogUtil.logGateway = logGateway;
    }

    protected static void saveLog (LogBO logBO) {
        saveLog(logBO, false);
    }

    protected static void saveLog (LogBO logBO, boolean isCallback) {
        LogContent tLogContent = null;
        if (isCallback) {
            tLogContent = callbackLogContentMap.get(logBO.getReqId());
        } else {
            tLogContent = logContentMap.get(logBO.getReqId());
        }
        tLogContent.setLogDisplayPriority(tLogContent.getLogDisplayPriority() + 1);
        logBO.setLogDisplayPriority(tLogContent.getLogDisplayPriority());
        log.info("init ptxid");
        logBO.setPinpointId(ThreadContext.get("PtxId")); // 在日志对象中新增pinpointId字段
        executor.execute(() -> logGateway.writeLog(logBO));
    }

    public static LogContent initLog (String reqId, int... priority) {
        return initLog(reqId, false, priority);
    }

    public static LogContent initLog (String reqId, boolean isCallback, int... priority) {
        LogContent tLogContent = new LogContent();
        if (priority.length == 0) {
            tLogContent.setLogDisplayPriority(0);
        } else {
            tLogContent.setLogDisplayPriority(priority[0]);
        }
        if (isCallback) {
            callbackLogContentMap.put(reqId, tLogContent);
        } else {
            logContentMap.put(reqId, tLogContent);
        }
        tLogContent.setReqId(reqId);
        return tLogContent;
    }

    protected static LogContent getLogContent (String reqId) {
        return getLogContent(reqId, false);
    }

    protected static LogContent getLogContent (String reqId, boolean isCallback) {
        if (isCallback) {
            return callbackLogContentMap.get(reqId);
        } else {
            return logContentMap.get(reqId);
        }
    }

    protected static void finalizeLog (String reqId) {
        finalizeLog(reqId, false);
    }

    protected static void finalizeLog (String reqId, boolean isCallback) {
        if (isCallback) {
            callbackLogContentMap.remove(reqId);
        } else {
            logContentMap.remove(reqId);
        }
    }

    public static byte[] convertRestLogMessage (Map<String, String> headerMap, String body) {
        return convertRestLogMessage(null, headerMap, body);
    }

    public static byte[] convertRestLogMessage (String httpLine, Map<String, String> headerMap, String body) {
        return convertRestLogMessage(httpLine, headerMap, body, null);
    }

    ;

    public static byte[] convertRestLogMessage (String httpLine, Map<String, String> headerMap, String body, String ptxId) {
        StringBuffer tStrBuffer = new StringBuffer();
        String pinpointId = ThreadContext.get("PtxId");
        if (ptxId != null) {
            pinpointId = ptxId;
        }
        if (pinpointId != null) {
            tStrBuffer.append("pinpointId").append(": ").append(pinpointId).append(System.lineSeparator());
        }
        //URL
        if (httpLine != null) {
            tStrBuffer.append(httpLine).append(System.lineSeparator());
        }

        //header
        tStrBuffer.append(System.lineSeparator());
        headerMap.forEach(
                (headerName, headerValue) ->
                        tStrBuffer.append(headerName).append(": ").append(headerValue).append(System.lineSeparator())
        );

        //body
        if (StringUtils.isNotBlank(body)) {
            tStrBuffer.append(System.lineSeparator());
            tStrBuffer.append(body);
        }

        return compress(tStrBuffer.toString());
    }

    protected static String getAppId (Map<String, String> headers) {
        if (headers.containsKey(HeaderNamesConstant.APP_AUTH)) {
            try {
                String tAppToken = headers.get(HeaderNamesConstant.APP_AUTH);
                if (StringUtils.isBlank(tAppToken)) {
                    return null;
                }
                DecodedJWT jwt = JWT.decode(tAppToken);
                return jwt.getClaim("id").asString();
            } catch (Exception e) {
                log.error(e.getMessage(), e);
            }
        }
        return null;
    }

    protected static void alarmBigRes(LogBO logBO) {
        if (logBO.getMessageSize() >= ApplicationSystemParameter.BIG_RES_SIZE * 1024) {
            logGateway.alarmBigRes(logBO);
        }
    }

    private static byte[] compress (String message) {
        if (StringUtils.isNotBlank(message)) {
            ByteArrayOutputStream tOutput = new ByteArrayOutputStream();

            try (GZIPOutputStream tZip = new GZIPOutputStream(tOutput)) {
                tZip.write(message.getBytes(StandardCharsets.UTF_8));
            } catch (IOException e) {
                log.error(e.getMessage(), e);
            }
            return tOutput.toByteArray();
        } else {
            return null;
        }

    }

}