package com.digiwin.cross.domain.service.impl;

import com.digiwin.cross.domain.bo.CallResponseBO;
import com.digiwin.cross.domain.bo.FullAsyncRequestBO;
import com.digiwin.cross.domain.bo.RequestBO;
import com.digiwin.cross.domain.bo.ResponseBO;
import com.digiwin.cross.domain.bo.ServiceResponseBO;
import com.digiwin.cross.domain.bo.entity.EaiEntity;
import com.digiwin.cross.domain.common.HeaderNamesConstant;
import com.digiwin.cross.domain.common.OperationConstant;
import com.digiwin.cross.domain.common.ProductType;
import com.digiwin.cross.domain.exception.BaseException;
import com.digiwin.cross.domain.gateway.IInvokeGateway;
import com.digiwin.cross.domain.parameter.ApplicationSystemParameter;
import com.digiwin.cross.domain.service.ICallbackService;
import com.digiwin.cross.domain.state.StateEnum;
import com.digiwin.cross.domain.state.StateMessageUtil;
import com.digiwin.cross.domain.utils.CallBackLogUtil;
import com.digiwin.cross.domain.utils.FilterHeaderUtil;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.Map;

import static com.digiwin.cross.domain.state.StateEnum.FASYNC_CALLBACK_RESULT;
import static com.digiwin.cross.domain.state.StateEnum.UNION_FASYNC_CALLBACK_RESULT;

/**
 * @description:
 * @author: liunansheng
 * @date: 2023/7/12 15:22
 */
@Service
public class CallBackService implements ICallbackService {


    private final IInvokeGateway invokeGateway;

    public CallBackService(IInvokeGateway invokeGateway) {
        this.invokeGateway = invokeGateway;
    }

    @Override
    public CallResponseBO call(FullAsyncRequestBO context, ServiceResponseBO serviceResponseBO) throws BaseException {
        RequestBO requestBO = buildInvokerRequestBO(serviceResponseBO, context);
        CallBackLogUtil.callBackRequester(context, requestBO);
        ResponseBO response;
        StateEnum stateEnum;
        if (context.getRequester().isAgile()) {
            response = invokeGateway.callProduct(requestBO);
            stateEnum = context.getProvider().isAgile() ? FASYNC_CALLBACK_RESULT : UNION_FASYNC_CALLBACK_RESULT;
        } else {
            response = invokeGateway.callEai(requestBO);
            if (response.isEaiSuccess()) {
                stateEnum = context.getProvider().isAgile() ? FASYNC_CALLBACK_RESULT : UNION_FASYNC_CALLBACK_RESULT;
            } else {
                stateEnum = StateEnum.STATEENUM_CODE_MAP.get(response.getHeaders().get(HeaderNamesConstant.DIGI_CODE));
                stateEnum = null == stateEnum ? StateEnum.UNION_REMOTE_RETURNED_SRV_FASYNC_FAILED : stateEnum;
            }
        }
        CallBackLogUtil.afterCallBackRequester(context, response);
        return new CallResponseBO(response, stateEnum, context.getHostInfo().getLang());
    }

    private RequestBO buildInvokerRequestBO(ServiceResponseBO serviceResponseBO, FullAsyncRequestBO context) {
        RequestBO tRequestBO = new RequestBO();
        tRequestBO.setOperation(OperationConstant.CALLBACK);
        tRequestBO.setReqId(serviceResponseBO.getReqId());
        Map<String, String> tHeaders = new HashMap<>();
        if (MapUtils.isNotEmpty(context.getCallbackHeaders())) {
            //先放callbackHeaders
            tHeaders.putAll(context.getCallbackHeaders());
        }
        tHeaders.putAll(serviceResponseBO.getHeaders());
        tHeaders.put(HeaderNamesConstant.DIGI_REQID, serviceResponseBO.getReqId());
        ProductType productType = context.getRequester().isAgile() ? ProductType.AGILE : ProductType.STANDARD;
        FilterHeaderUtil.filterBlockHeaders(tHeaders, productType);
        tRequestBO.setHeaders(tHeaders);
        tRequestBO.setBody(serviceResponseBO.getBody());
        StateEnum state;
        if (context.getRequester().isAgile()) {
            state = StateEnum.FASYNC_CALLBACK;
            String targetRestUrl = context.getRequester().getProduct().getRestUrl();
            targetRestUrl = concatCallBackUrl(targetRestUrl);
            if (StringUtils.isNotBlank(context.getDispatchId())) {
                tHeaders.put(HeaderNamesConstant.DISPATCHID, context.getDispatchId());
                tHeaders.put(HeaderNamesConstant.CALLBACK_URL, targetRestUrl);
                String dispatchUrl = concatCallBackUrl(ApplicationSystemParameter.ESPDP_URL);
                tRequestBO.setTargetUrl(dispatchUrl);
            } else {
                tRequestBO.setTargetUrl(targetRestUrl);
            }
        } else {
            state = StateEnum.UNION_FASYNC_CALLBACK;
            EaiEntity tEaiEntity = context.getRequester().getEai();
            tRequestBO.setTargetLongConnId(tEaiEntity.getLongConnId());
            tRequestBO.setTargetGatewayId(tEaiEntity.getGatewayId());
        }
        tHeaders.put(HeaderNamesConstant.DIGI_CODE, state.getCode());
        tHeaders.put(HeaderNamesConstant.DIGI_MESSAGE, StateMessageUtil.getBase64StateCodeMessage(state, context.getHostInfo().getLang()));
        return tRequestBO;
    }

    private static String concatCallBackUrl(String url) {
        return url + (url.endsWith("/") ? "callback" : "/callback");
    }
}
