package com.digiwin.cross.domain.service.impl;

import com.digiwin.cross.domain.bo.entity.ProductEntity;
import com.digiwin.cross.domain.gateway.IProductEditGateway;
import com.digiwin.cross.domain.gateway.IProductQueryGateWay;
import com.digiwin.cross.domain.parameter.ApplicationSystemParameter;
import com.digiwin.cross.domain.service.IProductService;
import lombok.extern.apachecommons.CommonsLog;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.List;

/**
 * @description:
 * @author: liunansheng
 * @date: 2023/5/26 15:37
 */
@CommonsLog
@Service
public class ProductService implements IProductService {

    private final IProductQueryGateWay productQueryGateWay;

    private final IProductEditGateway productEditGateway;

    public ProductService(IProductQueryGateWay productQueryGateWay, IProductEditGateway productEditGateway) {
        this.productQueryGateWay = productQueryGateWay;
        this.productEditGateway = productEditGateway;
    }

    @Override
    public boolean validExistsProduct(String uid, String name, String ip, String apId) {
        return findExistsProduct(uid, name, ip, apId) != null;
    }

    @Override
    public boolean validExistsOtherProduct(String name, String ip, String apId, Long currentProdId) {
        List<ProductEntity> existsProducts = productQueryGateWay.queryProductByApIdAndIpAndName(apId, ip, name, false);
        if (CollectionUtils.isEmpty(existsProducts)) {
            return false;
        }
        for (ProductEntity one : existsProducts) {
            if (!one.getId().equals(currentProdId)) {
                return true;
            }
        }
        return false;
    }

    @Override
    public boolean validExistsUnionProduct(String uid, String name, String ip, String apId) {
        List<ProductEntity> existsProducts = null;
        if (StringUtils.isNotBlank(uid)) {
            existsProducts = productQueryGateWay.queryUnionProductByUid(uid);
        }
        if (StringUtils.isBlank(uid) || CollectionUtils.isEmpty(existsProducts)) {
            existsProducts = productQueryGateWay.queryUnionProductByApIdAndIpAndName(apId, ip, name);
        }
        return CollectionUtils.isNotEmpty(existsProducts);
    }

    @Override
    public void saveProductBO(ProductEntity productBO) {
        if (BooleanUtils.isTrue(productBO.getClientOnly())) {
            productBO.setIsDefault(false);
        } else {
            List<ProductEntity> sampleNameProducts = productQueryGateWay.queryNotClientOnlyProductByName(productBO.getName());
            productBO.setIsDefault(CollectionUtils.isEmpty(sampleNameProducts));
        }
        if (StringUtils.isBlank(productBO.getTimezone())) {
            productBO.setTimezone(ApplicationSystemParameter.APREG_DEFAULT_TIMEZONE);
        }
        //生成默认uid
        if (StringUtils.isBlank(productBO.getUid())) {
            productBO.setUid(generateSuggestUID(productBO.getName()));
        }
        productEditGateway.saveProduct(productBO);
    }

    @Override
    public ProductEntity updateProductBO(ProductEntity productBO) {
        return productEditGateway.updateProduct(productBO);
    }

    @Override
    public ProductEntity findExistsProduct(String uid, String name, String ip, String apId) {
        List<ProductEntity> existsProducts = new ArrayList<>();
        if (StringUtils.isNotBlank(uid)) {
            existsProducts = productQueryGateWay.queryProductByUid(uid);
        }
        if (StringUtils.isBlank(uid) || CollectionUtils.isEmpty(existsProducts)) {
            existsProducts = productQueryGateWay.queryProductByApIdAndIpAndName(apId, ip, name, false);
        }
        return CollectionUtils.isNotEmpty(existsProducts) ? existsProducts.get(0) : null;
    }

    @Override
    public ProductEntity findProduct(String uid, String name, String ip, String apId) {
        List<ProductEntity> existsProducts = new ArrayList<>();
        if (StringUtils.isNotBlank(uid)) {
            existsProducts = productQueryGateWay.queryProductByUid(uid);
        } else  {
            existsProducts = productQueryGateWay.queryProductByApIdAndIpAndName(apId, ip, name, true);
        }
        return CollectionUtils.isNotEmpty(existsProducts) ? existsProducts.get(0) : null;
    }

    @Override
    public ProductEntity deleteProduct(String uid) {
        List<ProductEntity> productEntitys = productQueryGateWay.queryProductByUid(uid);
        if (CollectionUtils.isEmpty(productEntitys)) {
            log.warn("delete product, but product is not exists");
            return null;
        }
        productEditGateway.deleteProduct(productEntitys.get(0));
        return productEntitys.get(0);

    }


    private String generateSuggestUID(String pName) {
        String tSuggestUID = null;
        while (true) {
            SecureRandom secureRandom = new SecureRandom();
            int randomNumber = secureRandom.nextInt(1000);
            String randomThreeDigitNumber = String.format("%03d", randomNumber);
            tSuggestUID = pName + '_' + randomThreeDigitNumber;
            boolean tIsExist = productQueryGateWay.existsUid(tSuggestUID);
            if (tIsExist) {
                continue;
            } else {
                break;
            }
        }
        return tSuggestUID;
    }
}
