package com.digiwin.cross.infrastructure.cache.service;

import com.digiwin.cross.domain.parameter.ApplicationSystemParameter;
import com.digiwin.cross.domain.parameter.SaasParameter;
import com.digiwin.cross.domain.utils.SpringContextUtil;
import com.digiwin.cross.infrastructure.cache.CacheKeyEnum;
import lombok.extern.apachecommons.CommonsLog;
import org.apache.commons.lang3.RandomUtils;
import org.apache.commons.lang3.StringUtils;
import org.redisson.Redisson;
import org.redisson.api.LocalCachedMapOptions;
import org.redisson.api.RBucket;
import org.redisson.api.RKeys;
import org.redisson.api.RLocalCachedMap;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.redisson.config.Config;
import org.redisson.config.ConfigSupport;
import org.redisson.config.SingleServerConfig;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.io.Serializable;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * @description:
 * @author: clay
 * @date: 2023/6/7
 */
@Service
@CommonsLog
public class EspRedisService {


    {
        System.out.println("=========RedisService========");
    }

    private RedissonClient client;

    private boolean isEnable = false;

    private static Map<String, RLocalCachedMap> RLOCALCACHE_CONTEXT = new ConcurrentHashMap<>();

    public static final Object lock = new Object();
    @Autowired
    public  ApplicationContext context;

    //这个东西执行的时机太早了，有点反常
    @PostConstruct
    public void init () {
        if(null==SpringContextUtil.context){
            SpringContextUtil.context = context;
        }
        ApplicationSystemParameter.load();
        SaasParameter.load();
        if (ApplicationSystemParameter.REDIS_ENABLE) {
            Config tConfig = new Config();
            tConfig.setNettyThreads(128);
            SingleServerConfig tServerConfig = tConfig.useSingleServer();
            String tAddress = String.format("redis://%s:%s", ApplicationSystemParameter.REDIS_HOST, ApplicationSystemParameter.REDIS_PORT);
            tServerConfig.setAddress(tAddress);
            if(null!=ApplicationSystemParameter.REDIS_POOL_SIZE){
                tServerConfig.setConnectionPoolSize(ApplicationSystemParameter.REDIS_POOL_SIZE);
            }
            if (StringUtils.isNotEmpty(ApplicationSystemParameter.REDIS_PASSWORD)) {
                tServerConfig.setPassword(ApplicationSystemParameter.REDIS_PASSWORD);
            }
            if(null!=ApplicationSystemParameter.REDIS_TIMEOUT){
                tServerConfig.setTimeout(ApplicationSystemParameter.REDIS_TIMEOUT);
            }

            tServerConfig.setRetryAttempts(0);
            tServerConfig.setRetryInterval(500);
            tServerConfig.setDatabase(ApplicationSystemParameter.REDIS_DATABASE);
            try {
                redissonCreate(tConfig);
            } catch (Exception e) {
                log.error(e.getMessage(), e);
            }
            if (!isEnable) {
                ScheduledExecutorService executor = Executors.newScheduledThreadPool(1);
                executor.scheduleAtFixedRate(() -> {
                    try {
                        redissonCreate(tConfig);
                        executor.shutdown();
                    } catch (Exception e) {
                        log.error(e.getMessage(), e);
                    }
                }, 0, 10, TimeUnit.SECONDS);
            }
        }
    }

    private void redissonCreate (Config pConfig) throws Exception {
        client = Redisson.create(pConfig);
        log.info("Redis connect success");
        setEnable(true);
        String jsonFormat = new ConfigSupport().toJSON(pConfig);
        log.info("Redisson config in jsonFormat: " + jsonFormat);
    }

    public boolean isEnable () {
        return isEnable;
    }

    public void setEnable (boolean enable) {
        isEnable = enable;
    }

    public <T extends Serializable> T getFromL2Cache (CacheKeyEnum pCacheKeyEnum, Serializable pKey) {
        try {
            RLocalCachedMap<String, T> tRMap = getRLocalCachedMap(pCacheKeyEnum);
            return tRMap.get(pKey.toString());
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            return null;
        }
    }

    public <T extends Serializable> void putToL2Cache (CacheKeyEnum pCacheKeyEnum, Serializable pKey, T val) {
        try {
            RLocalCachedMap<String, T> tRMap = getRLocalCachedMap(pCacheKeyEnum);
            tRMap.put(pKey.toString(), val);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
    }

    public <T extends Serializable> void put (CacheKeyEnum pCacheKeyEnum, Serializable pKey, T pValue) {
        try {
            RBucket<T> bucket = client.getBucket(buildKey(pCacheKeyEnum, pKey));
            Long timeToLiveSeconds = getTimeToLiveSeconds(pCacheKeyEnum);
            if (null == timeToLiveSeconds) {
                bucket.set(pValue);
            } else {
                bucket.set(pValue, timeToLiveSeconds, TimeUnit.SECONDS);
            }
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
    }

    public <T extends Serializable> void put (CacheKeyEnum pCacheKeyEnum, Serializable pKey, List<T> pList) {
        try {
            RBucket<List<T>> bucket = client.getBucket(buildKey(pCacheKeyEnum, pKey));
            Long timeToLiveSeconds = getTimeToLiveSeconds(pCacheKeyEnum);
            if (null == timeToLiveSeconds) {
                bucket.set(pList);
            } else {
                bucket.set(pList, timeToLiveSeconds, TimeUnit.SECONDS);
            }
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
    }

    public <T extends Serializable> void put (CacheKeyEnum pCacheKeyEnum, Serializable pKey, Map<String, T> pMap) {
        try {
            RBucket<Map<String, T>> bucket = client.getBucket(buildKey(pCacheKeyEnum, pKey));
            Long timeToLiveSeconds = getTimeToLiveSeconds(pCacheKeyEnum);
            if (null == timeToLiveSeconds) {
                bucket.set(pMap);
            } else {
                bucket.set(pMap, timeToLiveSeconds, TimeUnit.SECONDS);
            }
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
    }

    public <T extends Serializable> T get (CacheKeyEnum pCacheKeyEnum, Serializable pKey) {
        try {
            RBucket<T> bucket = client.getBucket(buildKey(pCacheKeyEnum, pKey));
            return bucket.get();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        return null;
    }

    public <T extends Serializable> T remove (CacheKeyEnum pCacheKeyEnum, String pKey) {
        RBucket<T> bucket = client.getBucket(buildKey(pCacheKeyEnum, pKey));
        return bucket.getAndDelete();
    }

//    public void lock (String key) {
//        RLock tLock = client.getFairLock(key);
//        tLock.lock(TimeUnit.HOURS.toSeconds(2), TimeUnit.SECONDS);
//    }
//
//    public void lock (String key, int seconds) {
//        RLock tLock = client.getFairLock(key);
//        tLock.lock(seconds, TimeUnit.SECONDS);
//    }

    public boolean tryLock (String key) throws InterruptedException {
        RLock tLock = client.getFairLock(key);
        return tLock.tryLock(3, TimeUnit.HOURS.toSeconds(2), TimeUnit.SECONDS);
    }

    public boolean tryLock (String key, int waitLockSeconds) throws InterruptedException {
        RLock tLock = client.getFairLock(key);
        return tLock.tryLock(waitLockSeconds, TimeUnit.HOURS.toSeconds(2), TimeUnit.SECONDS);
    }

    public boolean tryLock (String key, int waitLockSeconds, int maxLockSecond) throws InterruptedException {
        RLock tLock = client.getFairLock(key);
        return tLock.tryLock(waitLockSeconds, maxLockSecond, TimeUnit.SECONDS);
    }

    public void unLock (String pLockKey) {
        RLock tLock = client.getFairLock(pLockKey);
        tLock.unlock();
    }

    public void clearByNamespace (CacheKeyEnum pCacheKeyEnum) {
        RKeys rKeys = client.getKeys();
        rKeys.deleteByPattern(pCacheKeyEnum.getCacheName() + ":*");
    }

    private <T> RLocalCachedMap<String, T> getRLocalCachedMap (CacheKeyEnum cacheKeyEnum) {
        String key = cacheKeyEnum.getCacheName();
        RLocalCachedMap<String, T> rLocalCachedMap = RLOCALCACHE_CONTEXT.get(key);
        if (null == rLocalCachedMap) {
            synchronized (lock) {
                rLocalCachedMap = RLOCALCACHE_CONTEXT.get(cacheKeyEnum.getCacheName());
                if (rLocalCachedMap == null) {
                    rLocalCachedMap = client.getLocalCachedMap(key, LocalCachedMapOptions.defaults());
                    RLOCALCACHE_CONTEXT.put(key, rLocalCachedMap);
                }
            }
        }
        return rLocalCachedMap;
    }

    public static String buildKey (CacheKeyEnum nameSpace, Serializable key) {
        key = key==null?"":key.toString();
        return nameSpace.getCacheName() + ":" + key.toString();
    }

    public RedissonClient getClient () {
        return client;
    }

    public static Long getTimeToLiveSeconds (CacheKeyEnum nameSpace) {
        if (null == nameSpace.getTimeToLiveSeconds()) {
            return null;
        }
        Long timeToLiveSeconds = nameSpace.getTimeToLiveSeconds();
        if (nameSpace.getMaxRandomLiveSeconds() != null) {
            timeToLiveSeconds += RandomUtils.nextLong(0, nameSpace.getMaxRandomLiveSeconds());
        }
        return timeToLiveSeconds;
    }
}
