package com.digiwin.cross.infrastructure.cache.service;

import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.digiwin.cross.infrastructure.cache.CacheKeyEnum;
import com.digiwin.cross.infrastructure.database.entity.EaiTenantMappingPO;
import com.digiwin.cross.infrastructure.database.entity.ProductPO;
import com.digiwin.cross.infrastructure.database.entity.UnionEAIPO;
import com.digiwin.cross.infrastructure.database.entity.UnionProductPO;
import com.digiwin.cross.infrastructure.database.mapper.EaiTenantMappingMapper;
import com.digiwin.cross.infrastructure.database.mapper.ProductMapper;
import com.digiwin.cross.infrastructure.database.mapper.UnionEaiMapper;
import com.digiwin.cross.infrastructure.database.mapper.UnionProductMapper;
import lombok.extern.apachecommons.CommonsLog;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

/**
 * @description:
 * @author: clay
 * @date: 2023/6/7
 */
@Component
@CommonsLog
public class ProductInfoCacheService {

    private final EspRedisService espRedisService;
    private final TwoLevelCacheService twoLevelCacheService;

    private final ProductMapper productMapper;
    private final EaiTenantMappingMapper eaiTenantMappingMapper;
    private final UnionEaiMapper unionEaiMapper;
    private final UnionProductMapper unionProductMapper;


    public ProductInfoCacheService(EspRedisService espRedisService, TwoLevelCacheService twoLevelCacheService, ProductMapper productMapper, EaiTenantMappingMapper eaiTenantMappingMapper, UnionEaiMapper unionEaiMapper, UnionProductMapper unionProductMapper) {
        this.espRedisService = espRedisService;
        this.twoLevelCacheService = twoLevelCacheService;
        this.productMapper = productMapper;
        this.eaiTenantMappingMapper = eaiTenantMappingMapper;
        this.unionEaiMapper = unionEaiMapper;
        this.unionProductMapper = unionProductMapper;
    }

//    /**
//     * 敏態
//     */
//    //key=prodName
//    private final Map<String, List<ProductPO>> productCacheMap = new HashMap<>();
//
//    /**
//     * 穩態
//     */
//    //key=tenantId value=eaiuid
//    private final Map<String, String> tenantEaiCacheMap = new HashMap<>();
//    //key tenantId
//    private final Map<String, UnionEAIPO> unionEaiCacheMap = new HashMap<>();
//    //key1 tenantId  key2 prodUid
//    private final Map<String, Map<String, UnionProductPO>> unionProductCacheMap = new HashMap<>();

    @PostConstruct
    public void init() {
        ScheduledExecutorService executor = Executors.newScheduledThreadPool(1);
        executor.scheduleAtFixedRate(() -> {
            try {
                loadCache();
                //載入完成
                executor.shutdown();
            } catch (Exception e) {
                log.error(e.getMessage(), e);
            }
        }, 0, 30, TimeUnit.SECONDS);
    }

    public void loadCache() throws Exception {
        if (!espRedisService.isEnable()) {
            return;
        }
        //清空redis緩存
        twoLevelCacheService.clearByNamespace(CacheKeyEnum.ESP_PRODUCT_INFO_NEW_V2);
        espRedisService.clearByNamespace(CacheKeyEnum.ESP_TENANT_EAI_MAPPING_NEW);
        espRedisService.clearByNamespace(CacheKeyEnum.ESP_UNION_EAI_INFO_NEW);
        espRedisService.clearByNamespace(CacheKeyEnum.ESP_UNION_PRODUCT_INFO_NEW);
        //讀取敏態產品
      //  List<ProductPO> productPOList = productMapper.selectList(Wrappers.<ProductPO>lambdaQuery().eq(ProductPO::getIsvalid, true));
        List<ProductPO> productPOList = productMapper.selectList2(true);
        if (CollectionUtils.isNotEmpty(productPOList)) {
            //敏態產品
            Map<String, List<ProductPO>> productCacheMap = new HashMap<>();
            productPOList.forEach(productPO -> {
                String tName = productPO.getName();
                List<ProductPO> tProdList = productCacheMap.computeIfAbsent(tName, k -> new ArrayList<>());
                tProdList.add(productPO);
            });
            productCacheMap.forEach((name, productList) -> twoLevelCacheService.putToL2Cache(CacheKeyEnum.ESP_PRODUCT_INFO_NEW_V2, name, productList));
        }

        //讀取EaiTenantMapping
      //  List<EaiTenantMappingPO> eaiTenantMappingPOList = eaiTenantMappingMapper.selectList(Wrappers.query());
        List<EaiTenantMappingPO> eaiTenantMappingPOList = eaiTenantMappingMapper.selectList2();
        if (CollectionUtils.isNotEmpty(eaiTenantMappingPOList)) {
            //EaiTenantMapping
            toTenantUidMap(eaiTenantMappingPOList).forEach((tenantId, eaiUid) -> espRedisService.put(CacheKeyEnum.ESP_TENANT_EAI_MAPPING_NEW, tenantId, eaiUid));
        }

        //讀取UnionEAI
        List<UnionEAIPO> unionEAIPOList = unionEaiMapper.selectList(Wrappers.<UnionEAIPO>lambdaQuery().eq(UnionEAIPO::getIsvalid, true));
        if (CollectionUtils.isNotEmpty(unionEAIPOList)) {
            unionEAIPOList.forEach(unionEAIPO -> espRedisService.put(CacheKeyEnum.ESP_UNION_EAI_INFO_NEW, unionEAIPO.getUid(), unionEAIPO));
            final Map<Long, String> tEaiUidMap = unionEAIPOList.stream().collect(Collectors.toMap(UnionEAIPO::getId, UnionEAIPO::getUid));

            //讀取UnionProduct
            List<UnionProductPO> unionProductPOList = unionProductMapper.selectList(Wrappers.<UnionProductPO>lambdaQuery().eq(UnionProductPO::getIsvalid, true));
            if (CollectionUtils.isNotEmpty(unionProductPOList)) {
                Map<String, Map<String, UnionProductPO>> unionProductCacheMap = new HashMap<>();
                unionProductPOList.forEach(unionProductPO -> {
                    String tEaiUid = tEaiUidMap.get(unionProductPO.getEaiId());
                    String tProdUid = unionProductPO.getUid();
                    Map<String, UnionProductPO> tUnionProductPOMap = unionProductCacheMap.computeIfAbsent(tEaiUid, k -> new HashMap<>());
                    tUnionProductPOMap.put(tProdUid, unionProductPO);
                });
                unionProductCacheMap.forEach((eaiUid, unionProductMap) -> espRedisService.put(CacheKeyEnum.ESP_UNION_PRODUCT_INFO_NEW, eaiUid, unionProductMap));
            }
        }
    }

    public List<ProductPO> getProductsByName(String name) {
        if (espRedisService.isEnable()) {
            return twoLevelCacheService.getFromL2Cache(CacheKeyEnum.ESP_PRODUCT_INFO_NEW_V2, name);
        }
        return null;
    }

    public ProductPO getProductByUid(String name, String uid) {
        List<ProductPO> tProductList = getProductsByName(name);

        AtomicReference<ProductPO> tResult = new AtomicReference<>();

        if (tProductList != null) {
            tProductList.forEach(productPO -> {
                if (uid.equals(productPO.getUid())) {
                    tResult.set(productPO);
                }
            });
        }

        return tResult.get();
    }

    public List<ProductPO> getProductByIpId(String name, String ip, String apId) {
        List<ProductPO> tProductList = getProductsByName(name);
        List<ProductPO> tFilterProductList = null;
        if (CollectionUtils.isNotEmpty(tProductList)) {
            tFilterProductList = tProductList.stream().filter(productPO -> {
                if (StringUtils.isNotBlank(ip)) {
                    if(!ip.equals(productPO.getIpAddress())) return false;
                }
                if(StringUtils.isNotBlank(apId)) {
                    if(!apId.equals(productPO.getApid())) return false;
                }
                return true;
            }).collect(Collectors.toList());
        }

        return tFilterProductList;
    }

    public String getEaiIdByTenantId(String tenantId) {
        String tEaiId = null;
        if (espRedisService.isEnable()) {
            tEaiId = espRedisService.get(CacheKeyEnum.ESP_TENANT_EAI_MAPPING_NEW, tenantId);
        }
        return tEaiId;
    }

    public UnionEAIPO getUnionEai(String eaiUid) {
        UnionEAIPO tEaiPO = null;
        if (espRedisService.isEnable()) {
            tEaiPO = espRedisService.get(CacheKeyEnum.ESP_UNION_EAI_INFO_NEW, eaiUid);
        }
        return tEaiPO;
    }

    public UnionProductPO getUnionProduct(String eaiUid, String prodUid, String prodName) {
        UnionProductPO tUnionProductPO = null;

        Map<String, UnionProductPO> tEaiProductPOMap = null;
        if (espRedisService.isEnable()) {
            tEaiProductPOMap = espRedisService.get(CacheKeyEnum.ESP_UNION_PRODUCT_INFO_NEW, eaiUid);
        }
        if (tEaiProductPOMap != null) {
            tUnionProductPO = tEaiProductPOMap.get(prodUid);
        }

        if (tUnionProductPO != null) {
            if (!tUnionProductPO.getName().equals(prodName)) {
                tUnionProductPO = null;
            }
        }

        return tUnionProductPO;
    }

    public void deleteEaiInfo(String eaiUid, List<String> tenants) {
        deleteEaiTenants(eaiUid, tenants);
        if (espRedisService.isEnable()) {
            //UnionEai
            espRedisService.remove(CacheKeyEnum.ESP_UNION_EAI_INFO_NEW, eaiUid);
            //穩態產品
            espRedisService.remove(CacheKeyEnum.ESP_UNION_PRODUCT_INFO_NEW, eaiUid);
        }
    }

    public void addEaiInfo(UnionEAIPO eai, List<String> tenants, List<UnionProductPO> products) {
        Map<String, UnionProductPO> productMap = new HashMap<>();
        products.forEach(productPO -> {
            productMap.putIfAbsent(productPO.getUid(), productPO);
        });
        if (espRedisService.isEnable()) {
            //EaiTenantMapping
            tenants.forEach(tenantId -> espRedisService.put(CacheKeyEnum.ESP_TENANT_EAI_MAPPING_NEW, tenantId, eai.getUid()));
            //UnionEai
            espRedisService.put(CacheKeyEnum.ESP_UNION_EAI_INFO_NEW, eai.getUid(), eai);
            //穩態產品
            espRedisService.put(CacheKeyEnum.ESP_UNION_PRODUCT_INFO_NEW, eai.getUid(), productMap);
        }
    }

    public void addEaiInfo(UnionEAIPO eai) {
        if (espRedisService.isEnable()) {
            espRedisService.put(CacheKeyEnum.ESP_UNION_EAI_INFO_NEW, eai.getUid(), eai);
        }
    }

    public void deleteEaiTenants(String eaiUid, List<String> tenants) {
        if (espRedisService.isEnable()) {
            tenants.forEach(tenantId -> {
                String val = espRedisService.remove(CacheKeyEnum.ESP_TENANT_EAI_MAPPING_NEW, tenantId);
                if (val != null && !val.equals(eaiUid)) {
                    //如果租户在另外一个eai上注册，不可以删除，理论上不会有这种情况出现
                    espRedisService.put(CacheKeyEnum.ESP_TENANT_EAI_MAPPING_NEW, tenantId, val);
                }
            });
        }
    }

    public void addProduct(ProductPO productPO) {
        if (espRedisService.isEnable()) {
            List<ProductPO> prods = twoLevelCacheService.getFromL2Cache(CacheKeyEnum.ESP_PRODUCT_INFO_NEW_V2, productPO.getName());
            if (null == prods) {
                prods = new ArrayList<>();
            } else {
                deleteProductInList(prods, productPO);
            }
            prods.add(productPO);
            twoLevelCacheService.putToL2Cache(CacheKeyEnum.ESP_PRODUCT_INFO_NEW_V2, productPO.getName(), prods);
        }
    }

    public void updateProduct(ProductPO productPO) {
        if (espRedisService.isEnable()) {
            List<ProductPO> prods = twoLevelCacheService.getFromL2Cache(CacheKeyEnum.ESP_PRODUCT_INFO_NEW_V2, productPO.getName());
            if (null == prods) {
                prods = new ArrayList<>();
            } else {
                deleteProductInList(prods, productPO);
            }
            prods.add(productPO);
            twoLevelCacheService.putToL2Cache(CacheKeyEnum.ESP_PRODUCT_INFO_NEW_V2, productPO.getName(), prods);
        }
    }

    public void deleteProduct(ProductPO productPO) {
        if (espRedisService.isEnable()) {
            List<ProductPO> tProdList = twoLevelCacheService.getFromL2Cache(CacheKeyEnum.ESP_PRODUCT_INFO_NEW_V2, productPO.getName());
            if (null == tProdList) {
                return;
            }
            int c = deleteProductInList(tProdList, productPO);
            if (c > 0) {
                twoLevelCacheService.putToL2Cache(CacheKeyEnum.ESP_PRODUCT_INFO_NEW_V2, productPO.getName(), tProdList);
            }
        }
    }

    private int deleteProductInList(List<ProductPO> products, ProductPO deletedProd) {
        if (CollectionUtils.isEmpty(products)) {
            return 0;
        }
        for (int i = 0; i < products.size(); i++) {
            if (deletedProd.getId().equals(products.get(i).getId())) {
                products.remove(i);
                return 1;
            }
        }
        return 0;
    }

    private Map<String, String> toTenantUidMap(List<EaiTenantMappingPO> tenantMappingPOList) {
        if (CollectionUtils.isEmpty(tenantMappingPOList)) {
            return new HashMap<>();
        }
        Map<String, String> map = new HashMap<>();
        Map<String, EaiTenantMappingPO> poMap = new HashMap<>();
        tenantMappingPOList.forEach(one -> {
            if (map.containsKey(one.getTenantId())) {
                EaiTenantMappingPO last = poMap.get(one.getTenantId());
                if (null!=last && null!=last.getBuildTime() && null!=one.getBuildTime() && last.getBuildTime().getTime() < one.getBuildTime().getTime()) {
                    map.put(one.getTenantId(), one.getEaiUid());
                    poMap.put(one.getTenantId(), one);
                }
            } else {
                map.put(one.getTenantId(), one.getEaiUid());
                poMap.put(one.getTenantId(), one);
            }
        });
        return map;
    }
}
