package com.digiwin.cross.infrastructure.cache.service;

import com.digiwin.cross.infrastructure.cache.CacheKeyEnum;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import lombok.extern.apachecommons.CommonsLog;
import org.apache.commons.lang3.StringUtils;
import org.redisson.api.RTopic;
import org.redisson.api.RedissonClient;
import org.springframework.stereotype.Service;

import java.io.Serializable;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.TimeUnit;

/**
 * @description:
 * @author: liunansheng
 * @date: 2024/5/23 17:31
 */
@CommonsLog
@Service
public class TwoLevelCacheService {

    private static final Object lock = new Object();

    private static final String CLEAR_ALL_MSG= "[clear_all]";

    private static final String CURRENT_CLIENT_ID = UUID.randomUUID().toString().replaceAll("-", "");
    private static final String CHANGE_MSG_SLIPTOR = "|||";

    private Map<String, Cache> CACHE_CONTEXT = new HashMap<>();
    private Map<String, RTopic> TOPIC_CONTEXT = new HashMap<>();

    private RedissonClient client;

    private EspRedisService espRedisService;

    public TwoLevelCacheService(EspRedisService espRedisService) {
        this.client = espRedisService.getClient();
        this.espRedisService = espRedisService;
    }

    public <T extends Serializable> void putToL2Cache(CacheKeyEnum pCacheKeyEnum, String pKey, List<T> pList) {
        try {
            Cache<String, List<T>> cache = getCache(pCacheKeyEnum);
            espRedisService.put(pCacheKeyEnum, pKey, pList);
            cache.put(pKey, pList);
            publishChange(pCacheKeyEnum, pKey);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
    }

    public <T extends Serializable> T getFromL2Cache(CacheKeyEnum pCacheKeyEnum, String pKey) {
        try {
            Cache<String, T> cache = getCache(pCacheKeyEnum);
            return cache.get(pKey, key -> {
                return espRedisService.get(pCacheKeyEnum, key);
            });
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        return null;
    }

    public void clearByNamespace(CacheKeyEnum pCacheKeyEnum) {
        getCache(pCacheKeyEnum).invalidateAll();
        espRedisService.clearByNamespace(pCacheKeyEnum);
        publishChange(pCacheKeyEnum, CLEAR_ALL_MSG);
    }

    public <T extends Serializable> T remove(CacheKeyEnum pCacheKeyEnum, String pKey) {
        getCache(pCacheKeyEnum).invalidate(pKey);
        T v = espRedisService.remove(pCacheKeyEnum, pKey);
        publishChange(pCacheKeyEnum, pKey);
        return v;
    }

    private void publishChange(CacheKeyEnum pCacheKeyEnum, String key) {
        try {
            TOPIC_CONTEXT.get(pCacheKeyEnum.getCacheName()).publish(CURRENT_CLIENT_ID + CHANGE_MSG_SLIPTOR + key);
        } catch (Exception e) {
            log.error("publish modify err", e);
        }
    }

    private <T> Cache<String, T> getCache(CacheKeyEnum cacheKeyEnum) {
        String namespace = cacheKeyEnum.getCacheName();
        Cache<String, T> cache = CACHE_CONTEXT.get(namespace);
        if (null == cache) {
            synchronized (lock) {
                cache = CACHE_CONTEXT.get(cacheKeyEnum.getCacheName());
                if (cache == null) {
                    cache = Caffeine.newBuilder()
                            .maximumSize(1024)              //最大数量
                            .expireAfterWrite(30, TimeUnit.MINUTES) //过期时间
                            .build();
                    CACHE_CONTEXT.put(namespace, cache);
                    RTopic topic = client.getTopic(cacheKeyEnum.getCacheName() + ":change");
                    TOPIC_CONTEXT.put(namespace, topic);
                    //监听缓存变化事件
                    topic.addListener(String.class, (channel, msg) -> {
                        String[] msgInfo = StringUtils.split(msg, CHANGE_MSG_SLIPTOR);
                        if (!CURRENT_CLIENT_ID.equals(msgInfo[0])) {
                            //消息发送端是当前机器，则不做处理
                            if (CLEAR_ALL_MSG.equals(msgInfo[1])) {
                                CACHE_CONTEXT.get(namespace).invalidateAll();
                            } else {
                                CACHE_CONTEXT.get(namespace).invalidate(msgInfo[1]);
                            }
                        }
                    });
                }
            }
        }
        return cache;
    }
}
