package com.digiwin.cross.infrastructure.gatewayimpl;

import com.digiwin.cross.domain.bo.RequestBO;
import com.digiwin.cross.domain.bo.ResponseBO;
import com.digiwin.cross.domain.exception.BaseException;
import com.digiwin.cross.domain.gateway.IInvokeGateway;
import com.digiwin.cross.domain.parameter.ApplicationSystemParameter;
import com.digiwin.cross.domain.utils.Base64Util;
import com.digiwin.cross.domain.utils.InvokeLogUtil;
import com.digiwin.cross.domain.utils.JsonUtil;
import com.digiwin.cross.infrastructure.cache.CacheKeyEnum;
import com.digiwin.cross.infrastructure.cache.service.EspRedisService;
import com.digiwin.cross.infrastructure.convertor.MqConvertor;
import com.digiwin.cross.infrastructure.exception.GatewayConnectionFailedException;
import com.digiwin.cross.infrastructure.exception.RemoteEAIExecutionTimeoutException;
import com.digiwin.cross.infrastructure.exception.RemoteEAIFusingException;
import com.digiwin.cross.infrastructure.exception.ServiceConnectionFailedException;
import com.digiwin.cross.infrastructure.exception.ServiceExecutionTimeoutException;
import com.digiwin.cross.infrastructure.exception.ServiceThrowException;
import com.digiwin.cross.infrastructure.http.InvokeRetryEventHandler;
import com.digiwin.cross.infrastructure.mq.po.EaiResponse;
import com.digiwin.cross.infrastructure.mq.po.EaiResult;
import com.digiwin.cross.infrastructure.mq.po.MqMessage;
import com.digiwin.cross.infrastructure.mq.po.MqResult;
import com.digiwin.cross.infrastructure.mq.po.RestResponse;
import com.digiwin.http.client.DWHttpClient;
import com.digiwin.http.client.DWRequestOption;
import com.digiwin.queue.saas.SaasClient;
import lombok.Setter;
import lombok.extern.apachecommons.CommonsLog;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.Header;
import org.apache.http.HttpResponse;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.protocol.BasicHttpContext;
import org.apache.http.util.EntityUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import java.io.IOException;
import java.net.SocketTimeoutException;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * @description:
 * @author: clay
 * @date: 2023/6/20
 */
@Service
@CommonsLog
public class InvokeGateway implements IInvokeGateway {

    private final SaasClient mqClient;
    private final DWHttpClient dwHttpClient;
    private final EspRedisService espRedisService;
    @Value("${dwHttpRequestConfigConnectTimeout}")
    @Setter
    private Integer connectTimeout;
    @Value("${dwHttpRequestConfigConnectionRequestTimeout}")
    @Setter
    private Integer connectRequestTimeout;

    public InvokeGateway(SaasClient mqClient, DWHttpClient dwHttpClient, EspRedisService espRedisService) {
        this.mqClient = mqClient;
        this.dwHttpClient = dwHttpClient;
        this.espRedisService = espRedisService;
    }

    @Override
    public ResponseBO callProduct(RequestBO request) throws BaseException {
        StringBuilder tUrl = new StringBuilder(request.getTargetUrl());
        if (StringUtils.isNotBlank(request.getPathParam())) {
            tUrl.append(request.getTargetUrl().endsWith("/") ? "" : "/").append(request.getPathParam());
        }
        if (StringUtils.isNotBlank(request.getQueryParams())) {
            tUrl.append("?").append(request.getQueryParams());
        }

        HttpPost tRequest = new HttpPost(tUrl.toString());

        request.getHeaders().forEach(tRequest::addHeader);
        tRequest.setEntity(new StringEntity(request.getBody(), ContentType.APPLICATION_JSON));

        //逾時時間
        Long tTimeout = TimeUnit.SECONDS.toMillis(ApplicationSystemParameter.SERVICE_ENGINE_SRV_TIMEOUT);
        RequestConfig tRequestConfig = RequestConfig.custom()
                .setConnectTimeout(connectTimeout)
                .setConnectionRequestTimeout(connectRequestTimeout)
                .setSocketTimeout(tTimeout.intValue())
                .build();
        tRequest.setConfig(tRequestConfig);


        BasicHttpContext httpContext = new BasicHttpContext();
        httpContext.setAttribute(InvokeRetryEventHandler.HTTP_CONTEXT_ATTRIBUTE_NAME_START_TIME,  System.currentTimeMillis());
        httpContext.setAttribute(InvokeRetryEventHandler.HTTP_CONTEXT_ATTRIBUTE_NAME_SOCKET_TIMEOUT, TimeUnit.SECONDS.toMillis(tTimeout));
        httpContext.setAttribute(InvokeRetryEventHandler.HTTP_CONTEXT_ATTRIBUTE_NAME_REQUEST, request);
        try {
            HttpResponse tResponse = dwHttpClient.execute(tRequest,null, new DWRequestOption(false), httpContext);
            int tStatusCode = tResponse.getStatusLine().getStatusCode();
            String tMessage = EntityUtils.toString(tResponse.getEntity());
            String tErrorMessage = null;
            if (tStatusCode != HttpStatus.SC_OK) {
                Map<String, String> tMap = new HashMap<>();
                tMap.put("rest_status_code", String.valueOf(tStatusCode));
                tMap.put("message", tMessage);
                tErrorMessage = JsonUtil.toJson(tMap);
            }
            switch (tStatusCode) {
                case HttpStatus.SC_OK:
                    ResponseBO tResponseBO = new ResponseBO();
                    Map<String, String> tHeaders = new HashMap<>();
                    for (Header tHeader : tResponse.getAllHeaders()) {
                        tHeaders.put(tHeader.getName(), tHeader.getValue());
                    }
                    tResponseBO.setHeaders(tHeaders);
                    tResponseBO.setBody(tMessage);
                    return tResponseBO;
                case HttpStatus.SC_NOT_FOUND:
                case HttpStatus.SC_BAD_GATEWAY:
                case HttpStatus.SC_SERVICE_UNAVAILABLE:
                    throw new ServiceConnectionFailedException(tErrorMessage, tStatusCode);
                default:
                    throw new ServiceThrowException(tErrorMessage, tStatusCode);
            }
        } catch (SocketTimeoutException e) {
            throw new ServiceExecutionTimeoutException(e);
        } catch (IOException e) {
            throw new ServiceConnectionFailedException(e);
        }

    }

    @Override
    public ResponseBO callEai(RequestBO request) throws BaseException {
        String fusingKey = request.getTargetLongConnId() + ":" + request.getTargetGatewayId();
        if (espRedisService.get(CacheKeyEnum.ESP_EAI_FUSING, fusingKey) != null) {
            log.info(String.format("地中台：%s熔断生效", fusingKey));
            throw new RemoteEAIFusingException();
        }
        MqMessage tMqMessage = MqConvertor.convertToMqMessage(request);

        Long tTimeout = ApplicationSystemParameter.UNION_SERVICE_ENGINE_SRV_TIMEOUT;
        Long tRetryTimes = ApplicationSystemParameter.UNION_SRV_RETRY_TIMES;
        Long tRetryInterval = ApplicationSystemParameter.UNION_SRV_RETRY_INTERVAL;

        while (true) {

            try {
                String sdkResult = mqClient.onSyncSend(request.getTargetLongConnId(), request.getTargetGatewayId(), JsonUtil.toJson(tMqMessage), tTimeout.intValue(), TimeUnit.SECONDS, true);
                log.info("esp invoker接到SDK回傳");
                MqResult tMqResult = JsonUtil.toJavaObj(sdkResult, MqResult.class);
                if (tMqResult.isTimeout()) {
                    throw new RemoteEAIExecutionTimeoutException();
                }
                EaiResult tEaiResult = tMqResult.getDetail().get(0);
                if (tEaiResult.getResult()) {
                    EaiResponse tEaiResponse = JsonUtil.toJavaObj(tEaiResult.getResponse(), EaiResponse.class);
                    String tRestMessage = Base64Util.decodeString(tEaiResponse.getMessage(), StandardCharsets.UTF_8.name());
                    RestResponse tRestResponse = JsonUtil.toJavaObj(tRestMessage, RestResponse.class);
                    ResponseBO tResponseBO = new ResponseBO();
                    tResponseBO.setHeaders(tRestResponse.getHeaders());
                    tResponseBO.setBody(tRestResponse.getMessagebody());
                    return tResponseBO;
                } else {
                    throw new ServiceThrowException(tEaiResult.getResponse());
                }
            } catch (BaseException e) {
                throw e;
            } catch (Exception e) {
                log.error(e.getMessage(), e);

                tRetryTimes--;
                if (tRetryTimes < 0) {
                    //重試次數用完
                    throw new GatewayConnectionFailedException(e.getMessage());
                } else {
                    sleep(tRetryInterval);
                    InvokeLogUtil.resendMq(request);
                }
            }

        }
    }

    private void sleep(long t) {
        try {
            Thread.sleep(t);
        } catch (InterruptedException e) {
            log.error("Thread sleep throw exception - " + e.getMessage());
            Thread.currentThread().interrupt();
        }
    }
}
