package com.digiwin.cross.infrastructure.gatewayimpl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.digiwin.cross.domain.bo.entity.ProductEntity;
import com.digiwin.cross.domain.gateway.IProductQueryGateWay;
import com.digiwin.cross.infrastructure.cache.service.ProductInfoCacheService;
import com.digiwin.cross.infrastructure.convertor.ProductConvertor;
import com.digiwin.cross.infrastructure.database.entity.ProductPO;
import com.digiwin.cross.infrastructure.database.entity.UnionProductPO;
import com.digiwin.cross.infrastructure.database.mapper.ProductMapper;
import com.digiwin.cross.infrastructure.database.mapper.UnionProductMapper;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

/**
 * @description:
 * @author: liunansheng
 * @date: 2023/5/26 15:40
 */
@Component
public class ProductQueryGateway implements IProductQueryGateWay {

    private final ProductMapper productMapper;

    private final UnionProductMapper unionProductMapper;

    private final ProductInfoCacheService productInfoCacheService;

    public ProductQueryGateway(ProductMapper productMapper, UnionProductMapper unionProductMapper, ProductInfoCacheService productInfoCacheService) {
        this.productMapper = productMapper;
        this.unionProductMapper = unionProductMapper;
        this.productInfoCacheService = productInfoCacheService;
    }

    @Override
    public List<ProductEntity> queryProductByUid(String uid) {
        List<ProductPO> productPOList = productMapper.selectList(Wrappers.<ProductPO>lambdaQuery()
                .eq(ProductPO::getUid, uid).eq(ProductPO::getIsvalid, true));
        if (CollectionUtils.isEmpty(productPOList)) {
            return null;
        }
        return productPOList.stream().map(po -> ProductConvertor.convertToBOEntity(po)).collect(Collectors.toList());
    }

    @Override
    public ProductEntity queryProductById(Long id) {
        ProductPO productPO = productMapper.selectById(id);
        ProductEntity productBO = ProductConvertor.convertToBOEntity(productPO);
        return productBO;
    }

    @Override
    public ProductEntity queryUnionProductById(Long id) {
        UnionProductPO productPO = unionProductMapper.selectById(id);
        ProductEntity productBO = ProductConvertor.convertToBOEntity(productPO);
        return productBO;
    }

    @Override
    public List<ProductEntity> queryProductByApIdAndIpAndName(String apId, String ip, String name, boolean ignoreNullId) {
        LambdaQueryWrapper<ProductPO> wrapper = Wrappers.<ProductPO>lambdaQuery()
                .eq(StringUtils.isNotBlank(name), ProductPO::getName, name)
                .eq(StringUtils.isNotBlank(ip), ProductPO::getIpAddress, ip)
                .eq(ProductPO::getIsvalid, true);
        if (StringUtils.isNotBlank(apId)) {
            wrapper.eq(ProductPO::getApid, apId);
        } else if (!ignoreNullId) {
            wrapper.isNull(ProductPO::getApid);
        }
        List<ProductPO> productPOList = productMapper.selectList(wrapper);
        if (CollectionUtils.isEmpty(productPOList)) {
            return null;
        }
        return productPOList.stream().map(po -> ProductConvertor.convertToBOEntity(po)).collect(Collectors.toList());
    }

    @Override
    public List<ProductEntity> queryUnionProductByUid(String uid) {
        List<UnionProductPO> productPOList = unionProductMapper.selectList(Wrappers.<UnionProductPO>lambdaQuery()
                .eq(UnionProductPO::getUid, uid)
                .eq(UnionProductPO::getIsvalid, true)
        );
        if (CollectionUtils.isEmpty(productPOList)) {
            return null;
        }
        return productPOList.stream().map(po -> ProductConvertor.convertToBOEntity(po)).collect(Collectors.toList());
    }

    @Override
    public List<ProductEntity> queryUnionProductByApIdAndIpAndName(String apId, String ip, String name) {
        LambdaQueryWrapper<UnionProductPO> wrapper = Wrappers.<UnionProductPO>lambdaQuery()
                .eq(StringUtils.isNotBlank(name), UnionProductPO::getName, name)
                .eq(StringUtils.isNotBlank(ip), UnionProductPO::getIpAddress, ip)
                .eq(UnionProductPO::getIsvalid, true);
        if (StringUtils.isNotBlank(apId)) {
            wrapper.eq(UnionProductPO::getApid, apId);
        } else {
            wrapper.isNull(UnionProductPO::getApid);
        }
        List<UnionProductPO> productPOList = unionProductMapper.selectList(wrapper);
        if (CollectionUtils.isEmpty(productPOList)) {
            return null;
        }
        return productPOList.stream().map(po -> ProductConvertor.convertToBOEntity(po)).collect(Collectors.toList());
    }

    @Override
    public List<ProductEntity> queryNotClientOnlyProductByName(String name) {
        LambdaQueryWrapper<ProductPO> wrapper = Wrappers.<ProductPO>lambdaQuery()
                .eq(ProductPO::getName, name)
                .eq(ProductPO::getClientOnly, false)
                .eq(ProductPO::getIsvalid, true);
        List<ProductPO> productPOList = productMapper.selectList(wrapper);
        if (CollectionUtils.isEmpty(productPOList)) {
            return null;
        }
        return productPOList.stream().map(po -> ProductConvertor.convertToBOEntity(po)).collect(Collectors.toList());
    }

    @Override
    public List<ProductEntity> queryAllNotClientOnlyProduct() {
        LambdaQueryWrapper<ProductPO> wrapper = Wrappers.<ProductPO>lambdaQuery()
                .eq(ProductPO::getClientOnly, false)
                .eq(ProductPO::getIsvalid, true);
        List<ProductPO> productPOList = productMapper.selectList(wrapper);
        if (CollectionUtils.isEmpty(productPOList)) {
            return null;
        }
        return productPOList.stream().map(po -> ProductConvertor.convertToBOEntity(po)).collect(Collectors.toList());
    }

    @Override
    public ProductEntity findProductInCache(String name, String ip, String apId, String uid) {
        AtomicReference<ProductPO> tProductPO = new AtomicReference<>();
        if (StringUtils.isNotBlank(uid)) {
            tProductPO.set(productInfoCacheService.getProductByUid(name, uid));
        } else {
            List<ProductPO> tProductPOList = productInfoCacheService.getProductByIpId(name, ip, apId);
            if (CollectionUtils.isNotEmpty(tProductPOList)) {
                if (tProductPOList.size() == 1) {
                    tProductPO.set(tProductPOList.get(0));
                } else if (tProductPOList.size() > 1) {
                    tProductPOList.forEach(productPO -> {
                        if (productPO.getIsdefault()) {
                            tProductPO.set(productPO);
                        }
                    });
                }
            }
        }
        return ProductConvertor.convertToBOEntity(tProductPO.get());
    }

    @Override
    public ProductEntity findProductInDB(String name, String ip, String apId, String uid) {
        AtomicReference<ProductEntity> tProductBO = new AtomicReference<>();
        if (StringUtils.isNotBlank(uid)) {
            List<ProductEntity> tList = queryProductByUid(uid);
            if (CollectionUtils.isNotEmpty(tList)) {
                tList.forEach(productEntity -> {
                    if (productEntity.getName().equals(name)) {
                        tProductBO.set(productEntity);
                    }
                });
            }
        } else {
            LambdaQueryWrapper<ProductPO> tWrapper = Wrappers.<ProductPO>lambdaQuery()
                    .eq(ProductPO::getName, name)
                    .eq(ProductPO::getIsvalid, true);
            if (StringUtils.isNotBlank(ip)) {
                tWrapper.eq(ProductPO::getIpAddress, ip);
            }
            if (StringUtils.isNotBlank(apId)) {
                tWrapper.eq(ProductPO::getApid, apId);
            }
            AtomicReference<ProductPO> tProductPO = new AtomicReference<>();
            List<ProductPO> tProductPOList = productMapper.selectList(tWrapper);
            if (tProductPOList.size() == 1) {
                tProductPO.set(tProductPOList.get(0));
            } else if (tProductPOList.size() > 1) {
                tProductPOList.forEach(productPO -> {
                    if (productPO.getIsdefault()) {
                        tProductPO.set(productPO);
                    }
                });
            }
            tProductBO.set(ProductConvertor.convertToBOEntity(tProductPO.get()));
        }
        return tProductBO.get();
    }

    @Override
    public void saveProductCache(ProductEntity productEntity) {
        productInfoCacheService.addProduct(ProductConvertor.convertToProductPO(productEntity));
    }

    @Override
    public ProductEntity findUnionProductInCache(String eaiUid, String prodUid, String prodName) {
        UnionProductPO tUnionProductPO = productInfoCacheService.getUnionProduct(eaiUid, prodUid, prodName);

        if (tUnionProductPO != null) {
            return ProductConvertor.convertToBOEntity(tUnionProductPO);
        } else {
            return null;
        }
    }

    @Override
    public List<ProductEntity> findProductInCache(String name) {
        List<ProductPO> tProductPOList = productInfoCacheService.getProductsByName(name);

        if (CollectionUtils.isNotEmpty(tProductPOList)) {
            return tProductPOList.stream().map(ProductConvertor::convertToBOEntity).collect(Collectors.toList());
        }
        return null;
    }

    @Override
    public List<ProductEntity> findProductInDB(String name) {
        List<ProductPO> tProductPOList = productMapper.selectList(
                Wrappers.lambdaQuery(ProductPO.class)
                        .eq(ProductPO::getName, name)
                        .eq(ProductPO::getIsvalid, true));
        if (CollectionUtils.isNotEmpty(tProductPOList)) {
            return tProductPOList.stream().map(ProductConvertor::convertToBOEntity).collect(Collectors.toList());
        }
        return null;
    }

    @Override
    public boolean existsUid(String uid) {
        long count = productMapper.selectCount(Wrappers.<ProductPO>lambdaQuery()
                .eq(ProductPO::getUid, uid)
                .eq(ProductPO::getIsvalid, true));
        return count > 0;
    }

    @Override
    public List<ProductEntity> queryAllProduct() {
        List<ProductPO> productPOList = productMapper.selectList(Wrappers.<ProductPO>lambdaQuery().eq(ProductPO::getIsvalid, true));
        List<UnionProductPO> unionProductPOList = unionProductMapper.selectList(Wrappers.<UnionProductPO>lambdaQuery().eq(UnionProductPO::getIsvalid, true));
        List<ProductEntity> entityList = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(productPOList)) {
            productPOList.forEach(one -> entityList.add(ProductConvertor.convertToBOEntity(one)));
        }
        if (CollectionUtils.isNotEmpty(unionProductPOList)) {
            unionProductPOList.forEach(one -> entityList.add(ProductConvertor.convertToBOEntity(one)));
        }
        return entityList;
    }

    @Override
    public List<ProductEntity> queryUnionProduct(Long eaiId) {
        List<UnionProductPO> unionProductPOList = unionProductMapper.selectList(Wrappers.<UnionProductPO>lambdaQuery().eq(UnionProductPO::getIsvalid, true).eq(UnionProductPO::getEaiId, eaiId));
        List<ProductEntity> entityList = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(unionProductPOList)) {
            unionProductPOList.forEach(one -> entityList.add(ProductConvertor.convertToBOEntity(one)));
        }
        return entityList;
    }

    @Override
    public List<ProductEntity> queryProductByCondition(ProductEntity productEntity) {
        ProductPO productPO = ProductConvertor.convertToProductPO(productEntity);
        List<ProductPO> productPOList = null;
        if (StringUtils.isNotBlank(productPO.getUid())) {
            productPOList = productMapper.selectList(Wrappers.<ProductPO>lambdaQuery().eq(ProductPO::getName, productPO.getName()).eq(ProductPO::getUid, productPO.getUid()));
        } else {
            productPOList = productMapper.selectList(Wrappers.query(productPO));
        }
        if (CollectionUtils.isEmpty(productPOList)) {
            return null;
        }
        return productPOList.stream().map(one -> ProductConvertor.convertToBOEntity(one)).collect(Collectors.toList());
    }
}
