package com.digiwin.cross.infrastructure.gatewayimpl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.digiwin.cross.domain.bo.FullAsyncRequestBO;
import com.digiwin.cross.domain.bo.entity.EaiEntity;
import com.digiwin.cross.domain.bo.entity.ProductEntity;
import com.digiwin.cross.domain.bo.entity.ServiceEntity;
import com.digiwin.cross.domain.bo.entity.ServiceVersionEntity;
import com.digiwin.cross.domain.gateway.IEaiGateway;
import com.digiwin.cross.domain.gateway.IFullAsyncRequestGateway;
import com.digiwin.cross.domain.gateway.IProductQueryGateWay;
import com.digiwin.cross.domain.parameter.ApplicationSystemParameter;
import com.digiwin.cross.domain.service.validator.bo.ProviderBO;
import com.digiwin.cross.domain.service.validator.bo.RequesterBO;
import com.digiwin.cross.infrastructure.cache.service.CallbackTimeoutCacheMapper;
import com.digiwin.cross.infrastructure.cache.service.FullAsyncRequestCacheMapper;
import com.digiwin.cross.infrastructure.convertor.FullAsyncRequestConvertor;
import com.digiwin.cross.infrastructure.database.entity.FullAsyncRequestPO;
import com.digiwin.cross.infrastructure.database.mapper.FullAsyncRequestMapper;
import com.digiwin.cross.infrastructure.database.mapper.LogMapper;
import com.digiwin.cross.infrastructure.rpc.mapper.AlarmSenderMapper;
import com.digiwin.cross.infrastructure.rpc.req.CallbackTimeOutAlarmReq;
import lombok.extern.apachecommons.CommonsLog;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.stereotype.Component;

import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @description:
 * @author: liunansheng
 * @date: 2023/6/26 16:34
 */
@CommonsLog
@Component
public class FullAsyncRequestGateway implements IFullAsyncRequestGateway {

    private final FullAsyncRequestMapper fullAsyncRequestMapper;

//    private final FullAsyncRequestCacheMapper fullAsyncRequestCacheMapper;

    private final IProductQueryGateWay productQueryGateWay;

    private final IEaiGateway eaiGateway;

    private final AlarmSenderMapper alarmSenderMapper;

    private final LogMapper logMapper;

    private final CallbackTimeoutCacheMapper callbackTimeoutCacheMapper;

    public FullAsyncRequestGateway (FullAsyncRequestMapper fullAsyncRequestMapper,
                                    FullAsyncRequestCacheMapper fullAsyncRequestCacheMapper,
                                    IProductQueryGateWay productQueryGateWay,
                                    IEaiGateway eaiGateway,
                                    AlarmSenderMapper alarmSenderMapper,
                                    LogMapper logMapper,
                                    CallbackTimeoutCacheMapper callbackTimeoutCacheMapper) {
        this.fullAsyncRequestMapper = fullAsyncRequestMapper;
//        this.fullAsyncRequestCacheMapper = fullAsyncRequestCacheMapper;

        this.productQueryGateWay = productQueryGateWay;
        this.eaiGateway = eaiGateway;
        this.alarmSenderMapper = alarmSenderMapper;
        this.logMapper = logMapper;
        this.callbackTimeoutCacheMapper = callbackTimeoutCacheMapper;
    }


    @Override
    public FullAsyncRequestBO getFullAsyncRequestInfo (String reqId, int retryTimes, long retryIntervalMillis) {
        FullAsyncRequestBO bo = null;
        while (retryTimes >= 0) {
            FullAsyncRequestPO po = fullAsyncRequestMapper.selectById(reqId);
            if (po != null) {
                bo = FullAsyncRequestConvertor.convertToBO(po);
                bo.setRequester(getRequester(po));
                bo.setProvider(getProvider(po));
                break;
            } else {
                retryTimes--;
                if (retryTimes < 0) {
                    break;
                }
                log.info(String.format("Callback requestId not found; waiting to retry:%s", reqId));
                sleep(retryIntervalMillis);
            }
        }

        return bo;
    }

    @Override
    public void saveFullAsyncRequestInfo (FullAsyncRequestBO fullAsyncRequestBO) {


//        EXECUTOR.execute(() -> {
//            try {
//                fullAsyncRequestCacheMapper.save(po);
//            } catch (Exception e) {
//                log.error(e.getMessage(), e);
//            }
//        });

        String tReqId = fullAsyncRequestBO.getReqId();
        try {
            FullAsyncRequestPO po = FullAsyncRequestConvertor.convertToPO(fullAsyncRequestBO);
            Calendar nowTime = Calendar.getInstance();
            po.setPlatformTime(nowTime.getTime());

            String productName = fullAsyncRequestBO.getProvider().getProduct().getName();
            String serviceName = fullAsyncRequestBO.getProvider().getService().getServiceName();

            Integer timeout = callbackTimeoutCacheMapper.getServiceTimeout(productName, serviceName);
            if (timeout == null) {
                timeout = callbackTimeoutCacheMapper.getProductTimeout(productName);
            }
            if (timeout == null) {
                timeout = ApplicationSystemParameter.FULL_ASYNC_CALLBACK_TIMEOUT;
            }

            nowTime.add(Calendar.SECOND, timeout);
            po.setTimeout(nowTime.getTime());

            fullAsyncRequestMapper.insert(po);
        } catch (Exception e) {
            log.error("Save fasync requester failed, req_id=" + tReqId, e);
        }

    }

    @Override
    public void removeFullAsyncRequestInfo (String reqId) {
        try {
            fullAsyncRequestMapper.deleteById(reqId);
        } catch (Exception e) {
            log.error("Delete fasync requester failed, req_id=" + reqId, e);
        }
//        EXECUTOR.execute(() -> {
//            try {
//                fullAsyncRequestCacheMapper.remove(reqId);
//            } catch (Exception e) {
//                log.error(e.getMessage(), e);
//            }
//        });
    }

    @Override
    public List<FullAsyncRequestBO> getTimeOutFullAsyncRequestInfo (Date start, Date end) {
        List<FullAsyncRequestPO> infos = fullAsyncRequestMapper.selectList(
                Wrappers.<FullAsyncRequestPO>lambdaQuery()
                        .ge(FullAsyncRequestPO::getPlatformTime, start)
                        .lt(FullAsyncRequestPO::getTimeout, end)
                        .eq(FullAsyncRequestPO::getTimeoutChecked, false)
                        .eq(FullAsyncRequestPO::getCallBacked, false)
        );
        if (CollectionUtils.isEmpty(infos)) {
            return null;
        }
        return infos.stream().map(po -> FullAsyncRequestConvertor.convertToBO(po)).collect(Collectors.toList());
    }

    @Override
    public void saveTimeOutCheckTag (String reqId) {
        FullAsyncRequestPO po = new FullAsyncRequestPO();
        po.setReqId(reqId);
        po.setTimeoutChecked(true);
        fullAsyncRequestMapper.updateById(po);
    }

    /**
     * 发送告警超时信息
     *
     * @param fullAsyncRequestBO
     */
    @Override
    public void alarmCallbackTimeOut (FullAsyncRequestBO fullAsyncRequestBO) {
        CallbackTimeOutAlarmReq po = FullAsyncRequestConvertor.convertToCallbackTimeOutAlarmPO(fullAsyncRequestBO);
        alarmSenderMapper.sendInvokeAlarm(fullAsyncRequestBO.getReqId(), po);
    }

    /**
     * 设置当前请求已经callBack了
     *
     * @param reqId
     */
    @Override
    public void setCallBackTag (String reqId) {
        FullAsyncRequestPO po = new FullAsyncRequestPO();
        po.setReqId(reqId);
        po.setCallBacked(true);
        fullAsyncRequestMapper.updateById(po);
    }

    @Override
    public void deleteOlderThanDays (int days, int batchSize) {
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.DAY_OF_YEAR, -days);
        LambdaQueryWrapper<FullAsyncRequestPO> queryWrapper = Wrappers.lambdaQuery(FullAsyncRequestPO.class)
                .lt(FullAsyncRequestPO::getPlatformTime, calendar.getTime());

        while (true) {
            //迴圈批次刪除
            queryWrapper.last("LIMIT " + batchSize);
            List<FullAsyncRequestPO> deleteList = fullAsyncRequestMapper.selectList(queryWrapper);

            //沒有紀錄退出迴圈
            if (CollectionUtils.isEmpty(deleteList)) {
                break;
            }

            List<String> reqIdToDelete = deleteList.stream()
                    .map(FullAsyncRequestPO::getReqId)
                    .collect(Collectors.toList());

            int deleteCount = fullAsyncRequestMapper.deleteBatchIds(reqIdToDelete);

            log.info("Deleted a batch of expired fullAsyncRequest, total: " + deleteCount + " records.");

            // 延遲5秒
            try {
                Thread.sleep(TimeUnit.SECONDS.toMillis(5));
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                log.error(e.getMessage(), e);
            }
        }
    }

    private RequesterBO getRequester (FullAsyncRequestPO po) {
        if (po.getReqProdId() != null) {
            ProductEntity productBO = productQueryGateWay.queryProductById(po.getReqProdId());
            return new RequesterBO(productBO, null);
        }
        if (po.getReqEaiUid() != null) {
            EaiEntity eaiEntity = eaiGateway.queryEai(po.getReqEaiUid());
            return new RequesterBO(null, eaiEntity);
        }
        return null;
    }

    private ProviderBO getProvider (FullAsyncRequestPO po) {
        ServiceEntity serviceEntity = new ServiceEntity();
        serviceEntity.setServiceName(po.getSrvName());
        serviceEntity.setServiceDescription(po.getSrvDesc());
        serviceEntity.setVersion(new ServiceVersionEntity(null, null, po.getSrvVer()));
        if (po.getSrvProdId() != null) {
            ProductEntity productBO = productQueryGateWay.queryProductById(po.getSrvProdId());
            return new ProviderBO(productBO, serviceEntity, null);
        }
        if (po.getUnionSrvProdId() != null) {
            ProductEntity productBO = productQueryGateWay.queryUnionProductById(po.getUnionSrvProdId());
            EaiEntity eaiEntity = eaiGateway.queryEai(po.getUnionSrvEaiId());
            return new ProviderBO(productBO, serviceEntity, eaiEntity);
        }
        return null;
    }

    private void sleep (long retryIntervalMillis) {
        try {
            Thread.sleep(retryIntervalMillis);
        } catch (InterruptedException e) {
            log.error(e);
            Thread.currentThread().interrupt();
        }
    }
}
