package com.digiwin.cross.infrastructure.gatewayimpl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.digiwin.athena.esp.sdk.util.StringUtil;
import com.digiwin.cross.domain.bo.LogBO;
import com.digiwin.cross.domain.enums.WriteLogEnum;
import com.digiwin.cross.domain.gateway.ILogGateway;
import com.digiwin.cross.domain.parameter.ApplicationSystemParameter;
import com.digiwin.cross.infrastructure.cache.CacheKeyEnum;
import com.digiwin.cross.infrastructure.cache.service.EspRedisService;
import com.digiwin.cross.infrastructure.convertor.LogConvertor;
import com.digiwin.cross.infrastructure.database.entity.LogMessagePO;
import com.digiwin.cross.infrastructure.database.entity.LogPO;
import com.digiwin.cross.infrastructure.database.mapper.LogMapper;
import com.digiwin.cross.infrastructure.database.mapper.LogMessageMapper;
import com.digiwin.cross.infrastructure.rpc.mapper.AlarmSenderMapper;
import com.digiwin.cross.infrastructure.rpc.req.BigResAlarmReq;
import com.digiwin.cross.infrastructure.rpc.mapper.WriteMmcMapper;
import lombok.extern.apachecommons.CommonsLog;
import org.apache.commons.lang3.time.DateUtils;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.util.Date;
import java.util.List;

/**
 * @description:
 * @author: liunansheng
 * @date: 2023/6/20 16:19
 */
@Component
@CommonsLog
public class LogGateway implements ILogGateway {

    private LogMapper logMapper;
    private LogMessageMapper logMessageMapper;
    private WriteMmcMapper writeMmcMapper;
    private final AlarmSenderMapper alarmSenderMapper;
    private EspRedisService espRedisService;

    public LogGateway(LogMapper logMapper, LogMessageMapper logMessageMapper, WriteMmcMapper writeMmcMapper, AlarmSenderMapper alarmSenderMapper, EspRedisService espRedisService) {
        this.logMapper = logMapper;
        this.logMessageMapper = logMessageMapper;
        this.writeMmcMapper = writeMmcMapper;
        this.alarmSenderMapper = alarmSenderMapper;
        this.espRedisService = espRedisService;
    }

    /**
     * 写日志
     * @param logBO
     */
    @Override
    @Transactional("espTransactionManager")
    public void writeLog(LogBO logBO) {
        // 获取当前存储日志配置
        String logConfig = espRedisService.getFromL2Cache(CacheKeyEnum.ESP_SWITCH, "log_config");
        if (StringUtil.isEmpty(logConfig)) {
            logConfig = ApplicationSystemParameter.LOG_CONFIG;
            espRedisService.putToL2Cache(CacheKeyEnum.ESP_SWITCH, "log_config", logConfig);
        }
        WriteLogEnum logEnum = WriteLogEnum.valueOf(logConfig);

        // 写db日志
        if (logEnum == WriteLogEnum.WRITE_LOG_DB || logEnum == WriteLogEnum.WRITE_LOG_ALL) {
            dbLogSave(LogConvertor.convertToLogPO(logBO), LogConvertor.convertToLogMessagePO(logBO));
        }

        // 写mcc日志
        if (logEnum == WriteLogEnum.WRITE_LOG_MMC || logEnum == WriteLogEnum.WRITE_LOG_ALL) {
            writeMmcMapper.createMmcLog(LogConvertor.convertToWriteMmcLogReq(logBO));
        }
    }

    /**
     * 数据库存储
     * @param tLogPO
     * @param tLogMessagePO
     */
    private void dbLogSave(LogPO tLogPO, LogMessagePO tLogMessagePO){
        try {
            logMapper.insert(tLogPO);
            tLogMessagePO.setLogId(tLogPO.getId());
            logMessageMapper.insert(tLogMessagePO);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
    }

    @Override
    @Transactional("espTransactionManager")
    public void writeLogs(List<LogBO> logBOList) {
        logBOList.forEach(this::writeLog);
    }

    @Override
    public void cleanExpiredLogs() {
        if (null == ApplicationSystemParameter.LOG_RESERVED_DAY) {
            log.info("log saveOrDelete dates not config");
            return;
        }
        Date endDate = DateUtils.addDays(new Date(), ApplicationSystemParameter.LOG_RESERVED_DAY * -1);
        QueryWrapper<LogPO> logWrapper = Wrappers.query();
        logWrapper.select("max(id) as id");
        logWrapper.lt("created_time", endDate);
        LogPO logPO = logMapper.selectOne(logWrapper);
        if (null == logPO) {
            log.info("no log need clean");
            return;
        }
        log.info("start clean log, max logId is " + logPO.getId());
        while (true) {
            LambdaQueryWrapper<LogMessagePO> condition = Wrappers.lambdaQuery();
            condition.le(LogMessagePO::getLogId, logPO.getId());
            condition.last("limit 5000");
            int c = logMessageMapper.delete(condition);
            if (c < 5000) {
                break;
            }
        }
        while (true) {
            LambdaQueryWrapper<LogPO> condition = Wrappers.lambdaQuery();
            condition.le(LogPO::getId, logPO.getId());
            condition.last("limit 5000");
            int c = logMapper.delete(condition);
            if (c < 5000) {
                break;
            }
        }
        log.info("clean log success");
    }

    @Override
    public void alarmBigRes(LogBO logBO) {
        BigResAlarmReq po = LogConvertor.convertToBigResAlarmPO(logBO);
        alarmSenderMapper.sendInvokeAlarm(logBO.getReqId(), po);
    }

}
