package com.digiwin.cross.infrastructure.cache.service;

import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.digiwin.cross.infrastructure.cache.CacheKeyEnum;
import com.digiwin.cross.infrastructure.database.entity.*;
import com.digiwin.cross.infrastructure.database.mapper.*;
import lombok.extern.apachecommons.CommonsLog;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @description:
 * @author: clay
 * @date: 2023/6/12
 */
@Component
@CommonsLog
public class ServiceInfoCacheService {

    private EspRedisService espRedisService;

    private ProductMapper productMapper;
    private ServiceMapper serviceMapper;
    private UnionEaiMapper unionEaiMapper;
    private UnionProductMapper unionProductMapper;
    private UnionServiceMapper unionServiceMapper;

    public ServiceInfoCacheService(EspRedisService espRedisService, ProductMapper productMapper, ServiceMapper serviceMapper, UnionEaiMapper unionEaiMapper, UnionProductMapper unionProductMapper, UnionServiceMapper unionServiceMapper) {
        this.espRedisService = espRedisService;
        this.productMapper = productMapper;
        this.serviceMapper = serviceMapper;
        this.unionEaiMapper = unionEaiMapper;
        this.unionProductMapper = unionProductMapper;
        this.unionServiceMapper = unionServiceMapper;
    }

//    //key=prodUid
//    private final Map<String, Map<String, ServicePO>> serviceCacheMap = new HashMap<>();
//
//    //key=unionEaiUid@unionProdUid
//    private final Map<String, Map<String, UnionServicePO>> unionServiceCacheMap = new HashMap<>();

    @PostConstruct
    public void init() {
        Timer timer = new Timer();
        timer.schedule(new TimerTask() {
            @Override
            public void run() {
                try {
                    loadCache();
                    //載入完成
                    timer.cancel();
                } catch (Exception e) {
                    log.error(e.getMessage(), e);
                }
            }
        }, 0, TimeUnit.SECONDS.toMillis(30));
    }

    public void loadCache() throws Exception {
        if (espRedisService.isEnable()) {
            return;
        }
        espRedisService.clearByNamespace(CacheKeyEnum.ESP_SERVICE_INFO_NEW);
        espRedisService.clearByNamespace(CacheKeyEnum.ESP_UNION_SERVICE_INFO_NEW);
        //讀取敏態服務
        //找有啟用的產品
        List<ProductPO> tProductPOList = productMapper.selectList(
                Wrappers.<ProductPO>lambdaQuery()
                        .eq(ProductPO::getIsvalid, true)
        );
        Map<Long, String> tProdMap = tProductPOList.stream().filter(productPO -> !productPO.getClientOnly()).collect(Collectors.toMap(ProductPO::getId, ProductPO::getUid));

        List<ServicePO> tServicePOList = serviceMapper.selectList(
                Wrappers.<ServicePO>lambdaQuery()
                        .in(ServicePO::getProductId, tProdMap.keySet())
        );
        if (CollectionUtils.isNotEmpty(tServicePOList)) {
            Map<String, Map<String, ServicePO>> serviceCacheMap = new HashMap<>();
            tServicePOList.forEach(servicePO -> {
                Map<String, ServicePO> tServicePOMap = serviceCacheMap.computeIfAbsent(tProdMap.get(servicePO.getProductId()), k -> new HashMap<>());
                tServicePOMap.put(servicePO.getServiceName(), servicePO);
            });
            serviceCacheMap.forEach((prodId, servicePOMap) -> espRedisService.put(CacheKeyEnum.ESP_SERVICE_INFO_NEW, prodId, servicePOMap));
        }

        List<UnionEAIPO> tUnionEAIPOList = unionEaiMapper.selectList(
                Wrappers.<UnionEAIPO>lambdaQuery()
                        .eq(UnionEAIPO::getIsvalid, true)
        );
        if (CollectionUtils.isNotEmpty(tUnionEAIPOList)) {
            Map<Long, String> tEaiMap = tUnionEAIPOList.stream().collect(Collectors.toMap(UnionEAIPO::getId, UnionEAIPO::getUid));

            List<UnionProductPO> tUnionProductPOList = unionProductMapper.selectList(
                    Wrappers.<UnionProductPO>lambdaQuery()
                            .eq(UnionProductPO::getIsvalid, true)
                            .in(UnionProductPO::getEaiId, tEaiMap.keySet())
            );
            if (CollectionUtils.isNotEmpty(tUnionProductPOList)) {
                Map<Long, UnionProductPO> tUnionProdMap = tUnionProductPOList.stream().collect(Collectors.toMap(UnionProductPO::getId, Function.identity()));
                List<UnionServicePO> tUnionServicePOList = unionServiceMapper.selectList(
                        Wrappers.<UnionServicePO>lambdaQuery()
                                .in(UnionServicePO::getUnionProductId, tUnionProdMap.keySet())
                );
                if (CollectionUtils.isNotEmpty(tUnionServicePOList)) {
                    Map<String, Map<String, UnionServicePO>> unionServiceCacheMap = new HashMap<>();
                    tUnionServicePOList.forEach(unionServicePO -> {
                        UnionProductPO tUnionProductPO = tUnionProdMap.get(unionServicePO.getUnionProductId());
                        String tEaiUid = tEaiMap.get(tUnionProductPO.getEaiId());

                        String tKey = getUnionKey(tEaiUid, tUnionProductPO.getUid());

                        Map<String, UnionServicePO> tUnionServicePOMap = unionServiceCacheMap.computeIfAbsent(tKey, k -> new HashMap<>());
                        tUnionServicePOMap.put(unionServicePO.getName(), unionServicePO);
                    });
                    unionServiceCacheMap.forEach((unionProdId, servicePOMap) -> espRedisService.put(CacheKeyEnum.ESP_UNION_SERVICE_INFO_NEW, unionProdId, servicePOMap));
                }

            }

        }

    }

    private String getUnionKey(String eaiUid, String unionPordUid) {
        return String.format("%s@%s", eaiUid, unionPordUid);
    }

    public ServicePO findService(String prodUid, String serviceName) {
        ServicePO tServicePO = null;

        Map<String, ServicePO> tServicePOMap = null;
        if (espRedisService.isEnable()) {
            tServicePOMap = espRedisService.get(CacheKeyEnum.ESP_SERVICE_INFO_NEW, prodUid);
        }
        if (tServicePOMap != null && CollectionUtils.isNotEmpty(tServicePOMap)) {
            tServicePO = tServicePOMap.get(serviceName);
        }
        return tServicePO;
    }

    public UnionServicePO findUnionService(String eaiUid, String prodUid, String serviceName) {
        UnionServicePO tServicePO = null;

        String tKey = getUnionKey(eaiUid, prodUid);
        Map<String, UnionServicePO> tServicePOMap = null;
        if (espRedisService.isEnable()) {
            tServicePOMap = espRedisService.get(CacheKeyEnum.ESP_UNION_SERVICE_INFO_NEW, tKey);
        }
        if (tServicePOMap != null && CollectionUtils.isNotEmpty(tServicePOMap)) {
            tServicePO = tServicePOMap.get(serviceName);
        }
        return tServicePO;
    }

    public void deleteUnionService(String eaiUid, List<String> prodUids) {
        prodUids.forEach(prodUid -> {
            String key = getUnionKey(eaiUid, prodUid);
            if (espRedisService.isEnable()) {
                espRedisService.remove(CacheKeyEnum.ESP_UNION_SERVICE_INFO_NEW, key);
            }
        });
    }

    public void addUnionService(String eaiUid, String prodUid, UnionServicePO unionServicePO) {
        String key = getUnionKey(eaiUid, prodUid);
        if (espRedisService.isEnable()) {
            Map<String, UnionServicePO> tUnionServiceMap = espRedisService.get(CacheKeyEnum.ESP_UNION_SERVICE_INFO_NEW, key);
            if (tUnionServiceMap == null) {
                tUnionServiceMap = new HashMap<>();
            }
            tUnionServiceMap.put(unionServicePO.getName(), unionServicePO);
            espRedisService.put(CacheKeyEnum.ESP_UNION_SERVICE_INFO_NEW, key, tUnionServiceMap);
        }
    }

    public void addUnionService(String eaiUid, List<Pair<UnionProductPO, List<UnionServicePO>>> prodServiceList) {
        prodServiceList.forEach(prodService -> {
            if (CollectionUtils.isEmpty(prodServiceList)) {
                return;
            }
            String key = getUnionKey(eaiUid, prodService.getLeft().getUid());
            if (espRedisService.isEnable()) {
                Map<String, UnionServicePO> serviceMap = prodService.getRight().stream().collect(Collectors.toMap(UnionServicePO::getName, srv -> srv));
                espRedisService.put(CacheKeyEnum.ESP_UNION_SERVICE_INFO_NEW, key, serviceMap);
            }
        });
    }

    public void addOrUpdateService(String prodUid, ServicePO servicePO) {
        if (espRedisService.isEnable()) {
            Map<String, ServicePO> serviceMap = espRedisService.get(CacheKeyEnum.ESP_SERVICE_INFO_NEW, prodUid);
            if (null == serviceMap) {
                serviceMap = new HashMap<>();
            }
            serviceMap.put(servicePO.getServiceName(), servicePO);
            espRedisService.put(CacheKeyEnum.ESP_SERVICE_INFO_NEW, prodUid, serviceMap);
        }
    }

    public void deleteService(String prodUid, String serviceName) {
        if (espRedisService.isEnable()) {
            Map<String, ServicePO> serviceMap = espRedisService.get(CacheKeyEnum.ESP_SERVICE_INFO_NEW, prodUid);
            if (serviceMap != null && serviceMap.remove(serviceName) != null) {
                espRedisService.put(CacheKeyEnum.ESP_SERVICE_INFO_NEW, prodUid, serviceMap);
            }
        }
    }

    public void deleteServiceByProduct(String prodUid) {
        if (espRedisService.isEnable()) {
            espRedisService.remove(CacheKeyEnum.ESP_SERVICE_INFO_NEW, prodUid);
        }
    }
}
