package com.digiwin.cross.infrastructure.gatewayimpl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.digiwin.cross.domain.bo.CallbackTimeoutDataBO;
import com.digiwin.cross.domain.bo.CallbackTimeoutProductBO;
import com.digiwin.cross.domain.bo.CallbackTimeoutServiceBO;
import com.digiwin.cross.domain.bo.entity.CallbackTimeoutEntity;
import com.digiwin.cross.domain.enums.CallbackTimeoutTypeEnum;
import com.digiwin.cross.domain.gateway.ICallbackTimeoutGateway;
import com.digiwin.cross.infrastructure.cache.service.CallbackTimeoutCacheMapper;
import com.digiwin.cross.infrastructure.database.entity.CallbackTimeoutPO;
import com.digiwin.cross.infrastructure.database.mapper.CallbackTimeoutMapper;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.stream.Collectors;

/**
 * @author clay
 * on 2024/9/3
 */
@Component
public class CallbackTimeoutGateway implements ICallbackTimeoutGateway {

    private CallbackTimeoutMapper callbackTimeoutMapper;
    private CallbackTimeoutCacheMapper callbackTimeoutCacheMapper;
    private static ExecutorService executorService = Executors.newSingleThreadExecutor();

    public CallbackTimeoutGateway (CallbackTimeoutMapper callbackTimeoutMapper, CallbackTimeoutCacheMapper callbackTimeoutCacheMapper) {
        this.callbackTimeoutMapper = callbackTimeoutMapper;
        this.callbackTimeoutCacheMapper = callbackTimeoutCacheMapper;
    }

    @Override
    @Transactional("espTransactionManager")
    public void saveOrDelete (List<CallbackTimeoutDataBO> callbackTimeoutDataBOList) {
        Set<String> productNames = callbackTimeoutDataBOList.stream()
                .map(CallbackTimeoutDataBO::getProductName)
                .collect(Collectors.toSet());

        Set<String> serviceNames = callbackTimeoutDataBOList.stream()
                .map(CallbackTimeoutDataBO::getServiceName)
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());

        // 使用 LambdaQueryWrapper 構建批量查詢條件
        LambdaQueryWrapper<CallbackTimeoutPO> queryWrapper = Wrappers.lambdaQuery(CallbackTimeoutPO.class);

        // 構建條件：productName 和 serviceName 組合的查詢
        queryWrapper.in(CallbackTimeoutPO::getProductName, productNames)
                .and(wrapper -> {
                         if (CollectionUtils.isNotEmpty(serviceNames)) {
                             wrapper.in(CallbackTimeoutPO::getServiceName, serviceNames)
                                     .or()
                                     .isNull(CallbackTimeoutPO::getServiceName);
                         } else {
                             wrapper.isNull(CallbackTimeoutPO::getServiceName);
                         }
                     }

                );

        // 批量查詢數據庫中是否存在相同的 productName + serviceName 記錄
        List<CallbackTimeoutPO> existingRecords = callbackTimeoutMapper.selectList(queryWrapper);

        // 将结果转换为 Map 以便快速查找
        Map<String, CallbackTimeoutPO> existingMap = existingRecords.stream()
                .collect(Collectors.toMap(
                        record -> record.getProductName() + "_" + record.getServiceName(),
                        record -> record
                ));

        // 處理保存操作
        callbackTimeoutDataBOList.forEach(callbackTimeoutBO -> {
            if(callbackTimeoutBO.getTimeout() == null) {
                return;
            }
            String key = callbackTimeoutBO.getProductName() + "_" + callbackTimeoutBO.getServiceName();
            CallbackTimeoutPO callbackTimeoutPO = existingMap.get(key);

            if (callbackTimeoutPO != null) {
                // 如果記錄存在，更新或刪除
                if (callbackTimeoutBO.getTimeout() != 0) {
                    callbackTimeoutPO.setTimeout(callbackTimeoutBO.getTimeout());
                    callbackTimeoutMapper.updateById(callbackTimeoutPO);
                } else {
                    callbackTimeoutMapper.deleteById(callbackTimeoutPO.getId());
                }
            } else {
                // 如果記錄不存在，插入新記錄 時間為0不處理
                if (callbackTimeoutBO.getTimeout() != 0) {
                    callbackTimeoutPO = new CallbackTimeoutPO();
                    callbackTimeoutPO.setType(callbackTimeoutBO.getType().toString());
                    callbackTimeoutPO.setProductName(callbackTimeoutBO.getProductName());
//                callbackTimeoutPO.setProductUid(callbackTimeoutBO.getProductUid());
//                callbackTimeoutPO.setTenantId(callbackTimeoutBO.getTenantId());
                    callbackTimeoutPO.setServiceName(callbackTimeoutBO.getServiceName());
                    callbackTimeoutPO.setTimeout(callbackTimeoutBO.getTimeout());
                    callbackTimeoutMapper.insert(callbackTimeoutPO);
                }
            }
        });


        executorService.execute(() -> callbackTimeoutCacheMapper.loadCache());

    }

    @Override
    public CallbackTimeoutEntity find (CallbackTimeoutProductBO product, List<CallbackTimeoutServiceBO> serviceList) {
        LambdaQueryWrapper<CallbackTimeoutPO> queryWrapper = Wrappers.lambdaQuery(CallbackTimeoutPO.class).eq(CallbackTimeoutPO::getProductName, product.getProductName());
        if (CollectionUtils.isNotEmpty(serviceList)) {
            Set<String> serviceNameSet = serviceList.stream().map(CallbackTimeoutServiceBO::getServiceName).collect(Collectors.toSet());
            queryWrapper.and(wrapper -> wrapper.in(CallbackTimeoutPO::getServiceName, serviceNameSet)
                    .or()
                    .isNull(CallbackTimeoutPO::getServiceName));
        }

        List<CallbackTimeoutPO> callbackTimeoutPOList = callbackTimeoutMapper.selectList(queryWrapper);

        CallbackTimeoutEntity entity = new CallbackTimeoutEntity();
        CallbackTimeoutProductBO productBO = new CallbackTimeoutProductBO();
        List<CallbackTimeoutServiceBO> serviceBOList = new ArrayList<>();

        callbackTimeoutPOList.forEach(po -> {
            CallbackTimeoutTypeEnum type = CallbackTimeoutTypeEnum.valueOf(po.getType());
            switch (type) {
                case PRODUCT:
                    productBO.setProductName(po.getProductName());
                    productBO.setTimeout(po.getTimeout());
                    entity.setCallbackTImeoutProductBO(productBO);
                    break;
                case SERVICE:
                    CallbackTimeoutServiceBO serviceBO = new CallbackTimeoutServiceBO();
                    serviceBO.setServiceName(po.getServiceName());
                    serviceBO.setTimeout(po.getTimeout());
                    serviceBOList.add(serviceBO);
                    break;
            }
        });

        if (CollectionUtils.isNotEmpty(serviceBOList)) {
            entity.setCallbackTimeoutServiceBOList(serviceBOList);
        }

        return entity;
    }

}
