package com.digiwin.cross.infrastructure.gatewayimpl;

import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.digiwin.cross.domain.bo.EaiBO;
import com.digiwin.cross.domain.bo.ProductBO;
import com.digiwin.cross.domain.bo.entity.EaiEntity;
import com.digiwin.cross.domain.bo.entity.EaiTenantEntity;
import com.digiwin.cross.domain.gateway.IEaiGateway;
import com.digiwin.cross.domain.state.StateEnum;
import com.digiwin.cross.infrastructure.cache.service.ProductInfoCacheService;
import com.digiwin.cross.infrastructure.cache.service.ServiceInfoCacheService;
import com.digiwin.cross.infrastructure.convertor.EaiConvertor;
import com.digiwin.cross.infrastructure.convertor.ProductConvertor;
import com.digiwin.cross.infrastructure.convertor.ServiceConvertor;
import com.digiwin.cross.infrastructure.database.entity.EaiTenantMappingPO;
import com.digiwin.cross.infrastructure.database.entity.UnionEAIPO;
import com.digiwin.cross.infrastructure.database.entity.UnionProductPO;
import com.digiwin.cross.infrastructure.database.entity.UnionServicePO;
import com.digiwin.cross.infrastructure.database.mapper.EaiTenantMappingMapper;
import com.digiwin.cross.infrastructure.database.mapper.UnionEaiMapper;
import com.digiwin.cross.infrastructure.database.mapper.UnionProductMapper;
import com.digiwin.cross.infrastructure.database.mapper.UnionServiceMapper;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @description:
 * @author: liunansheng
 * @date: 2023/6/13 14:14
 */
@Component
public class EaiGateway implements IEaiGateway {

    private final UnionEaiMapper unionEaiMapper;

    private final EaiTenantMappingMapper eaiTenantMappingMapper;

    private final UnionProductMapper unionProductMapper;

    private final UnionServiceMapper unionServiceMapper;

    private final ProductInfoCacheService productInfoCacheService;

    private final ServiceInfoCacheService serviceInfoCacheService;

    public EaiGateway(UnionEaiMapper unionEaiMapper, EaiTenantMappingMapper eaiTenantMappingMapper, UnionProductMapper unionProductMapper, UnionServiceMapper unionServiceMapper, ProductInfoCacheService productInfoCacheService, ServiceInfoCacheService serviceInfoCacheService) {
        this.unionEaiMapper = unionEaiMapper;
        this.eaiTenantMappingMapper = eaiTenantMappingMapper;
        this.unionProductMapper = unionProductMapper;
        this.unionServiceMapper = unionServiceMapper;
        this.productInfoCacheService = productInfoCacheService;
        this.serviceInfoCacheService = serviceInfoCacheService;
    }

    @Override
    public EaiEntity queryEai(String eaiUid) {
        UnionEAIPO po = unionEaiMapper.selectOne(Wrappers.<UnionEAIPO>lambdaQuery().eq(UnionEAIPO::getUid, eaiUid));
        return EaiConvertor.convertToBO(po);
    }

    @Override
    public EaiEntity queryEai(Long id) {
        UnionEAIPO po = unionEaiMapper.selectById(id);
        return EaiConvertor.convertToBO(po);
    }

    @Override
    public EaiBO queryEaiWithUnionProductByTenantId(String tenantId) {
        List<EaiTenantMappingPO> tTenantMappingPOList = eaiTenantMappingMapper.selectList(Wrappers.<EaiTenantMappingPO>lambdaQuery().eq(EaiTenantMappingPO::getTenantId, tenantId));
        if (CollectionUtils.isNotEmpty(tTenantMappingPOList)) {
            UnionEAIPO tUnionEAIPO = unionEaiMapper.selectOne(Wrappers.<UnionEAIPO>lambdaQuery().eq(UnionEAIPO::getUid, tTenantMappingPOList.get(0).getEaiUid()));
            if (tUnionEAIPO != null) {
                List<UnionProductPO> tUnionProductPOList = unionProductMapper.selectList(Wrappers.<UnionProductPO>lambdaQuery().eq(UnionProductPO::getEaiId, tUnionEAIPO.getId()));
                EaiEntity eaiEntity = EaiConvertor.convertToBO(tUnionEAIPO);
                List<ProductBO> products = tUnionProductPOList.stream().map(ProductConvertor::convertToBO).collect(Collectors.toList());
                return new EaiBO(eaiEntity, null, products);
            }
        }
        return null;
    }

    @Transactional("espTransactionManager")
    @Override
    public void addEai(EaiBO eaiBO) {
        UnionEAIPO eaiPO = EaiConvertor.convertToPO(eaiBO.getEai());
        final Date now = new Date();
        eaiPO.setBuildTime(now);
        eaiPO.setLastUpdateResult(StateEnum.REGEAI_REGED.getCode());
        eaiPO.setLastUpdateTime(now);
        eaiPO.setIsvalid(true);
        unionEaiMapper.insert(eaiPO);
        eaiBO.getEai().setId(eaiPO.getId());
        addEaiChild(eaiPO, eaiBO);
    }

    @Transactional("espTransactionManager")
    @Override
    public void updateEai(EaiBO eaiBO) {
        EaiEntity eaiEntity = eaiBO.getEai();
        List<EaiTenantMappingPO> eaiTenantMappings = eaiTenantMappingMapper.selectList((Wrappers.<EaiTenantMappingPO>lambdaQuery().eq(EaiTenantMappingPO::getEaiUid, eaiEntity.getUid())));
        List<String> deleteTenantIds = eaiTenantMappings.stream().map(EaiTenantMappingPO::getTenantId)
                .filter(one -> {
                    for (EaiTenantEntity eaiTenantEntity : eaiBO.getTenants()) {
                        if (StringUtils.equals(one, eaiTenantEntity.getTenantId())) {
                            return false;
                        }
                    }
                    return true;
                }).collect(Collectors.toList());
        deleteEaiChild(eaiBO.getEai());
        productInfoCacheService.deleteEaiTenants(eaiEntity.getUid(), deleteTenantIds);
        UnionEAIPO eaiPO = EaiConvertor.convertToPO(eaiBO.getEai());
        eaiPO.setLastUpdateTime(new Date());
        eaiPO.setLastUpdateResult(StateEnum.REGEAI_UPDED.getCode());
        unionEaiMapper.updateById(eaiPO);
        addEaiChild(eaiPO, eaiBO);
    }

    @Override
    public void updateEaiStatus(Long id, StateEnum stateEnum) {
        UnionEAIPO eaiPO = new UnionEAIPO();
        eaiPO.setId(id);
        eaiPO.setLastUpdateTime(new Date());
        eaiPO.setLastUpdateResult(stateEnum.getCode());
        unionEaiMapper.updateById(eaiPO);
    }

    @Transactional("espTransactionManager")
    @Override
    public void deleteEai(EaiEntity eaiBO) {
        List<EaiTenantMappingPO> eaiTenantMappings = eaiTenantMappingMapper.selectList((Wrappers.<EaiTenantMappingPO>lambdaQuery().eq(EaiTenantMappingPO::getEaiUid, eaiBO.getUid())));
        List<String> tenantIds = eaiTenantMappings.stream().map(EaiTenantMappingPO::getTenantId).collect(Collectors.toList());
        productInfoCacheService.deleteEaiInfo(eaiBO.getUid(), tenantIds);
        List<UnionProductPO> prods = deleteEaiChild(eaiBO);
        unionEaiMapper.deleteById(eaiBO.getId());
        if (CollectionUtils.isNotEmpty(prods)) {
            List<String> prodUids = prods.stream().map(UnionProductPO::getUid).collect(Collectors.toList());
            serviceInfoCacheService.deleteUnionService(eaiBO.getUid(), prodUids);
        }
    }

    private List<UnionProductPO> deleteEaiChild(EaiEntity eaiBO) {
        List<UnionProductPO> productPOS = unionProductMapper.selectList(Wrappers.<UnionProductPO>lambdaQuery().eq(UnionProductPO::getEaiId, eaiBO.getId()));
        eaiTenantMappingMapper.delete(Wrappers.<EaiTenantMappingPO>lambdaQuery().eq(EaiTenantMappingPO::getEaiUid, eaiBO.getUid()));
        productPOS.forEach(one -> {
            unionServiceMapper.delete(Wrappers.<UnionServicePO>lambdaQuery().eq(UnionServicePO::getUnionProductId, one.getId()));
        });
        unionProductMapper.delete(Wrappers.<UnionProductPO>lambdaQuery().eq(UnionProductPO::getEaiId, eaiBO.getId()));
        return productPOS;
    }

    private void addEaiChild(UnionEAIPO eaiPO, EaiBO eaiBO) {
        Date now = new Date();
        EaiEntity eaiEntity = eaiBO.getEai();
        List<String> tenants = new ArrayList<>();
        eaiBO.getTenants().forEach(tenantMappingBO -> {
            EaiTenantMappingPO tEaiTenantMapping = new EaiTenantMappingPO();
            tEaiTenantMapping.setEaiUid(eaiEntity.getUid());
            tEaiTenantMapping.setTenantId(tenantMappingBO.getTenantId());
            tEaiTenantMapping.setBuildTime(now);
            eaiTenantMappingMapper.insert(tEaiTenantMapping);
            tenantMappingBO.setId(tEaiTenantMapping.getId());
            tenants.add(tenantMappingBO.getTenantId());
        });
        List<Pair<UnionProductPO, List<UnionServicePO>>> prodServices = new ArrayList<>();
        List<UnionProductPO> prods = new ArrayList<>();
        eaiBO.getProducts().forEach(productBO -> {
            UnionProductPO productPO = ProductConvertor.convertToUnionProductPO(productBO.getProduct());
            productPO.setIsvalid(true);
            productPO.setBuildTime(now);
            productPO.setEaiId(eaiEntity.getId());
            unionProductMapper.insert(productPO);
            productBO.getProduct().setId(productPO.getId());
            prods.add(productPO);
            List<UnionServicePO> srvPOs = new ArrayList<>();
            Set<String> tRegisteredService = new HashSet<>();
            if (CollectionUtils.isNotEmpty(productBO.getServices())) {
                productBO.getServices().forEach(srv -> {
                    UnionServicePO srvPO = ServiceConvertor.convertToUnionSericePO(srv);
                    if(!tRegisteredService.contains(srvPO.getName())) {
                        srvPO.setUnionProductId(productPO.getId());
                        srvPO.setBuildTime(now);
                        unionServiceMapper.insert(srvPO);
                        srv.setId(srvPO.getId());
                        srvPOs.add(srvPO);
                        tRegisteredService.add(srvPO.getName());
                    }
                });
            }
            prodServices.add(Pair.of(productPO, srvPOs));
        });
        productInfoCacheService.addEaiInfo(eaiPO, tenants, prods);
        serviceInfoCacheService.addUnionService(eaiPO.getUid(), prodServices);
    }

    @Override
    public EaiEntity getEaiInCache(String eaiUid) {
        UnionEAIPO tUnionEAIPO = productInfoCacheService.getUnionEai(eaiUid);

        if (tUnionEAIPO != null) {
            return EaiConvertor.convertToBO(tUnionEAIPO);
        } else {
            return null;
        }
    }

    @Override
    public void saveEaiCache(EaiEntity eaiEntity) {
        productInfoCacheService.addEaiInfo(EaiConvertor.convertToPO(eaiEntity));
    }

    @Override
    public void saveEaiCache(EaiBO eaiBO, String tenantId) {
        UnionEAIPO tUnionEAIPO = EaiConvertor.convertToPO(eaiBO.getEai());
        List<String> tTenantIdList = new ArrayList<>();
        tTenantIdList.add(tenantId);
        List<UnionProductPO> tUnionProductPOList = eaiBO.getProducts().stream().map(productBO -> ProductConvertor.convertToUnionProductPO(productBO.getProduct())).collect(Collectors.toList());
        productInfoCacheService.addEaiInfo(tUnionEAIPO, tTenantIdList, tUnionProductPOList);
    }

    @Override
    public EaiEntity getEaiInCacheByTenantId(String tenantId) {
        String tEaiId = productInfoCacheService.getEaiIdByTenantId(tenantId);
        if (tEaiId == null) {
            return null;
        }
        UnionEAIPO tUnionEAIPO = productInfoCacheService.getUnionEai(tEaiId);
        if (tUnionEAIPO != null) {
            return EaiConvertor.convertToBO(tUnionEAIPO);
        } else {
            return null;
        }
    }

    @Override
    public EaiEntity getEaiByTenantId(String tenantId) {
        EaiEntity eaiEntity = getEaiInCacheByTenantId(tenantId);
        if (eaiEntity != null) {
            return eaiEntity;
        } else {
            List<EaiTenantMappingPO> tTenantMappingPOList = eaiTenantMappingMapper.selectList(Wrappers.<EaiTenantMappingPO>lambdaQuery().eq(EaiTenantMappingPO::getTenantId, tenantId));
            if (CollectionUtils.isNotEmpty(tTenantMappingPOList)) {
                UnionEAIPO tUnionEAIPO = unionEaiMapper.selectOne(Wrappers.<UnionEAIPO>lambdaQuery().eq(UnionEAIPO::getUid, tTenantMappingPOList.get(0).getEaiUid()));
                return EaiConvertor.convertToBO(tUnionEAIPO);
            }
        }
        return null;
    }

    @Override
    public List<EaiTenantEntity> queryOtherEaiTenantByTenantIds(List<String> tenantIds, String eaiUid) {
        if (CollectionUtils.isEmpty(tenantIds)) {
            return null;
        }
        List<EaiTenantMappingPO> poList = eaiTenantMappingMapper.selectList(Wrappers.<EaiTenantMappingPO>lambdaQuery()
                .in(EaiTenantMappingPO::getTenantId, tenantIds)
                .ne(StringUtils.isNotBlank(eaiUid), EaiTenantMappingPO::getEaiUid, eaiUid));
        if (CollectionUtils.isEmpty(poList)) {
            return null;
        }
        return poList.stream().map(one -> new EaiTenantEntity(one.getTenantId(), one.getId(), one.getEaiUid())).collect(Collectors.toList());
    }
}
