package com.digiwin.cross.infrastructure.gatewayimpl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.CollectionUtils;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.digiwin.athena.esp.sdk.util.StringUtil;
import com.digiwin.cross.domain.bo.entity.ServiceProdListEntity;
import com.digiwin.cross.domain.bo.entity.ServiceEntity;
import com.digiwin.cross.domain.gateway.IServiceQueryGateway;
import com.digiwin.cross.infrastructure.cache.service.ServiceInfoCacheService;
import com.digiwin.cross.infrastructure.convertor.ServiceConvertor;
import com.digiwin.cross.infrastructure.database.entity.*;
import com.digiwin.cross.infrastructure.database.mapper.ProductMapper;
import com.digiwin.cross.infrastructure.database.mapper.ServiceMapper;
import com.digiwin.cross.infrastructure.database.mapper.ServiceVersionMapper;
import com.digiwin.cross.infrastructure.database.mapper.UnionEaiMapper;
import com.digiwin.cross.infrastructure.database.mapper.UnionProductMapper;
import com.digiwin.cross.infrastructure.database.mapper.UnionServiceMapper;
import com.digiwin.cross.infrastructure.rpc.mapper.AlarmSenderMapper;
import com.digiwin.cross.infrastructure.rpc.req.ServiceDuplicateAlarmReq;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @description:
 * @author: liunansheng
 * @date: 2023/6/9 13:57
 */
@Component
public class ServiceQueryGateway implements IServiceQueryGateway {

    private final ProductMapper productMapper;
    private final ServiceMapper serviceMapper;
    private final ServiceVersionMapper versionMapper;
    private final UnionEaiMapper unionEaiMapper;
    private final UnionProductMapper unionProductMapper;
    private final UnionServiceMapper unionServiceMapper;
    private final AlarmSenderMapper alarmSenderMapper;

    private final ServiceInfoCacheService serviceInfoCacheService;

    public ServiceQueryGateway(ProductMapper productMapper, ServiceMapper serviceMapper, ServiceVersionMapper versionMapper, UnionEaiMapper unionEaiMapper, UnionProductMapper unionProductMapper, UnionServiceMapper unionServiceMapper, ServiceInfoCacheService serviceInfoCacheService, AlarmSenderMapper alarmSenderMapper) {
        this.productMapper = productMapper;
        this.serviceMapper = serviceMapper;
        this.versionMapper = versionMapper;
        this.unionEaiMapper = unionEaiMapper;
        this.unionProductMapper = unionProductMapper;
        this.unionServiceMapper = unionServiceMapper;
        this.serviceInfoCacheService = serviceInfoCacheService;
        this.alarmSenderMapper = alarmSenderMapper;
    }

    @Override
    public ServiceEntity getSrvByName(Long productId, String srvName, boolean needVersion) {
        ServicePO servicePO = serviceMapper.selectOne(Wrappers.<ServicePO>lambdaQuery()
                .eq(ServicePO::getProductId, productId).eq(ServicePO::getServiceName, srvName));
        if (null == servicePO) {
            return null;
        }
        ServiceVersionPO versionPO = null;
        if (needVersion) {
            versionPO = versionMapper.selectOne(Wrappers.<ServiceVersionPO>lambdaQuery().eq(ServiceVersionPO::getSrvId, servicePO.getId()));
        }
        return ServiceConvertor.convertToServiceBO(servicePO, versionPO);
    }

    @Override
    public ServiceEntity findService(String prodUid, String srvName) {
        //cache
        ServicePO tServicePO = serviceInfoCacheService.findService(prodUid, srvName);

        if (tServicePO != null) {
            return ServiceConvertor.convertToServiceBO(tServicePO, null);
        } else {
            //DB
            ProductPO tProductPO = productMapper.selectOne(
                    Wrappers.<ProductPO>lambdaQuery()
                            .eq(ProductPO::getUid, prodUid)
                            .eq(ProductPO::getIsvalid, true)
            );
            ServiceEntity tServiceEntity = getSrvByName(tProductPO.getId(), srvName, false);
            if (tServiceEntity != null) {
                serviceInfoCacheService.addOrUpdateService(prodUid, ServiceConvertor.convertToSericePO(tServiceEntity));
            }
            return tServiceEntity;
        }
    }

    @Override
    public ServiceEntity findUnionService(String eaiUid, String unionProdUid, String srvName) {
        //cache
        UnionServicePO tServicePO = serviceInfoCacheService.findUnionService(eaiUid, unionProdUid, srvName);

        if (tServicePO != null) {
            return ServiceConvertor.convertToServiceBO(tServicePO);
        } else {
            //DB
            UnionEAIPO tUnionEAIPO = unionEaiMapper.selectOne(
                    Wrappers.<UnionEAIPO>lambdaQuery()
                            .eq(UnionEAIPO::getUid, eaiUid)
                            .eq(UnionEAIPO::getIsvalid, true)
            );
            UnionProductPO tUnionProductPO = unionProductMapper.selectOne(
                    Wrappers.<UnionProductPO>lambdaQuery()
                            .eq(UnionProductPO::getUid, unionProdUid)
                            .eq(UnionProductPO::getEaiId, tUnionEAIPO.getId())
                            .eq(UnionProductPO::getIsvalid, true)
            );

            UnionServicePO tUnionServicePO = unionServiceMapper.selectOne(
                    Wrappers.<UnionServicePO>lambdaQuery()
                            .eq(UnionServicePO::getName, srvName)
                            .eq(UnionServicePO::getUnionProductId, tUnionProductPO.getId())
            );

            if (tUnionServicePO != null) {
                serviceInfoCacheService.addUnionService(tUnionEAIPO.getUid(), tUnionProductPO.getUid(), tUnionServicePO);
            }
            return ServiceConvertor.convertToServiceBO(tUnionServicePO);
        }

    }

    @Override
    public List<String> getSrvByProdId(Long prodId) {
        List<ServicePO> servicePOList = serviceMapper.selectList(Wrappers.<ServicePO>lambdaQuery().eq(ServicePO::getProductId, prodId));
        if (CollectionUtils.isEmpty(servicePOList)) {
            return null;
        }
        return servicePOList.stream().map(ServicePO::getServiceName).collect(Collectors.toList());
    }

    @Override
    public Page getUnionServiceLis(Long uid, String name, Page page) {
        LambdaQueryWrapper<UnionServicePO> queryWrapper = Wrappers.lambdaQuery(UnionServicePO.class).eq(UnionServicePO::getUnionProductId, uid);
        if (StringUtil.isNotEmpty(name)) {
            queryWrapper.like(UnionServicePO::getName, name);
        }
        return unionServiceMapper.selectPage(page, queryWrapper);
    }

    @Override
    public List<ServiceProdListEntity> queryServiceWithProductNames(List<String> serviceNameList) {
        List<ServicePO> servicePOList = serviceMapper.selectList(Wrappers.<ServicePO>lambdaQuery().in(ServicePO::getServiceName, serviceNameList));
        if (!serviceNameList.isEmpty()) {
            Set<Long> productIds = servicePOList.stream().map(ServicePO::getProductId).collect(Collectors.toSet());
            List<ProductPO> productPOList = productMapper.selectList(Wrappers.<ProductPO>lambdaQuery().in(ProductPO::getId, productIds));
            Map<Long, ProductPO> productMap = productPOList.stream().collect(Collectors.toMap(ProductPO::getId, Function.identity()));
            Map<String, ServiceProdListEntity> serviceProdListEntityMap = new LinkedHashMap<>();
            servicePOList.forEach(servicePO -> {
                String tServiceName = servicePO.getServiceName();
                ProductPO tProductPO = productMap.get(servicePO.getProductId());
                if (tProductPO.getIsvalid()) {
                    ServiceProdListEntity tServiceProdListEntity = serviceProdListEntityMap.computeIfAbsent(tServiceName, s -> new ServiceProdListEntity(tServiceName));
                    tServiceProdListEntity.getProductNames().add(tProductPO.getName());
                }
            });
            return new ArrayList<>(serviceProdListEntityMap.values());
        } else {
            return null;
        }
    }

    @Override
    public void alarmDuplicateService(List<ServiceProdListEntity> duplicateEntityList, String regProd) {
        ServiceDuplicateAlarmReq tAlarmPO = ServiceConvertor.convertToServiceDuplicateAlarmPO(duplicateEntityList, regProd);
        alarmSenderMapper.sendServiceDuplicateMessageToAlarm(tAlarmPO);
    }
}
