package com.digiwin.athena.atdm.retrieveData;

import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.TimeUtils;
import com.digiwin.athena.atdm.UiBotConstants;
import com.digiwin.athena.atdm.atmc.CommonAtmcService;
import com.digiwin.athena.atdm.retrieveData.dto.RetrieveDataDTO;
import com.digiwin.athena.atdm.retrieveData.po.RetrieveDataDO;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.BulkOperations;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

/**
 * RetrieveDataServiceImpl Description
 *
 * @author majianfu
 * @date 2022/10/13
 * @since
 */
@Service
public class CommonRetrieveDataServiceImpl implements CommonRetrieveDataService {
    private static final String COLLECTION_NAME = "retrieve_data";

    @Autowired(required = false)
    @Qualifier("retrieveDataMongoTemplate")
    private MongoTemplate retrieveDataMongoTemplate;

    @Autowired
    private RetrieveDataConverter retrieveDataConverter;

    @Autowired
    private CommonAtmcService atmcService;

    @Autowired
    private MessageUtils messageUtils;

    private MongoTemplate getRetrieveDataMongoTemplate() {
        if (null != retrieveDataMongoTemplate) {
            return retrieveDataMongoTemplate;
        }
        throw BusinessException.create("创建retrieveDataMongoTemplate失败，请检查数据源配置是否正确~");
    }

    @Override
    public void saveRetrieveData(RetrieveDataDTO retrieveDataDTO) {
        RetrieveDataDO retrieveDataDO = retrieveDataConverter.convert(retrieveDataDTO);
        retrieveDataDO.setDateTime(LocalDateTime.now());
        if (null == retrieveDataDTO.getTerminateTime()) {
            retrieveDataDO.setDateTime(LocalDateTime.now());
        } else {
            retrieveDataDO.setDateTime(retrieveDataDTO.getTerminateTime());
        }

        BulkOperations operations = getRetrieveDataMongoTemplate().bulkOps(BulkOperations.BulkMode.UNORDERED, COLLECTION_NAME);
        operations.insert(retrieveDataDO);
        operations.execute();
    }

    @Override
    public List queryRetrieveDataList(AuthoredUser authoredUser, String tmActivityId, List<String> workItemIds) {
        Criteria criteria = Criteria.where("tenantId").is(authoredUser.getTenantId())
                .and("activityId").in(workItemIds);

        Query query = Query.query(criteria);
        List<RetrieveDataDO> retrieveDataList = getRetrieveDataMongoTemplate().find(query, RetrieveDataDO.class, COLLECTION_NAME);

        // 查出来tmActivityId的名称
        List resultList = new ArrayList();
        if (CollectionUtils.isNotEmpty(retrieveDataList)) {
            // 操作类型
            String operationType = messageUtils.getMessage("shelved_uibot__operation_type_withdraw");
            // 撤回的数据集中都是同一种类型的任务，因此只需要查询第一个撤回任务的任务名称即可
            String firstActivityId = retrieveDataList.get(0).getActivityId();
            String firstBacklogId = retrieveDataList.get(0).getInitiateActivityId();
            Map<String, String> activityIdNameMap = queryActivityNameById(Arrays.asList(firstActivityId), Arrays.asList(firstBacklogId));
            String activityName = null;
            if (MapUtils.isNotEmpty(activityIdNameMap)) {
                activityName = activityIdNameMap.get(firstActivityId);
                activityName = StringUtils.isNotBlank(activityName) ? activityName : activityIdNameMap.get(firstBacklogId);
            }

            for (RetrieveDataDO retrieveDataDO : retrieveDataList) {
                resultList.addAll(buildRetrievePageData(operationType, activityName, retrieveDataDO));
            }
        }
        return resultList;
    }

    private Map<String, String> queryActivityNameById(List<String> workItemIdList, List<String> backlogIdList) {
        if (CollectionUtils.isEmpty(workItemIdList) && CollectionUtils.isEmpty(backlogIdList)) {
            return Collections.emptyMap();
        }

        return atmcService.queryActivityNameById(workItemIdList, backlogIdList);
    }


    private List buildRetrievePageData(String operationType, String activityName, RetrieveDataDO retrieveDataDO) {
        List resultList = new ArrayList();
        String withdrawDateTimeStr = TimeUtils.format(retrieveDataDO.getDateTime(), TimeUtils.DEFAULT_FORMAT1);
        for (Map<String, Object> rowData : retrieveDataDO.getPageData()) {
            // 终止执行人信息
            rowData.put("uibot__userId", retrieveDataDO.getUserId());
            rowData.put("uibot__userName", retrieveDataDO.getUserName());
            //操作原因列数据，（本次需求仅支持终止数据的操作原因展示，所以本次已撤回数据中，操作原因数据暂时赋值为空字符串
            rowData.put("uibot__operationReason", "");
            rowData.put("uibot__operationType", operationType);
            // 终止执行时间
            rowData.put(UiBotConstants.FieldName.UIBOT_DATETIME, retrieveDataDO.getDateTime());

            rowData.put("uibot__initiateActivityId", retrieveDataDO.getActivityId());
            rowData.put("uibot__initiateActivityCode", retrieveDataDO.getInitiateActivityId());

            rowData.put("uibot__initiateActivityName", activityName);
            rowData.put("uibot__link", false);

            // 去掉“已勾选”标记
            rowData.remove(UiBotConstants.UIBOT_FIELDS_CHECKED);

            // 改变__DATA_KEY
            if (null != rowData.get(UiBotConstants.DATA_SOURCE_DATA_KEY)) {
                String dataKey = rowData.get(UiBotConstants.DATA_SOURCE_DATA_KEY).toString();

                // 操作类型;执行人userId;操作时间;任务名称;老dataKey
                String initiateActivityCodeStr = null != rowData.get("uibot__initiateActivityCode") ?
                        rowData.get("uibot__initiateActivityCode").toString() : "NAN";
                String newDataKey = operationType + ";" + retrieveDataDO.getUserId() + ";" + withdrawDateTimeStr + ";" + initiateActivityCodeStr + ";" + dataKey;

                rowData.put(UiBotConstants.DATA_SOURCE_DATA_KEY, newDataKey);
            }

            resultList.add(rowData);
        }
        return resultList;
    }
}
