package com.digiwin.athena.atdm.atmc;

import com.digiwin.athena.appcore.domain.BaseResultDTO;
import com.digiwin.athena.appcore.util.JsonUtils;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.atdm.RemoteProperties;
import com.digiwin.athena.atdm.activity.domain.SubmitActionDTO;
import com.digiwin.athena.atdm.atmc.dto.ActivityData;
import com.digiwin.athena.atdm.atmc.dto.ApiErrorInfoDTO;
import com.digiwin.athena.atdm.datasource.domain.Action;
import com.digiwin.athena.atdm.datasource.domain.ExecuteContext;
import com.digiwin.athena.atdm.datasource.domain.ExecuteResult;
import com.digiwin.athena.common.sdk.manager.util.DwSpringHttpUtil;
import com.google.common.collect.ImmutableMap;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * ATMC 提供相关服务
 */
@Slf4j
@Service
public class CommonAtmcServiceImpl implements CommonAtmcService {

    private static final Logger logger = LoggerFactory.getLogger(CommonAtmcServiceImpl.class);

    @Autowired
    private RemoteProperties envProperties;

    @Autowired
    private RestTemplate restTemplate;

    private String getWebAtmcReqUrl(String path) {
        return UriComponentsBuilder.fromHttpUrl(envProperties.getWebAtmcUri()).path(path).toUriString();
    }

    @Override
    public void emptyActivityData(Map data) {
        String url = getWebAtmcReqUrl(UrlPathConstants.ATMC_DELETE_ACTIVITY_DATA);
        HttpHeaders headers = new HttpHeaders();
        HttpEntity httpEntity = new HttpEntity<>(data, headers);
        try {
            restTemplate.postForEntity(url, httpEntity, String.class);
        } catch (Exception e) {
            log.error("body: {}, error: ", JsonUtils.objectToString(data), e);
            throw e;
        }
    }

    @Override
    public void resetActivityData(Map data) {
        String url = getWebAtmcReqUrl(UrlPathConstants.ATMC_RESET_ACTIVITY_DATA);
        HttpHeaders headers = new HttpHeaders();
        HttpEntity httpEntity = new HttpEntity<>(data, headers);
        try {
            restTemplate.postForEntity(url, httpEntity, String.class);
        } catch (Exception e) {
            log.error("body: {}, error: ", JsonUtils.objectToString(data), e);
            throw e;
        }
    }

    /**
     * 跟踪数据的状态
     *
     * @param backlogId
     * @param data
     */
    @Override
    public void setBpmActivityTraceData(String backlogId, String startDataState, String targetDataState, Map data) {
        String url = getWebAtmcReqUrl(String.format(UrlPathConstants.ATMC_RESET_ACTIVITY_TRACE_DATA, backlogId, startDataState, targetDataState));
        HttpHeaders headers = new HttpHeaders();
        HttpEntity httpEntity = new HttpEntity<>(data, headers);
        try {
            restTemplate.postForEntity(url, httpEntity, String.class);
        } catch (Exception e) {
            log.error("body: {}, error: ", JsonUtils.objectToString(data), e);
            throw e;
        }
    }

    @Override
    public void setBpmActivityStateData(String backlogId, Map data) {
        String url = getWebAtmcReqUrl(String.format(UrlPathConstants.ATMC_RESET_ACTIVITY_STATE_DATA, backlogId));
        HttpHeaders headers = new HttpHeaders();
        HttpEntity httpEntity = new HttpEntity<>(data, headers);
        try {
            restTemplate.postForEntity(url, httpEntity, String.class);
        } catch (Exception e) {
            log.error("url: {}, body: {}, error: ", url, JsonUtils.objectToString(data), e);
            throw e;
        }
    }

    /**
     * 撤回关卡
     *
     * @param processSerialNumber
     * @param workitemId
     */
    public void abortTask(Long originalBacklogId, String processSerialNumber, String workitemId) {
        String url = getWebAtmcReqUrl("api/atmc/v1/activity/abortTask");
        Map<String, Object> data = new HashMap<>();
        data.put("originalBacklogId", originalBacklogId);
        data.put("processSerialNumber", processSerialNumber);
        data.put("workitemId", workitemId);

        HttpHeaders headers = new HttpHeaders();
        HttpEntity httpEntity = new HttpEntity<>(data, headers);
        try {
            ResponseEntity<BaseResultDTO<String>> respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<String>>() {
            });
            respEntity.getBody().getResponseWithException("");
        } catch (Exception ex) {
            log.error("body: {}, error: ", JsonUtils.objectToString(data), ex);
            throw ex;
        }
    }

    /**
     * 关闭指定任务
     *
     * @param backlogId
     */
    public void closeBpmActivityByBacklog(Long backlogId) {
        String url = getWebAtmcReqUrl("api/atmc/v1/activity/closeBpmActivityByBacklog");
        Map<String, Object> data = new HashMap<>();
        data.put("backlogId", backlogId);

        HttpHeaders headers = new HttpHeaders();
        HttpEntity httpEntity = new HttpEntity<>(data, headers);
        try {
            ResponseEntity<BaseResultDTO<String>> respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<String>>() {
            });
            respEntity.getBody().getResponseWithException("");
        } catch (Exception ex) {
            log.error("body: {}, error: ", JsonUtils.objectToString(data), ex);
            throw ex;
        }
    }

    /**
     * 更新待办的签核状态
     *
     * @param data
     */
    public void updateBacklogApprovalState(Map<String, Object> data) {
        String url = getWebAtmcReqUrl("api/atmc/v1/backlog/updateApprovalState");

        HttpHeaders headers = new HttpHeaders();
        HttpEntity httpEntity = new HttpEntity<>(data, headers);
        try {
            ResponseEntity<BaseResultDTO<String>> respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<String>>() {
            });
            respEntity.getBody().getResponseWithException("");
        } catch (Exception ex) {
            log.error("body: {}, error: ", JsonUtils.objectToString(data), ex);
            throw ex;
        }
    }

    public List<Map> getChangeTaskDataSource(Map param) {
        String url = getWebAtmcReqUrl("api/atmc/v1/datasource/changetask");
        HttpHeaders headers = new HttpHeaders();
        HttpEntity httpEntity = new HttpEntity<>(param, headers);
        try {
            ResponseEntity<BaseResultDTO<List<Map>>> respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<Map>>>() {
            });
            return respEntity.getBody().getResponseWithException("");
        } catch (Exception e) {
            log.error("body: {}, error: ", JsonUtils.objectToString(param), e);
            throw e;
        }
    }

    public List<Map> getOverdueTaskDataSource(Map param) {
        String url = getWebAtmcReqUrl("api/atmc/v1/datasource/overdueTask");
        HttpHeaders headers = new HttpHeaders();
        HttpEntity httpEntity = new HttpEntity<>(param, headers);
        try {
            ResponseEntity<BaseResultDTO<List<Map>>> respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<Map>>>() {
            });
            return respEntity.getBody().getResponse();
        } catch (Exception e) {
            log.error("body: {}, error: ", JsonUtils.objectToString(param), e);
            throw e;
        }
    }

    public List<Map> saveOverdueTaskDataSource(Map param) {
        String url = getWebAtmcReqUrl("api/atmc/v1/datasource/overdueTask/save");
        HttpHeaders headers = new HttpHeaders();
        HttpEntity httpEntity = new HttpEntity<>(param, headers);
        try {
            ResponseEntity<BaseResultDTO<List<Map>>> respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<Map>>>() {
            });
            return respEntity.getBody().getResponseWithException("");
        } catch (Exception e) {
            log.error("body: {}, error: ", JsonUtils.objectToString(param), e);
            throw e;
        }
    }

    public List<Map> getOverdueTaskListDataSource(Map param) {
        String url = getWebAtmcReqUrl("api/atmc/v1/datasource/overdueTask/traceList");
        HttpHeaders headers = new HttpHeaders();
        HttpEntity httpEntity = new HttpEntity<>(param, headers);
        try {
            ResponseEntity<BaseResultDTO<List<Map>>> respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<Map>>>() {
            });
            return respEntity.getBody().getResponseWithException("");
        } catch (Exception e) {
            log.error("body: {}, error: ", JsonUtils.objectToString(param), e);
            throw e;
        }
    }

    @Override
    public void sendEmailToHandleApiExecuteError(String type, Object emailData) {
        String url = getWebAtmcReqUrl("/api/atmc/v1/apiExecuteError/sendEmail");
        HttpHeaders headers = new HttpHeaders();
        HttpEntity httpEntity = new HttpEntity<>(emailData, headers);

        String finalUrl = url + "?type=" + type;
        try {
            ResponseEntity<BaseResultDTO> respEntity = restTemplate.exchange(finalUrl, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO>() {
            });
            respEntity.getBody().getResponseWithException("");
        } catch (Exception ex) {
            log.error("url: {}, body: {}, error: ", finalUrl, JsonUtils.objectToString(emailData), ex);
            throw ex;
        }
    }

    @Override
    public List<Map> getConfig(List<String> configName) {
        String url = getWebAtmcReqUrl("/api/atmc/v1/config/query");
        HttpHeaders headers = new HttpHeaders();
        HttpEntity httpEntity = new HttpEntity<>(null, headers);

        String finalUrl = UriComponentsBuilder.fromHttpUrl(url).queryParam("name", configName.toArray(new String[]{})).toUriString();
        try {
            ResponseEntity<BaseResultDTO<List<Map>>> respEntity = restTemplate.exchange(finalUrl, HttpMethod.GET, httpEntity, new ParameterizedTypeReference<BaseResultDTO<List<Map>>>() {
            });
            return respEntity.getBody().getResponseWithException("");
        } catch (Exception ex) {
            log.error("url: {}, error: ", finalUrl, ex);
            throw ex;
        }
    }

    @Override
    public void closeApiExecuteErrorWorkItem(String processSerialNumber) {
        String url = getWebAtmcReqUrl("/api/atmc/v1/apiExecuteError/closeBacklog");
        Map<String, Object> data = new HashMap<>();
        data.put("keyId", processSerialNumber);

        HttpHeaders headers = new HttpHeaders();
        HttpEntity httpEntity = new HttpEntity<>(data, headers);
        try {
            ResponseEntity<BaseResultDTO<String>> respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<String>>() {
            });
            respEntity.getBody().getResponseWithException("");
        } catch (Exception ex) {
            log.error("body: {}, error: ", JsonUtils.objectToString(data), ex);
            throw ex;
        }
    }

    /**
     * {@inheritDoc}
     *
     * @return
     */
    @Override
    public BaseResultDTO manualTaskStateChanged(List<Map<String, Object>> dataList) {
        String url = getWebAtmcReqUrl("api/atmc/v1/activity/changed/manualTask");

        HttpHeaders headers = new HttpHeaders();
        HttpEntity reqEntity = new HttpEntity<>(dataList, headers);
        try {
            ResponseEntity<BaseResultDTO> respEntity = this.restTemplate.exchange(url, HttpMethod.POST, reqEntity, new ParameterizedTypeReference<BaseResultDTO>() {
            });
            respEntity.getBody().getResponseWithException("");
            return respEntity.getBody();
        } catch (Exception ex) {
            log.error("body: {}, error: ", JsonUtils.objectToString(dataList), ex);
            throw ex;
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List<Map> queryCardListByBk(String type, Long backlogId, List<Map> bkList, boolean forceNeedEntityName) {
        String url = getWebAtmcReqUrl("api/atmc/v1/dataUniformity/queryCardListByBk");

        Map<String, Object> body = new HashMap<>();
        body.put("type", type);
        body.put("cardId", backlogId);
        body.put("forceNeedEntityName", forceNeedEntityName);
        body.put("bkList", bkList);

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        HttpEntity reqEntity = new HttpEntity<>(body, headers);
        try {
            ResponseEntity<BaseResultDTO<List<Map>>> respEntity = this.restTemplate.exchange(url, HttpMethod.POST, reqEntity, new ParameterizedTypeReference<BaseResultDTO<List<Map>>>() {
            });
            return respEntity.getBody().getResponseWithException("");
        } catch (Exception ex) {
            log.error("从ATMC获取BK信息报错：body: {}, error: ", JsonUtils.objectToString(body), ex);
            throw ex;
        }
    }

    @Override
    public List<Object> executeDataSourceDataQuery(ExecuteContext executeContext, Action action, Map<String, Object> parameter) {
        String url = getWebAtmcReqUrl("api/atmc/v1/datasource/execute/query/data");

        Map body = new HashMap();
        body.put("executeContext", executeContext);
        body.put("action", action);
        body.put("parameter", parameter);

        HttpHeaders headers = new HttpHeaders();
        HttpEntity reqEntity = new HttpEntity<>(body, headers);
        try {
            ResponseEntity<BaseResultDTO<List<Object>>> respEntity = this.restTemplate.exchange(url, HttpMethod.POST, reqEntity, new ParameterizedTypeReference<BaseResultDTO<List<Object>>>() {
            });
            return respEntity.getBody().getResponseWithException("");
        } catch (Exception ex) {
            log.error("body: {}, error: ", JsonUtils.objectToString(body), ex);
            throw ex;
        }
    }

    /**
     * 获取PTM-Data，专用于PTM的 submit-data Action
     *
     * @param workItemId
     * @return
     */
    @Override
    public Map getPtmData(Long workItemId) {
        String url = getWebAtmcReqUrl("api/atmc/v1/action/ptmData/" + workItemId.toString());

        HttpHeaders headers = new HttpHeaders();
        HttpEntity reqEntity = new HttpEntity<>(null, headers);
        try {
            ResponseEntity<BaseResultDTO<Map>> respEntity = this.restTemplate.exchange(url, HttpMethod.GET, reqEntity, new ParameterizedTypeReference<BaseResultDTO<Map>>() {
            });
            return respEntity.getBody().getResponseWithException("");
        } catch (Exception ex) {
            log.error("url: {}, error: ", url, ex);
            throw ex;
        }
    }

    @Override
    public Map getBpmWorkItemById(Long workitemId) {
        String url = getWebAtmcReqUrl("api/atmc/v1/backlog/getWorkitemById/" + workitemId);
        HttpHeaders headers = new HttpHeaders();
        HttpEntity reqEntity = new HttpEntity<>(null, headers);
        try {
            ResponseEntity<BaseResultDTO<Map>> respEntity = this.restTemplate.exchange(url, HttpMethod.GET, reqEntity, new ParameterizedTypeReference<BaseResultDTO<Map>>() {
            });
            return respEntity.getBody().getResponseWithException("");
        } catch (Exception ex) {
            log.error("url: {}, error: ", url, ex);
            throw ex;
        }
    }

    @Override
    public void closeDataUniformityBacklog(Long workitemId, String actionId, String dataFrom) {
        Map<String, Object> paramMap = new HashMap<>();
        paramMap.put("id", workitemId);
        paramMap.put("actionId", actionId);
        paramMap.put("dataFrom", dataFrom);

        String url = getWebAtmcReqUrl("/api/atmc/v1/backlog/closeDataUniformityBacklog");
        HttpHeaders headers = new HttpHeaders();
        HttpEntity reqEntity = new HttpEntity<>(paramMap, headers);
        try {
            ResponseEntity<BaseResultDTO<String>> respEntity = restTemplate.exchange(url, HttpMethod.POST, reqEntity, new ParameterizedTypeReference<BaseResultDTO<String>>() {
            });
            respEntity.getBody().getResponseWithException("");
        } catch (Exception ex) {
            log.error("body: {}, error: ", JsonUtils.objectToString(paramMap), ex);
            throw ex;
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Map<String, String> queryActivityNameById(List<String> workItemIdList, List<String> backlogIdList) {
        String url = getWebAtmcReqUrl("/api/atmc/v1/backlog/queryNameById");

        Map reqBody = new HashMap();
        reqBody.put("workItemIdList", workItemIdList);
        reqBody.put("backlogIdList", backlogIdList);

        HttpHeaders headers = new HttpHeaders();
        HttpEntity reqEntity = new HttpEntity<>(reqBody, headers);

        try {
            ResponseEntity<BaseResultDTO<Map<String, String>>> respEntity = restTemplate.exchange(url, HttpMethod.POST, reqEntity, new ParameterizedTypeReference<BaseResultDTO<Map<String, String>>>() {
            });
            return respEntity.getBody().getResponseWithException("");
        } catch (Exception ex) {
            log.error("body: {}, error: ", JsonUtils.objectToString(reqBody), ex);
            throw ex;
        }
    }

    /**
     * 向运维模组发送api错误数据
     *
     * @param apiErrorInfoDTO
     */
    @Override
    public void sendApiErrorInfoMq(ApiErrorInfoDTO apiErrorInfoDTO) {
        String url = getWebAtmcReqUrl("/api/atmc/v1/devops/sendApiErrorInfoMq");
        HttpHeaders headers = new HttpHeaders();
        HttpEntity reqEntity = new HttpEntity<>(apiErrorInfoDTO, headers);
        try {
            ResponseEntity<BaseResultDTO<Map<String, String>>> respEntity = restTemplate.exchange(url, HttpMethod.POST, reqEntity, new ParameterizedTypeReference<BaseResultDTO<Map<String, String>>>() {
            });
            respEntity.getBody().getResponseWithException("");
        } catch (Exception ex) {
            log.error("body: {}, error: ", JsonUtils.objectToString(apiErrorInfoDTO), ex);
            throw ex;
        }
    }

    @Override
    public Boolean operationConfig(String name, Integer operation, String configName) {
        String url = getWebAtmcReqUrl("/api/atmc/v1/config/operationConfig");
        HttpHeaders headers = new HttpHeaders();
        Map<String, Object> params = new HashMap<>();
        params.put("name", name);
        params.put("operation", operation);
        params.put("configName", configName);
        HttpEntity httpEntity = new HttpEntity<>(params, headers);

        try {
            ResponseEntity<BaseResultDTO<Boolean>> respEntity = restTemplate.exchange(url, HttpMethod.POST, httpEntity, new ParameterizedTypeReference<BaseResultDTO<Boolean>>() {
            });
            return respEntity.getBody().getResponseWithException("");
        } catch (Exception ex) {
            log.error("url: {}, error: ", url, ex);
            throw ex;
        }
    }

    private static final String ATMC_LOCAL = "locale";

    @Override
    public ActivityData getTaskData(String backlogId, ExecuteContext pageContext) {
        String url = getWebAtmcReqUrl(UrlPathConstants.ATMC_TASK_DATA_URL + "/" + backlogId) + "?isHistory=" + pageContext.getIsHistory();

        HttpHeaders headers = new HttpHeaders();
        headers.add(ATMC_LOCAL, pageContext.getLocale());
        HttpEntity<String> requestEntity = new HttpEntity<>(null, headers);

        try {
            ResponseEntity<BaseResultDTO<ActivityData>> activityDataResultResponse = restTemplate.exchange(url, HttpMethod.GET, requestEntity, new ParameterizedTypeReference<BaseResultDTO<ActivityData>>() {
            });
            return activityDataResultResponse.getBody().getResponseWithException("");
        } catch (Exception ex) {
            log.error("url: {}, error: ", url, ex);
            throw ex;
        }
    }


    /**
     * 获取任务信息
     *
     * @param backlogId
     * @param isHistory
     * @return
     */
    @Override
    public Map getTaskData(String backlogId, Boolean isHistory) {
        String url = getWebAtmcReqUrl(UrlPathConstants.ATMC_TASK_DATA_URL + "/" + backlogId) + "?isHistory=" + isHistory;

        HttpHeaders headers = new HttpHeaders();
        HttpEntity<String> requestEntity = new HttpEntity<>(null, headers);

        try {
            ResponseEntity<BaseResultDTO<Map>> activityDataResultResponse = restTemplate.exchange(url, HttpMethod.GET, requestEntity, new ParameterizedTypeReference<BaseResultDTO<Map>>() {
            });
            return activityDataResultResponse.getBody().getResponseWithException("");
        } catch (Exception ex) {
            log.error("url: {}, error: ", url, ex);
            throw ex;
        }
    }


    @Override
    public void signNotice(Integer type, Long backlogId, List<Map<String, Object>> users) {
        String url = getWebAtmcReqUrl("api/atmc/v1/backlog/signNotice");
        Map<String, Object> paramerMap = ImmutableMap.of("type", type, "backlogId", backlogId, "users", users);
        HttpHeaders headers = new HttpHeaders();
        HttpEntity reqEntity = new HttpEntity<>(paramerMap, headers);
        try {
            ResponseEntity<BaseResultDTO> respEntity = restTemplate.exchange(url, HttpMethod.POST, reqEntity, new ParameterizedTypeReference<BaseResultDTO>() {
            });
            respEntity.getBody().getResponseWithException("");
        } catch (Exception ex) {
            log.error("body: {}, error: ", JsonUtils.objectToString(paramerMap), ex);
            throw ex;
        }
    }

    private static final ParameterizedTypeReference<BaseResultDTO<ExecuteResult>> executeActionResultType = new ParameterizedTypeReference<BaseResultDTO<ExecuteResult>>() {
    };

    @Override
    public ExecuteResult executeCalendarAction(SubmitActionDTO submitAction) {
        String url = getWebAtmcReqUrl("/api/atmc/v1/action/calendar/submit");
        return DwSpringHttpUtil.post(url, submitAction, executeActionResultType).getResponse();
    }

    @Override
    public ExecuteResult executeAction(SubmitActionDTO submitAction) {
        String url = getWebAtmcReqUrl("/api/atmc/v1/action/submit");
        return DwSpringHttpUtil.post(url, submitAction, executeActionResultType).getResponse();
    }
}
