package com.digiwin.athena.atdm.datasource.domain;

import com.digiwin.athena.appcore.util.DataUtils;
import com.digiwin.athena.atdm.UiBotConstants;
import com.digiwin.athena.atdm.datasource.dto.PageInfo;
import com.fasterxml.jackson.annotation.JsonIgnore;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


/**
 * 每个数据源的查询结果，主要用来存储数据名，数据，元数据，数据索引
 */
public class QueryResult {
    /**
     * 数据节点名称
     */
    private String dataSourceName;

    /**
     * 数据的key字段
     */
    private List<String> dataKeys;

    public List<String> getDataKeys() {
        return dataKeys;
    }

    public void setDataKeys(List<String> dataKeys) {
        this.dataKeys = dataKeys;
    }

    /**
     * 分页属性:总数
     */
    private Integer totalResults;
    public Integer getTotalResults() {
        return totalResults;
    }
    public void setTotalResults(Integer totalResults) {
        this.totalResults = totalResults;
        if (totalResults == null) {
            this.hasNext = false;
            this.pageCount = 0;
        }else {
            this.pageCount = (totalResults + pageSize - 1) / pageSize;
        }
    }

    /**
     * 分页属性:是否有下一页
     */
    private boolean hasNext;
    public boolean isHasNext() {
        return hasNext;
    }
    public void setHasNext(boolean hasNext) {
        this.hasNext = hasNext;
    }

    /**
     * 分页属性:每一页的大小，默认是50
     */
    private Integer pageSize = 50;

    public Integer getPageSize() {
        return pageSize;
    }

    public void setPageSize(Integer pageSize) {
        this.pageSize = pageSize;
    }
    /**
     * 第几页，默认是1
     */
    private Integer pageNo = 1;

    public Integer getPageNo() {
        return pageNo;
    }

    public void setPageNo(Integer pageNo) {
        this.pageNo = pageNo;
    }

    /**
     * 分页属性:总共多少页
     */
    private Integer pageCount ;

    public Integer getPageCount() {
        return pageCount;
    }

    public void setPageCount(Integer pageCount) {
        this.pageCount = pageCount;
    }

    /**
     * 是否为集合数据
     */
    private boolean single;

    public boolean isSingle() {
        return single;
    }

    public void setSingle(boolean single) {
        this.single = single;
    }

    /**
     * 查询的笔数
     */
    private Integer limit;

    public Integer getLimit() {
        return limit;
    }

    public void setLimit(Integer limit) {
        this.limit = limit;
        if(CollectionUtils.isNotEmpty(this.data) && this.data.size() >limit){
            List<Map<String, Object>> newData = new ArrayList<>();
            for (int i = 0; i < limit; i++) {
                newData.add(this.data.get(i));
            }
            this.data = newData;
        }
    }

    /**
     * 数据结果集合
     */
    private List<Map<String, Object>> data;

    private ApiMetadataCollection apiMetadataCollection;
    private List<ApiMetadataCollection> apiMetadataCollectionList;
    private Map<String,Integer> dataKeyIndex;

    public void removeTop(Integer size){
        if (CollectionUtils.isNotEmpty(data) && data.size() > size){
            for ( int i = 0; i < size; i++) {
                data.remove( 0);
            }
            initializeDataKey();
        }
    }

    public static QueryResult empty(){
        QueryResult queryResult=new QueryResult();
        queryResult.setData(new ArrayList<>());
        queryResult.setDataKeyIndex(new HashMap<>());
        return queryResult;
    }

    public static QueryResult empty(String dataSourceName){
        QueryResult queryResult=new QueryResult();
        queryResult.setDataSourceName(dataSourceName);

        queryResult.setData(new ArrayList<>());
        queryResult.setDataKeyIndex(new HashMap<>());
        return queryResult;
    }

    public static QueryResult withSingleData(String dataSourceName,Map<String, Object> data){

        ArrayList  dataList =  new ArrayList<>();
        if(MapUtils.isNotEmpty(data) && data.size() > 0) {
            dataList.add(data);
        }
        QueryResult queryResult= withData(dataSourceName,dataList);
        queryResult.setSingle(true);
        return queryResult;
    }
//
//    public static QueryResult withPageInfoData(String dataSourceName,Object key,Object data, QueryResult queryResult){
//        if(queryResult == null){
//            queryResult =  new QueryResult();
//            queryResult.setDataSourceName(dataSourceName);
//        }
//        if("has_next".equals(key)){
//            if( data != null) {
//                queryResult.setHasNext(Boolean.valueOf(data.toString()));
//            }
//        }
//        if("total_results".equals(key)){
//            if( data != null) {
//                queryResult.setTotalResults( Integer.getInteger(String.valueOf(data)));
//            }
//        }
//        return queryResult;
//    }

//    public static QueryResult withPageInfoData(String dataSourceName,List<Map<String, Object>> dataList, QueryResult queryResult){
//        if(queryResult == null){
//            queryResult =  new QueryResult();
//            queryResult.setDataSourceName(dataSourceName);
//        }
//
//        queryResult.setData(dataList);
//        return queryResult;
//    }


    public static QueryResult withData(String dataSourceName,Map<String, List<Object>> data){
        List<Map<String, Object>> dataList=  DataUtils.getMapFirstValueToList(data);
        if ( StringUtils.isEmpty(dataSourceName) && data.size() >0){
            dataSourceName =(String) data.keySet().toArray()[0];
        }
        QueryResult queryResult= withData(dataSourceName,dataList);
        return queryResult;
    }

    public static QueryResult withData(String dataSourceName, Map<String, Object> data, PageInfo pageInfo){
        QueryResult queryResult=null;


        for (Map.Entry<String, Object> stringObjectEntry : data.entrySet()) {
            Object val = stringObjectEntry.getValue();
            if (val == null) {
                continue;
            }
            //获取数据源对应的API的值
            if (StringUtils.isEmpty(dataSourceName)) {
                dataSourceName = stringObjectEntry.getKey();
            }
            //多M结构，以数据源名称查询指定返回数组
            Object dataSourceVal = data.get(dataSourceName);
            if(data.size() > 1 && hasSourceVal(dataSourceVal)){
                val = dataSourceVal;
            }
            if (val instanceof Collection) {
                List<Map<String, Object>> value = (List<Map<String, Object>>) val;
                queryResult = QueryResult.withData(dataSourceName, value);
                if (CollectionUtils.isNotEmpty(value)) {
                    break;
                }
            } else if (val instanceof Map) {
                queryResult = QueryResult.withSingleData(dataSourceName, (Map<String, Object>) val);
                break;
            }
        }

        if (queryResult == null){
            queryResult = QueryResult.empty(dataSourceName);
        }
        if (pageInfo != null && pageInfo.getPageNo()!= null && pageInfo.getPageSize()!= null){
            queryResult.setPageSize(pageInfo.getPageSize());
            queryResult.setPageNo(pageInfo.getPageNo());
        }
        return queryResult;
    }

    public static boolean hasSourceVal(Object val){
        if(val == null){
            return false;
        }
        if (val instanceof Collection) {
            List<Map<String, Object>> value = (List<Map<String, Object>>) val;
            if (CollectionUtils.isNotEmpty(value)) {
                return true;
            }
        } else if (val instanceof Map) {
            Map<String,Object> value = (Map<String, Object>) val;
            if(MapUtils.isNotEmpty(value)){
                return true;
            }
            return true;
        }
        return false;
    }




    public static QueryResult withData(String dataSourceName,List<Map<String, Object>> dataList){
        QueryResult queryResult=new QueryResult();
        queryResult.setDataSourceName(dataSourceName);
        queryResult.setData(dataList);
        return queryResult;
    }

    public QueryResult withMetaData(ApiMetadataCollection apiMetadataCollection){
        this.setApiMetadataCollection(apiMetadataCollection);
        return this;
    }

    public QueryResult withDataIndex(Map<String,Integer> dataKeyIndex){
        this.setDataKeyIndex ( dataKeyIndex);
        return this;
    }

    public QueryResult appendData(QueryResult queryResult){
        int oldSize = this.data.size();
        this.getData().addAll(queryResult.getData());
        this.totalResults = this.calcResults(queryResult);
        Map<String,Integer> otherIndex = queryResult.getDataKeyIndex();
        otherIndex.forEach(((key, integer) -> {
            if (!this.dataKeyIndex.containsKey(key)){
                this.dataKeyIndex.put(key,integer+oldSize);
            }
        }));
        return this;
    }

    private Integer calcResults(QueryResult queryResult) {
        BigDecimal thisValue = new BigDecimal(0);
        BigDecimal addValue = new BigDecimal(0);
        if(this.totalResults != null){
            thisValue = new BigDecimal(this.totalResults);
        }
        if(queryResult.getTotalResults() != null){
            addValue = new BigDecimal(queryResult.getTotalResults());
        }
        return Integer.valueOf(thisValue.add(addValue).toBigInteger().toString());
    }

    public QueryResult resetData(List<Map<String, Object>> dataList){
        this.data.clear();
        if(this.dataKeyIndex != null){
            this.dataKeyIndex.clear();
        }
        this.data = dataList;
        this.initializeDataKey();
        return this;
    }

    @JsonIgnore
    public int size(){
        if (CollectionUtils.isEmpty( this.getData())){
            return 0;
        }
        return this.getData().size();
    }
    @JsonIgnore
    public int keySize(){
        if (MapUtils.isEmpty( this.getDataKeyIndex())){
            return 0;
        }
        return  this.getDataKeyIndex().size();
    }

    public List<Map<String, Object>> getData() {
        return data;
    }

    public void setData(List<Map<String, Object>> data) {
        this.data = data;
    }

    public ApiMetadataCollection getApiMetadataCollection() {
        return apiMetadataCollection;
    }

    private void setApiMetadataCollection(ApiMetadataCollection apiMetadataCollection) {
        this.apiMetadataCollection = apiMetadataCollection;
    }

    public List<ApiMetadataCollection> getApiMetadataCollectionList() {
        return apiMetadataCollectionList;
    }

    private void setApiMetadataCollectionList(List<ApiMetadataCollection> apiMetadataCollectionList) {
        this.apiMetadataCollectionList = apiMetadataCollectionList;
    }

    /**
     * 返回数据节点名称
     */
    public String getDataSourceName() {
        if (dataSourceName == null) {
            if (this.apiMetadataCollection != null
                    && this.apiMetadataCollection.getMasterApiMetadata() != null
                    && CollectionUtils.isNotEmpty(this.apiMetadataCollection.getMasterApiMetadata().getResponseFields())) {
                dataSourceName = this.apiMetadataCollection.getMasterApiMetadata().getResponseFields().get(0).getName();
            } else {
                dataSourceName = "data";
            }
        }
        return dataSourceName;
    }

    /**
     * 设置数据节点名称
     */
    public void setDataSourceName(String dataSourceName) {
        this.dataSourceName = dataSourceName;
    }

    public Map<String, Integer> getDataKeyIndex() {
        return dataKeyIndex;
    }

    private void setDataKeyIndex(Map<String, Integer> dataKeyIndex) {
        this.dataKeyIndex = dataKeyIndex;
    }
    @JsonIgnore
    public Map<String, Object> findByKey(String key){
        if (this.dataKeyIndex == null){
            return null;
        }
        if (StringUtils.isEmpty(key)){
            return null;
        }

        if (this.dataKeyIndex.containsKey(key)){
           Integer index=  this.dataKeyIndex.get(key);
           return this.data.get(index);
        }else {
            return null;
        }
    }
    @JsonIgnore
    public Map<String, List<Map<String, Object>>> getPageData(){
        Map pageData = new HashMap<>();
        if(this.single && CollectionUtils.isNotEmpty(this.getData())){
            pageData.put(this.getDataSourceName(),this.getData().get(0));
        }else{
            pageData.put(this.getDataSourceName(),this.getData());
        }
        return pageData;
    }


    /**
     * 根据数据集合，先算出key，并缓存到数据上
     */
    public void initializeDataKey() {
        Map<String,Integer> dataKeyIndex =new HashMap<>();

        if (CollectionUtils.isEmpty(this.getData())
                || CollectionUtils.isEmpty(this.dataKeys)) {
            this.withDataIndex(dataKeyIndex);
            return;
        }

        int index=0;
        for (Map<String, Object> datum : this.getData()) {
            StringBuilder stringBuilder = new StringBuilder();
            for (String key : dataKeys) {
                if (datum.containsKey(key)) {
                    stringBuilder.append( datum.get(key)==null ? "NAN" :datum.get(key).toString()).append(";");
                }else{
                    stringBuilder.append("NAN;");
                }
            }
            if (stringBuilder.length() >0) {
                String key = stringBuilder.toString();
                datum.put(UiBotConstants.DATA_SOURCE_DATA_KEY, key);
                //todo:如果存在相同的key应该怎么处理？暂时覆盖
                dataKeyIndex.put(key, index);
            }
            index++;

        }
        this.withDataIndex(dataKeyIndex);
    }

    /**
     * 异动数据key个数
     */
    private Integer abnormalDataKeyIndexSize;
    /**
     * 异动数据个数
     */
    private Integer abnormalDataSize;

    public Integer getAbnormalDataKeyIndexSize() {
        if(null == this.abnormalDataKeyIndexSize){
            return 0;
        }
        return abnormalDataKeyIndexSize;
    }

    public void setAbnormalDataKeyIndexSize(Integer abnormalDataKeyIndexSize) {
        this.abnormalDataKeyIndexSize = abnormalDataKeyIndexSize;
    }

    public Integer getAbnormalDataSize() {
        if(null == this.abnormalDataSize){
            return 0;
        }
        return abnormalDataSize;
    }

    public void setAbnormalDataSize(Integer abnormalDataSize) {
        this.abnormalDataSize = abnormalDataSize;
    }

    /**
     * 单身/子单身 分页信息
     * [{
     *      // 单身分页配置
     *      "detail_path": "demo_main_info",
     *      "detail_name": "demo_detail_order_info",
     *      "use_has_next": true,
     *      "page_size": 50,
     *      "page_no": 1
     * }]
     */
    private Collection<Map<String, Object>> returnPageInfo;

    public Collection<Map<String, Object>> getReturnPageInfo() {
        return returnPageInfo;
    }

    public void setReturnPageInfo(Collection<Map<String, Object>> returnPageInfo) {
        this.returnPageInfo = returnPageInfo;
    }

    /**
     * 合计 固化的合计对象key
     */
    private Map<String,Object> returnSummaryFields;

    public Map<String, Object> getReturnSummaryFields() {
        return returnSummaryFields;
    }

    public void setReturnSummaryFields(Map<String, Object> returnSummaryFields) {
        this.returnSummaryFields = returnSummaryFields;
    }

    private Map<String, Object> returnRequestData;

    public Map<String, Object> getReturnRequestData() {
        return returnRequestData;
    }

    public void setReturnRequestData(Map<String, Object> returnRequestData) {
        this.returnRequestData = returnRequestData;
    }

    /**
     *  保密字段信息
     */
    public List<String> desensitizationFields;

    public List<String> getDesensitizationFields() {
        return desensitizationFields;
    }

    public void setDesensitizationFields(List<String> desensitizationFields) {
        this.desensitizationFields = desensitizationFields;
    }
}
