package com.digiwin.athena.atdm;

import com.digiwin.athena.appcore.AppCoreProperties;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import java.lang.management.ManagementFactory;
import java.lang.management.RuntimeMXBean;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.time.LocalDateTime;
import java.time.ZoneOffset;

/**
 * 配置信息
 *
 * @author chenxsa
 */
@Component
@Data
@Slf4j
public class RemoteProperties {

    @Value("${server.port}")
    private String port;

    @Autowired
    private AppCoreProperties appCoreProperties;

    @Value("${web.uri:}")
    private String webUri;

    @Value("${uibot.uri:}")
    private String uibotUri;

    @Value("${idrive-business-report.uri:}")
    private String idriveBusinessReportUri;

    @Value("${smartdata.dataFootPrintUri:}")
    private String dataFootPrintUri;

    @Value("${smartdata.dataComparisonUri:}")
    private String dataComparisonUri;

    @Value("${smartdata.executionEngineUri:}")
    private String executionEngineUri;

    @Value("${smartdata.supportPerson:}")
    private String webSmartDataSupportPerson;

    @Value("${lcdp.uri:}")
    private String webLcdpUri;

    @Value("${atmc.uri:}")
    private String webAtmcUri;

    @Value("${adsc.uri:}")
    private String webAdscUri;

    /**
     * dap中间件
     */
    @Value("${iam.uri:}")
    private String iamUri;

    @Value("${themeMap.uri:}")
    private String webThemeMapUri;

    @Value("${themeMap.dataMapUri:}")
    private String dataMapUri;

    @Value("${themeMap.uri:}")
    private String knowledgeGraphUri;

    @Value("${themeMap.domainUri:}")
    private String webThemeMapDomainUri;

    @Value("${themeMap.supportPerson:}")
    private String webThemeMapSupportPerson;

    @Value("${bpm.uri:}")
    private String bpmUri;

    @Value("${bpm.supportPerson:}")
    private String bpmSupportPerson;

    @Value("${taskEngine.uri:}")
    private String taskEngineUri;

    @Value("${taskEngine.workflowUri:}")
    private String workflowUri;

    @Value("${taskEngine.supportPerson:}")
    private String taskSupportPerson;

    @Value("${taskEngine.ptmUri:}")
    private String ptmUri;

    @Value("${esp.uri:}")
    private String espUri;

    @Value("${esp.supportPerson:}")
    private String espSupportPerson;

    @Value("${esp.supportPersonApp:}")
    private String espSupportPersonApp;

    @Value("${mdc.uri:}")
    private String mdcUri;

    @Value("${mdc.lbUri:}")
    private String mdcLbUri;

    @Value("${eoc.uri:}")
    private String eocUri;

    @Value("${aam.uri}")
    private String aamUri;

    @Value("${audc.uri}")
    private String audcUri;

    @Value("${fi.uri:}")
    private String fiUri;

    @Value("${cloudeia.uri:}")
    private String cloudeiaUri;

    @Value("${athena.mock.mockUser:}")
    private String athenaMockUser;

    @Value("${athena.mock.mockTenantId:}")
    private String athenaMockTenantId;

    @Value("${athena.mock.mockEnv:}")
    private String athenaMockEnv;

    /**
     * dap中间件-日志中心
     */
    @Value("${dap.lmcUri:}")
    private String dapLmcUri;

    @Value("${cac.uri:}")
    private String cacUrl;

    @Value("${abt.uri:}")
    private String abtUri;

    @Value("${tdd.uri:}")
    private String tddUri;

    /**
     * 当前进程运行的主机名
     */
    private String host;
    /**
     * 当前进程所在的IP地址
     */
    private String ipAddress;
    /**
     * 空闲内存
     */
    private long freeMemory;
    /**
     * 内存总量
     */
    private long totalMemory;
    /**
     * java虚拟机允许开启的最大的内存
     */
    private long maxMemory;
    /**
     * 操作系统名称
     */
    private String osName;
    /**
     * 进程号
     */
    private long pid;
    /**
     * 程序启动时间
     */
    private LocalDateTime startTime;
    /**
     * 程序运行时间，单位毫秒
     */
    private long runtime;
    /**
     * 线程总量
     */
    private int threadCount;

    public void refresh() {
        RuntimeMXBean runtimeMXBean = ManagementFactory.getRuntimeMXBean();
        Runtime runtime = Runtime.getRuntime();
        //空闲内存
        long freeMemory = runtime.freeMemory();
        this.setFreeMemory(byteToM(freeMemory));
        //内存总量
        long totalMemory = runtime.totalMemory();
        this.setTotalMemory(byteToM(totalMemory));
        //最大允许使用的内存
        long maxMemory = runtime.maxMemory();
        this.setMaxMemory(byteToM(maxMemory));
        //操作系统
        this.setOsName(System.getProperty("os.name"));
        InetAddress localHost;
        try {
            localHost = InetAddress.getLocalHost();
            String hostName = localHost.getHostName();
            this.setHost(hostName);
            if (ipAddress == null) {
                ipAddress = localHost.getHostAddress();
            }
        } catch (UnknownHostException e) {
            log.error("UnknownHostException:{}",e.getMessage());
            this.setHost("未知");
        }
        //ip
        this.setIpAddress(ipAddress);

        //程序启动时间
        long startTime = runtimeMXBean.getStartTime();

        LocalDateTime startDate = LocalDateTime.ofEpochSecond(startTime / 1000, 0, ZoneOffset.ofHours(0));
        this.setStartTime(startDate);
        //类所在路径
        this.setRuntime(runtimeMXBean.getUptime());
        //线程总数
        this.setThreadCount(ManagementFactory.getThreadMXBean().getThreadCount());
        this.setPid(tryGetPid());
    }

    /**
     * 把byte转换成M
     *
     * @param bytes
     * @return
     */
    static long byteToM(long bytes) {
        long kb = (bytes / 1024 / 1024);
        return kb;
    }

    /**
     * 获取进程号，适用于windows与linux
     *
     * @return
     */
    static long tryGetPid() {
        try {
            String name = ManagementFactory.getRuntimeMXBean().getName();
            String pid = name.split("@")[0];
            return Long.parseLong(pid);
        } catch (NumberFormatException e) {
            return 0;
        }
    }

}
