package com.digiwin.athena.atdm.recycle;

import cn.hutool.core.collection.CollectionUtil;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.atdm.UiBotConstants;
import com.digiwin.athena.atdm.constant.ErrorCodeEnum;
import com.digiwin.athena.atdm.recycle.dto.RecycleRecordStateChangeDTO;
import com.digiwin.athena.atdm.recycle.po.RecycleDO;
import com.digiwin.athena.datamap.sdk.manager.DataMapManager;
import com.digiwin.athena.datamap.sdk.meta.dto.response.TmDataEntryDTO;
import com.mongodb.BasicDBObject;
import com.mongodb.client.model.IndexModel;
import com.mongodb.client.result.DeleteResult;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.BulkOperations;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.aggregation.Aggregation;
import org.springframework.data.mongodb.core.aggregation.AggregationResults;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.data.util.Pair;
import org.springframework.stereotype.Service;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import org.springframework.web.servlet.mvc.condition.RequestConditionHolder;

import javax.servlet.http.HttpServletRequest;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

@Service
@Slf4j
public class CommonRecycleServiceImpl implements CommonRecycleService {
    private static final String DATE_TIME = "dateTime";

    private static final String INTERNAL_ACCOUNT = "athenaAIRecycle";

    private static final String INTERNAL_ACCOUNT_PWD = "athena_20201015";

    @Autowired(required = false)
    @Qualifier("recycleMongoTemplate")
    private MongoTemplate recycleMongoTemplate;

    private MongoTemplate getRecycleMongoTemplate() {
        if (null != recycleMongoTemplate) {
            return recycleMongoTemplate;
        }
        throw BusinessException.create("创建recycleMongoTemplate实例失败，请检查数据源配置是否正确~");
    }

    @Override
    public int getRecycleCount(String tenantId) {

        // appCode   拿到所有的基础资料code
        List<Map<String, Object>> activityList = queryActivityList(tenantId);
        if (CollectionUtil.isEmpty(activityList)) {
            return 0;
        }

        IntSummaryStatistics summaryStatistics = activityList.stream().collect(Collectors.summarizingInt(t -> (int) t.get("count")));

        return Math.toIntExact(summaryStatistics.getSum());

    }

    @Override
    public List<Map<String, Object>> queryActivityList(String tenantId) {
        Aggregation aggregation = Aggregation.newAggregation(
                // 未被标记为过期
                Aggregation.match(new Criteria("state").ne(RecycleDO.STATE_EXPIRED)),
                Aggregation.sort(Sort.Direction.DESC, DATE_TIME),
                Aggregation.group("activityCode").sum("dataSize").as("count").first("activityCode").as("activityCode")
                        .first("dateTime").as("dateTime").first("lang").as("lang"));

        AggregationResults<RecycleDO> results = getRecycleMongoTemplate().aggregate(aggregation, tenantId, RecycleDO.class);
        List<Map<String, Object>> recycleList = (ArrayList<Map<String, Object>>) results.getRawResults().get("results");
        // 将最后一个被删除的作业排到最前面
        dataTimeOrder(recycleList, "dateTime");
        // 添加name属性
        recycleList = assignActivityName(recycleList);
//        HttpServletRequest request = ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getRequest();
//        String locale = request.getHeader("locale");
//        if(null==locale){locale= "zh_CN";}
//        String finalLocale = locale;
//        recycleList.forEach(map->{
//                try {
//                    Map<String,String> lang = (Map<String, String>) map.get("lang");
//                    if(null!=lang){
//                        map.put("activityName",lang.get(finalLocale));
//                    }
//                }catch (Exception e){
//                    e.printStackTrace();
//                }
//        });


        return recycleList;
    }

    private List<Map<String, Object>> assignActivityName(List<Map<String, Object>> recycleList) {

        if (CollectionUtil.isEmpty(recycleList)) {
            return recycleList;
        }
        Set<String> activityCodeList = new HashSet<>();
        recycleList.forEach(map->{
            String activityCode = (String) map.get("activityCode");
            if(null!=activityCode){
                activityCodeList.add(activityCode);
            }
        });
        // 查询km配置，获取基础资料数据
        List<TmDataEntryDTO> dataEntries = DataMapManager.getDataEntryListNames(activityCodeList).getResponse();
        Map<String, String> codeNameRel = dataEntries.stream().collect(Collectors.toMap(TmDataEntryDTO::getCode, TmDataEntryDTO::getName, (value1, value2) -> value2));
        recycleList.forEach(recycle -> {
            recycle.put("activityName", codeNameRel.get(recycle.get("activityCode")));
        });
        return recycleList;
    }

    @Override
    public List queryActivityDataList(String tenantId, String activityCode) {
        Criteria criteria = Criteria.where("activityCode").is(activityCode)
                // 未被标记为过期
                .and("state").ne(RecycleDO.STATE_EXPIRED);

        Query query = Query.query(criteria);
        List<RecycleDO> recycleList = getRecycleMongoTemplate().find(query, RecycleDO.class, tenantId);
        List dataList = new ArrayList();
        for (RecycleDO recycleDO : recycleList) {
            for (Map<String, Object> data : recycleDO.getPageData()) {
                if (MapUtils.isEmpty(data)) {
                    continue;
                }
                // 去掉“已勾选”标记
                data.remove(UiBotConstants.UIBOT_FIELDS_CHECKED);

                dataList.add(data);
            }
        }

        dataTimeOrder(dataList, "uibot__dateTime");
        return dataList;
    }

    @Override
    public void insertRecycle(AuthoredUser authoredUser, String activityCode, List<Map<String, Object>> dataList) {
        RecycleDO recycleDO = new RecycleDO();
        recycleDO.setTenantId(authoredUser.getTenantId());
        recycleDO.setActivityCode(activityCode);
        recycleDO.setActivityName(null);

        String sequenceNo = UUID.randomUUID().toString();
        LocalDateTime dateTime = LocalDateTime.now();

        for (Map<String, Object> data : dataList) {
            data.put("uibot__userId", authoredUser.getUserId());
            data.put("uibot__userName", authoredUser.getUserName());

            data.put("uibot__dateTime", dateTime);
            data.put("uibot__sequenceNo", sequenceNo);
        }
        recycleDO.setPageData(dataList);
        recycleDO.setDataSize(dataList.size());
        recycleDO.setDateTime(dateTime);

        recycleDO.setVersion(1);
        recycleDO.setSequenceNo(sequenceNo);
        recycleDO.setState(RecycleDO.STATE_ACTIVE);

        checkAndCreateCollection(authoredUser.getTenantId());

        getRecycleMongoTemplate().insert(recycleDO, authoredUser.getTenantId());
    }

    /**
     * 检查并创建Collection + index
     *
     * @param collectionName
     */
    private void checkAndCreateCollection(String collectionName) {
        if (getRecycleMongoTemplate().collectionExists(collectionName)) {
            return;
        }

        List<IndexModel> indexModelList = new ArrayList<>();
        BasicDBObject index = new BasicDBObject();
        // 1：升序；2：降序
        index.put(DATE_TIME, -1);
        indexModelList.add(new IndexModel(index));

        try {
            getRecycleMongoTemplate().createCollection(collectionName).createIndexes(indexModelList);
        } catch (Exception ex) {
            log.info("RecycleService-checkAndCreateCollection: create collection: {} failed, error: {}", collectionName, ex);
        }
    }

    @Override
    public void deleteRecycle(AuthoredUser authoredUser, String activityCode, List<Map<String, Object>> dataList) {
        // 1、按sequenceNo分类
        Map<String, List<Map<String, Object>>> seqDataList = new HashMap<>();
        for (Map<String, Object> data : dataList) {
            String sequenceNo = String.valueOf(data.get("uibot__sequenceNo"));
            List<Map<String, Object>> dataMapList = seqDataList.computeIfAbsent(sequenceNo, (seqNo) -> new ArrayList<>());
            dataMapList.add(data);
        }

        // 2、对每类sequenceNo执行删除操作
        for (Map.Entry<String, List<Map<String, Object>>> seqDataEntry : seqDataList.entrySet()) {
            deleteRecycle(authoredUser, activityCode, seqDataEntry.getKey(), seqDataEntry.getValue());
        }
    }

    public void deleteRecycle(AuthoredUser authoredUser, String activityCode, String sequenceNo, List<Map<String, Object>> toDeleteDataList) {
        Query query = Query.query(Criteria.where("sequenceNo").is(sequenceNo));
        List<RecycleDO> recycleDos = getRecycleMongoTemplate().find(query, RecycleDO.class, authoredUser.getTenantId());
        if (CollectionUtils.isEmpty(recycleDos)) {
            return;
        }

        RecycleDO recycleDO = recycleDos.get(0);
        List<Map<String, Object>> pageDataList = recycleDO.getPageData();
        if (CollectionUtils.isEmpty(pageDataList)) {
            deleteBySeqNoAndVersion(authoredUser.getTenantId(), recycleDO.getSequenceNo(), recycleDO.getVersion());
            return;
        }

        int origSize = pageDataList.size();

        Iterator<Map<String, Object>> iterator = pageDataList.iterator();
        while (iterator.hasNext()) {
            Map<String, Object> pageData = iterator.next();
            Object dataKey = pageData.containsKey(UiBotConstants.RECYCLE_DATA_KEY) ? pageData.get(UiBotConstants.RECYCLE_DATA_KEY) : pageData.get(UiBotConstants.DATA_SOURCE_DATA_KEY);

            for (Map<String, Object> toDeleteData : toDeleteDataList) {
                Object dataKey1 = toDeleteData.containsKey(UiBotConstants.RECYCLE_DATA_KEY) ? toDeleteData.get(UiBotConstants.RECYCLE_DATA_KEY) : toDeleteData.get(UiBotConstants.DATA_SOURCE_DATA_KEY);
                if (Objects.equals(dataKey, dataKey1)) {
                    iterator.remove();
                    break;
                }
            }
        }

        if (CollectionUtils.isEmpty(pageDataList)) {
            deleteBySeqNoAndVersion(authoredUser.getTenantId(), recycleDO.getSequenceNo(), recycleDO.getVersion());
            return;
        }

        if (origSize == pageDataList.size()) {
            // 数量没变，无需更新操作
            return;
        }

        // 执行更新操作：序号、版本号要相同
        query = Query.query(Criteria.where("sequenceNo").is(sequenceNo).and("version").is(recycleDO.getVersion()));
        // 更新pageData、dataSize、version
        Update update = Update.update("pageData", pageDataList).set("dataSize", pageDataList.size()).set("version", recycleDO.getVersion() + 1);
        getRecycleMongoTemplate().updateFirst(query, update, authoredUser.getTenantId());
    }

    public long deleteBySeqNoAndVersion(String tenantId, Object sequenceNo, Object version) {
        if (null == sequenceNo || null == version) {
            return 0L;
        }

        // 序号和版本号相同
        Query query = Query.query(Criteria.where("sequenceNo").is(sequenceNo).and("version").is(version));
        DeleteResult deleteRet = getRecycleMongoTemplate().remove(query, tenantId);
        return deleteRet.getDeletedCount();
    }

    @Override
    public void deleteAll(String tenantId) {
        getRecycleMongoTemplate().dropCollection(tenantId);
    }

    @Override
    public void deleteByActivityCode(AuthoredUser authoredUser, String tmActivityId) {
        Query query = Query.query(Criteria.where("activityCode").is(tmActivityId));
        getRecycleMongoTemplate().remove(query, authoredUser.getTenantId());
    }

    /**
     * 日期倒序
     *
     * @param list
     * @param compareParam 比较参数
     */
    private void dataTimeOrder(List list, String compareParam) {
        Collections.sort(list, (Comparator<Map<String, Object>>) (o1, o2) -> {
            Date time1 = (Date) o1.get(compareParam);
            Date time2 = (Date) o2.get(compareParam);
            return time2.compareTo(time1);
        });
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void markExpired(RecycleRecordStateChangeDTO recycleRecordStateChangeDTO) {
        // 调用接口不需要登录，这里增加内置账号、密码校验，提高安全性
        checkPwd(recycleRecordStateChangeDTO.getAccount(), recycleRecordStateChangeDTO.getPassword());

        List<Pair<Query, Update>> updateList = new ArrayList<>();
        BulkOperations operations = getRecycleMongoTemplate().bulkOps(BulkOperations.BulkMode.UNORDERED, recycleRecordStateChangeDTO.getTenantId());
        recycleRecordStateChangeDTO.getActivityCodeList().forEach(activityCode -> {
            Query query = Query.query(Criteria.where("activityCode").is(activityCode));

            Update update = new Update();
            update.set("state", RecycleDO.STATE_EXPIRED);
            Pair<Query, Update> updatePair = Pair.of(query, update);

            updateList.add(updatePair);
        });

        operations.updateMulti(updateList);
        operations.execute();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void markUnexpired(RecycleRecordStateChangeDTO recycleRecordStateChangeDTO) {
        // 调用接口不需要登录，这里增加内置账号、密码校验，提高安全性
        checkPwd(recycleRecordStateChangeDTO.getAccount(), recycleRecordStateChangeDTO.getPassword());

        List<Pair<Query, Update>> updateList = new ArrayList<>();
        BulkOperations operations = getRecycleMongoTemplate().bulkOps(BulkOperations.BulkMode.UNORDERED, recycleRecordStateChangeDTO.getTenantId());
        recycleRecordStateChangeDTO.getActivityCodeList().forEach(activityCode -> {
            Query query = Query.query(Criteria.where("activityCode").is(activityCode));

            Update update = new Update();
            update.set("state", RecycleDO.STATE_ACTIVE);
            Pair<Query, Update> updatePair = Pair.of(query, update);

            updateList.add(updatePair);
        });

        operations.updateMulti(updateList);
        operations.execute();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void markDeleted(RecycleRecordStateChangeDTO recycleRecordStateChangeDTO) {
        // 调用接口不需要登录，这里增加内置账号、密码校验，提高安全性
        checkPwd(recycleRecordStateChangeDTO.getAccount(), recycleRecordStateChangeDTO.getPassword());

        Query query = Query.query(Criteria.where("activityCode").in(recycleRecordStateChangeDTO.getActivityCodeList()));
        getRecycleMongoTemplate().remove(query, recycleRecordStateChangeDTO.getTenantId());
    }

    private void checkPwd(String account, String password) {
        if (!StringUtils.equals(account, INTERNAL_ACCOUNT) || !StringUtils.equals(password, INTERNAL_ACCOUNT_PWD)) {
            throw BusinessException.create(ErrorCodeEnum.RECYCLE_PWD_ERROR.getErrCode(), "账号、密码错误");
        }
    }
}
