package com.digiwin.athena.atdm.terminateData;

import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.TimeUtils;
import com.digiwin.athena.atdm.UiBotConstants;
import com.digiwin.athena.atdm.atmc.CommonAtmcService;
import com.digiwin.athena.atdm.iam.CommonUserService;
import com.digiwin.athena.atdm.terminateData.dto.TerminateDataDTO;
import com.digiwin.athena.atdm.terminateData.po.TerminateDataDO;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * TerminateDataServiceImpl Description
 *
 * @author majianfu
 * @date 2022/2/17
 * @since
 */
@Slf4j
@Service
public class CommonTerminateDataServiceImpl implements CommonTerminateDataService {
    @Autowired(required = false)
    @Qualifier("terminateDataMongoTemplate")
    private MongoTemplate terminateDataMongoTemplate;

    @Autowired
    private CommonAtmcService atmcService;

    @Autowired
    private CommonUserService commonUserService;

    @Autowired
    private TerminateDataConverter terminateDataConverter;

    @Autowired
    private MessageUtils messageUtils;

    private MongoTemplate getTerminateDataMongoTemplate() {
        if (null != terminateDataMongoTemplate) {
            return terminateDataMongoTemplate;
        }
        throw BusinessException.create("创建terminateDataMongoTemplate失败，请检查数据源配置是否正确~");
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void saveTerminateData(AuthoredUser authoredUser, TerminateDataDTO terminateDataDTO) {
        if (CollectionUtils.isEmpty(terminateDataDTO.getPageData())) {
            return;
        }

        TerminateDataDO terminateDataDO = initTerminateDataDO(authoredUser, terminateDataDTO);
        if (null != terminateDataDO) {
            getTerminateDataMongoTemplate().insert(terminateDataDO, authoredUser.getTenantId());
        }
    }

    private TerminateDataDO initTerminateDataDO(AuthoredUser authoredUser, TerminateDataDTO terminateDataDTO) {
        if (CollectionUtils.isEmpty(terminateDataDTO.getPageData())) {
            return null;
        }

        TerminateDataDO terminateDataDO = terminateDataConverter.convert(terminateDataDTO);
        terminateDataDO.setUserId(authoredUser.getUserId());
        terminateDataDO.setUserName(authoredUser.getUserName());
        terminateDataDO.setTenantId(authoredUser.getTenantId());

        if (null == terminateDataDO.getTerminateTime()) {
            terminateDataDO.setTerminateTime(LocalDateTime.now());
        }
        return terminateDataDO;
    }

    @Override
    public void saveTerminateData(AuthoredUser authoredUser, List<TerminateDataDTO> dataDTOList) {
        List<TerminateDataDO> resultList = Lists.newArrayListWithExpectedSize(dataDTOList.size());
        for (TerminateDataDTO terminateDataDTO : dataDTOList) {
            TerminateDataDO terminateDataDO = initTerminateDataDO(authoredUser, terminateDataDTO);
            if(terminateDataDO != null){
                resultList.add(terminateDataDO);
            }
        }
        if(!resultList.isEmpty()){
            getTerminateDataMongoTemplate().insert(resultList, authoredUser.getTenantId());
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public List queryTerminateDataList(String token, String tenantId, String userId, String activityCode, Collection<String> activityIds) {
        Criteria criteria = Criteria.where("tenantId").is(tenantId)
                .and("activityCode").is(activityCode)
                .and("activityId").in(activityIds);

        Query query = Query.query(criteria);
        List<TerminateDataDO> terminateDataList = getTerminateDataMongoTemplate().find(query, TerminateDataDO.class, tenantId);

        List resultList = new ArrayList();
        if (CollectionUtils.isNotEmpty(terminateDataList)) {
            // 收集无activityName的activityId
            List<String> needParseNameActivityIds = terminateDataList.stream()
                    .filter(terminateDataDO -> StringUtils.isBlank(terminateDataDO.getActivityName()))
                    .filter(terminateDataDO -> StringUtils.isNotBlank(terminateDataDO.getActivityId()))
                    .map(TerminateDataDO::getActivityId)
                    .collect(Collectors.toList());
            List<String> needParseNameBacklogIds = terminateDataList.stream()
                    .filter(terminateDataDO -> StringUtils.isBlank(terminateDataDO.getActivityName()))
                    .filter(terminateDataDO -> StringUtils.isNotBlank(terminateDataDO.getInitiateActivityId()))
                    .map(TerminateDataDO::getInitiateActivityId)
                    .collect(Collectors.toList());
            Map<String, String> activityIdNameMap = queryActivityNameById(needParseNameActivityIds, needParseNameBacklogIds);

            // 是否为外部用户
            boolean isExternalUser = isExternalUser(token, userId);
            String operationType = messageUtils.getMessage("shelved_uibot__operation_type_terminate");

            // 收集、规则pageDataList
            for (TerminateDataDO terminateDataDO : terminateDataList) {
                resultList.addAll(buildTerminatePageData(operationType, isExternalUser, terminateDataDO, activityIdNameMap));
            }
        }
        return resultList;
    }

    private boolean isExternalUser(String token, String userId) {
        try {
            Integer userType = commonUserService.queryUserType(userId, token);
            return null != userType ? 1 == userType : false;
        } catch (Exception ex) {
            log.error("请求IAM判断用户类型发生异常，参数：token：{}, userId：{}, 异常内容：{}", token, userId, ex.getMessage());
            // 如果请求获取用户类型失败，直接认为是外部用户
            return true;
        }
    }

    private Map<String, String> queryActivityNameById(List<String> workItemIdList, List<String> backlogIdList) {
        if (CollectionUtils.isEmpty(workItemIdList)) {
            return Collections.emptyMap();
        }

        return atmcService.queryActivityNameById(workItemIdList, backlogIdList);
    }

    private List buildTerminatePageData(String operationType, boolean isExternalUser, TerminateDataDO terminateDataDO, Map<String, String> activityIdNameMap) {
        List resultList = new ArrayList();
        String terminateDateTimeStr = TimeUtils.format(terminateDataDO.getTerminateTime(), TimeUtils.DEFAULT_FORMAT1);
        for (Map<String, Object> rowData : terminateDataDO.getPageData()) {
            // 终止执行人信息
            rowData.put("uibot__userId", terminateDataDO.getUserId());
            rowData.put("uibot__userName", terminateDataDO.getUserName());
            rowData.put("uibot__operationReason", terminateDataDO.getComment());
            // 终止执行时间
            rowData.put(UiBotConstants.FieldName.UIBOT_DATETIME, terminateDataDO.getTerminateTime());
            rowData.put("uibot__operationType", operationType);

            // 如果有发起终止的任务信息（initiateActivityId）
            if (StringUtils.isNotBlank(terminateDataDO.getInitiateActivityId())) {
                rowData.put("uibot__initiateActivityId", terminateDataDO.getInitiateActivityId());
                rowData.put("uibot__initiateActivityCode", terminateDataDO.getInitiateActivityCode());
                rowData.put("uibot__initiateActivityName", terminateDataDO.getInitiateActivityName());
                // 非外部用户 && 发起终止的任务不是当前任务，则允许链接到原始发起终止的任务；否则不允许链接
                rowData.put("uibot__link", !isExternalUser && !StringUtils.equals(terminateDataDO.getActivityId(), terminateDataDO.getInitiateActivityId()));
            }
            // 如果没有发起终止的任务信息（initiateActivityId），则activityId就是initiateActivityId
            else {
                rowData.put("uibot__initiateActivityId", terminateDataDO.getActivityId());
                rowData.put("uibot__initiateActivityCode", terminateDataDO.getActivityCode());

                String activityName = terminateDataDO.getActivityName();
                activityName = StringUtils.isNotBlank(activityName) ? activityName :
                        StringUtils.isNotBlank(activityIdNameMap.get(terminateDataDO.getActivityId())) ? activityIdNameMap.get(terminateDataDO.getActivityId())
                                : activityIdNameMap.get(terminateDataDO.getInitiateActivityId());
                rowData.put("uibot__initiateActivityName", activityName);
                rowData.put("uibot__link", false);
            }

            // 去掉“已勾选”标记
            rowData.remove(UiBotConstants.UIBOT_FIELDS_CHECKED);
            // 改变__DATA_KEY
            if (null != rowData.get(UiBotConstants.DATA_SOURCE_DATA_KEY)) {
                String dataKey = rowData.get(UiBotConstants.DATA_SOURCE_DATA_KEY).toString();

                // 操作类型;执行人userId;操作时间;任务名称;老dataKey
                String initiateActivityCodeStr = null != rowData.get("uibot__initiateActivityCode") ?
                        rowData.get("uibot__initiateActivityCode").toString() : "NAN";
                String newDataKey = operationType + ";" + terminateDataDO.getUserId() + ";" + terminateDateTimeStr + ";" + initiateActivityCodeStr + ";" + dataKey;

                rowData.put(UiBotConstants.DATA_SOURCE_DATA_KEY, newDataKey);
            }

            resultList.add(rowData);
        }
        return resultList;
    }
}
