package com.digiwin.athena.atdm.datasource.domain;

import org.apache.commons.collections4.CollectionUtils;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.BiConsumer;

/**
 * @description: API元数据
 * @author: wangminga
 * @create: 2020-05-14 11:25
 **/
public class ApiMetadata implements Serializable {

    public ApiMetadata(){
        responseFields = new ArrayList<>();
        requestFields = new ArrayList<>();
    }
    /**
     * 元数据ID
     */
    private String actionId;

    /**
     * action调用类型: sync同步，async异步，默认同步
     */
    private String invokeType;

    /**
     * 元数据名称
     */
    private String serviceName;

    /**
     * request 字段集合
     */
    private List<MetadataField> requestFields;

    /**
     * response 字段集合
     */
    private  List<MetadataField> responseFields;

    private Set<String> apiTags;

    public String getActionId() {
        return actionId;
    }

    public void setActionId(String actionId) {
        this.actionId = actionId;
    }

    public String getServiceName() {
        return serviceName;
    }

    public void setServiceName(String serviceName) {
        this.serviceName = serviceName;
    }

    public List<MetadataField> getRequestFields() {
        return requestFields;
    }

    public void addRequestFields(List<MetadataField> requestFields) {
        this.requestFields.addAll(requestFields);
    }

    public List<MetadataField> getResponseFields() {
        return responseFields;
    }

    /**
     *  添加字段
     * @param fields
     */
    public void addResponseFields(List<MetadataField> fields) {
        addResponseFieldsConsumer(fields,(e,field)->{
            // 现有逻辑的添加字段，有相同的直接添加
            this.responseFields.get(0).getSubFields().remove(e);
            this.responseFields.get(0).getSubFields().add(field);
        });
    }


    /**
     *  抽取增加字段的通用代码
     * @param fields
     * @param consumer
     */
    private void addResponseFieldsConsumer(List<MetadataField> fields, BiConsumer<MetadataField,MetadataField> consumer) {
        if (this.responseFields == null){
            this.responseFields = new ArrayList<>();
        }
        if (CollectionUtils.isEmpty(fields)){
            return;
        }
        // 如果响应字段为空直接添加
        if (this.responseFields.isEmpty()){
            this.responseFields.addAll(fields);
        }else {
            // 初始化当前元数据的的单身的值
            if ( this.responseFields.get(0).getSubFields() == null ){
                this.responseFields.get(0).setSubFields(new ArrayList<>());
            }
            // 找到单身的所有的名称用来判断是否存在
            Set<String> allFields=new HashSet<>();
            for (MetadataField subField : this.responseFields.get(0).getSubFields()) {
                allFields.add(subField.getName());
            }
            // 遍历添加的字段
            for (MetadataField field : fields) {
                // 如果不存在直接添加
                if (!allFields.contains(field.getName())){
                    this.responseFields.get(0).getSubFields().add(field);
                }else{
                    //已经存在的，替换掉，因为指定元数据目的应该有2个：增加缺少的和替换
                    Optional<MetadataField> existFieldOptional = this.responseFields.get(0).getSubFields().stream().filter(x->Objects.equals(x.getName(),field.getName())).findFirst();
                    // 如果有值直接调用消费的方法用来做不同的处理
                    existFieldOptional.ifPresent(metadataField -> consumer.accept(metadataField, field));

                }
            }
        }

    }

    /**
     * 添加虚拟字段
     * @param fields
     */
    public void addVirtuallyResponseFields(List<MetadataField> fields) {
        // 通用的前期处理
        addResponseFieldsConsumer(fields,(e,f)->{
            List<MetadataField> subFields = e.getSubFields();
            List<MetadataField> sourceSubFields = f.getSubFields();
            // 虚拟字段没有子的也要移除
            if(CollectionUtils.isEmpty(subFields) || CollectionUtils.isEmpty(sourceSubFields)){
                this.responseFields.get(0).getSubFields().remove(e);
                this.responseFields.get(0).getSubFields().add(f);
            }else {
                // 如果虚拟字段有单身就和元数据的字段单身进行合并，没有不处理
                if(CollectionUtils.isNotEmpty(sourceSubFields)){
                    mergeMetadataField(sourceSubFields,subFields);
                }
            }
        });
    }

    /**
     * 合并字段，如
     * [{name:"1",subFields:[{name:"1-1",remark:"111"},{name:"1-3"}]},{name:"2"},{name:"4",subFields:[{name:5}]}]
     * [{name:"1",subFields:[{name:"1-1"},{name:"1-4"}]},{name:"3"},{name:"4",subFields:[]}]
     * 转换为[{name:"1",subFields:[{name:"1-4"},{name:"1-1",remark:"111"},{name:"1-3"}]},{name:"3"},{name:"4",subFields:[{name:5}]}]
     * @param sourceFields 虚拟字段
     * @param targetFields 目标字段
     */
    private void mergeMetadataField(List<MetadataField> sourceFields,List<MetadataField> targetFields){
        // 遍历虚拟字段
        Iterator<MetadataField> iterator = sourceFields.iterator();
        while (iterator.hasNext()){
            MetadataField next = iterator.next();
            String name = next.getName();
            List<MetadataField> subFields = next.getSubFields();
            // 用来标记相同名称的目标字段位置
            int index = -1;
            for (int i = 0; i < targetFields.size(); i++) {
                MetadataField metadataField = targetFields.get(i);
                if(Objects.equals(metadataField.getName(),name)){
                    index = i;
                }
            }
            // 有相同值
            if(index>-1){
                MetadataField metadataField = targetFields.get(index);
                List<MetadataField> subTargetFields = metadataField.getSubFields();
                // 如果目标字段有单身
                if(CollectionUtils.isNotEmpty(subTargetFields)){
                    // 移除虚拟字段，不进行合并
                    iterator.remove();
                    // 如果虚拟字段含有单身，递归处理子层级
                    if(CollectionUtils.isNotEmpty(subFields)){
                        mergeMetadataField(subFields,subTargetFields);
                    }
                }else {
                    // 没有单身，直接用虚拟字段的信息
                    targetFields.remove(index);
                }
            }
        }
        // 将虚拟字段合入目标字段
        if(CollectionUtils.isNotEmpty(sourceFields)){
            targetFields.addAll(sourceFields);
        }
    }


    public Set<String> getApiTags() {
        return apiTags;
    }

    public void setApiTags(Set<String> apiTags) {
        this.apiTags = apiTags;
    }

    public String getInvokeType() {
        return invokeType;
    }

    public void setInvokeType(String invokeType) {
        this.invokeType = invokeType;
    }
}
