package com.digiwin.athena.atdm.datasource;

import cn.hutool.core.map.MapUtil;
import com.digiwin.athena.appcore.auth.GlobalConstant;
import com.digiwin.athena.appcore.auth.domain.AuthoredUser;
import com.digiwin.athena.appcore.domain.log.LogDto;
import com.digiwin.athena.atdm.DataSourceConstants;
import com.digiwin.athena.atdm.activity.domain.TmQueryAction;
import com.digiwin.athena.atdm.activity.service.BaseTmDataSourceService;
import com.digiwin.athena.atdm.adsc.CommonAdscService;
import com.digiwin.athena.atdm.datasource.datasource.process.AtmcBatchDataSourceService;
import com.digiwin.athena.atdm.datasource.domain.DataSourceProcessor;
import com.digiwin.athena.atdm.datasource.domain.ExecuteContext;
import com.digiwin.athena.atdm.datasource.domain.QueryResult;
import com.digiwin.athena.atdm.datasource.domain.QueryResultSet;
import com.digiwin.athena.atdm.datasource.dto.DataQueryDTO;
import com.digiwin.athena.atdm.datasource.dto.DataSourceDTO;
import com.digiwin.athena.atdm.datasource.dto.QueryDataByActionDTO;
import com.digiwin.athena.atdm.datasource.dto.QueryDataByActionIdDTO;
import com.digiwin.athena.atdm.datasource.dto.QueryDataByDataSourceDTO;
import com.digiwin.athena.atdm.esp.CommonESPService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

@Slf4j
@Service
public class DataQueryServiceWrapper {
    @Autowired
    private BaseTmDataSourceService baseTmDataSourceService;

    @Autowired
    private DataFetchingService dataFetchingService;

    @Autowired
    private CommonESPService commonEspService;

    @Autowired
    private CommonAdscService adscService;

    @Resource
    private AtmcBatchDataSourceService atmcBatchDataSourceService;


    /**
     * 根据代办的数据来查询获取数据
     *
     * @param request
     * @param dataQueryDTO
     * @return
     */
    public QueryResultSet queryWithMetaData(HttpServletRequest request, DataQueryDTO dataQueryDTO) {
        AuthoredUser authoredUser = (AuthoredUser) request.getAttribute(com.digiwin.athena.appcore.auth.GlobalConstant.AUTH_USER);
        LogDto logDto = new LogDto("根据代办的数据来查询获取数据开始:", authoredUser.getTenantId());
        log.info(logDto.toString());
        if (dataQueryDTO.getDataSourceSet() == null) {
            dataQueryDTO.setDataSourceSet(dataQueryDTO.getDataSourceSetDTO());
        }

        if (dataQueryDTO.getDataSourceSet() == null) {
            throw new IllegalArgumentException("queryDataByActivityDTO.getDataSourceSet()");
        }
        if (dataQueryDTO.getExecuteContext() == null) {
            throw new IllegalArgumentException("queryDataByActivityDTO.getExecuteContext()");
        }
        dataQueryDTO.getExecuteContext().appendHttpRequest(request);
        dataQueryDTO.getExecuteContext().appendExtend(dataQueryDTO.getSettings());
        //判断是否需要整理数据
        DataSourceProcessor dataSourceProcessor = dataFetchingService.judgeAbnormalProcessor(dataQueryDTO);
        QueryResultSet queryResult = dataFetchingService.queryWithMetaData(dataQueryDTO.getExecuteContext(), dataQueryDTO.getDataSourceSet(), dataQueryDTO.getParameter(), dataQueryDTO.getPageInfo(), dataQueryDTO.getSortInfo(), dataQueryDTO.getSearchInfo());
        dataFetchingService.dealQueryResult(queryResult, dataSourceProcessor);
        return queryResult;
    }

    /**
     * 根据代办的数据来查询获取数据
     *
     * @param request
     * @param dataQueryDTO
     * @return
     */
    public QueryResultSet queryByBacklog(HttpServletRequest request, DataQueryDTO dataQueryDTO) {
        AuthoredUser authoredUser = (AuthoredUser) request.getAttribute(com.digiwin.athena.appcore.auth.GlobalConstant.AUTH_USER);
        LogDto logDto = new LogDto("根据代办的数据来查询获取数据开始:", authoredUser.getTenantId());
        log.info(logDto.toString());
        if (dataQueryDTO.getDataSourceSet() == null) {
            dataQueryDTO.setDataSourceSet(dataQueryDTO.getDataSourceSetDTO());
        }

        if (dataQueryDTO.getDataSourceSet() == null) {
            throw new IllegalArgumentException("queryDataByActivityDTO.getDataSourceSetDTO()");
        }
        if (dataQueryDTO.getExecuteContext() == null) {
            throw new IllegalArgumentException("queryDataByActivityDTO.getExecuteContext()");
        }
        dataQueryDTO.getExecuteContext().appendHttpRequest(request);
        return dataFetchingService.query(dataQueryDTO.getExecuteContext(), dataQueryDTO.getDataSourceSet(), dataQueryDTO.getParameter(), dataQueryDTO.getPageInfo(), dataQueryDTO.getSortInfo(), dataQueryDTO.getSearchInfo());
    }


    /**
     * 根据数据源来查询获取数据
     *
     * @param request
     * @param dataQueryDTO
     * @return
     */
    public Map queryPageDataByDatasource(HttpServletRequest request, DataQueryDTO dataQueryDTO) {
        if (dataQueryDTO.getDataSourceSet() == null) {
            dataQueryDTO.setDataSourceSet(dataQueryDTO.getDataSourceSetDTO());
        }

        if (dataQueryDTO.getDataSourceSet() == null) {
            throw new IllegalArgumentException("queryDataByActivityDTO.getDataSourceSet()");
        }
        if (dataQueryDTO.getExecuteContext() == null) {
            throw new IllegalArgumentException("queryDataByActivityDTO.getExecuteContext()");
        }
        dataQueryDTO.getExecuteContext().appendHttpRequest(request);
        // 处理 PageInfo SortInfo SearchInfo
        dataFetchingService.dealDataQueryInfoBeforeQuery(dataQueryDTO);
        QueryResultSet queryResult = dataFetchingService.query(dataQueryDTO.getExecuteContext(), dataQueryDTO.getDataSourceSet(), dataQueryDTO.getParameter(), dataQueryDTO.getPageInfo(), dataQueryDTO.getSortInfo(), dataQueryDTO.getSearchInfo());
        Map result = new HashMap();
        Map pageData = new HashMap();
        pageData.put("uibot__page__parameter", dataQueryDTO.getParameter());
        if (MapUtils.isEmpty(dataQueryDTO.getParameter())) {
            pageData.put("uibot__page__parameter", new HashMap());
        }
        if (queryResult.getMainQueryResult() != null && !StringUtils.isEmpty(queryResult.getMainQueryResult().getDataSourceName())) {

            pageData.put(queryResult.getMainQueryResult().getDataSourceName(), queryResult.getPageData().get(queryResult.getMainQueryResult().getDataSourceName()));
        }
        result.put("pageData", pageData);
        result.put("pageDataIndex", queryResult.getPageDataIndex());
        result.put("pageDataKeys", queryResult.getPageDataKeys());
        result.put("pageCountSize", queryResult.getMainQueryResult().size());
        result.put("returnRequestData", queryResult.getMainQueryResult().getReturnRequestData());
        /**
         * 该方法里设置result.searchInfo的逻辑不能做合并，必须用最外层即dataQueryDTO.searchInfo
         * 这样：
         * 1、返回给前端的就是表格中做筛选的条件；
         * 2、下次前端再做请求时，还会走一遍合并逻辑；
         */
        dataFetchingService.queryInfo(dataQueryDTO, queryResult, result);
        dataFetchingService.dealPageDataAfterQuery(dataQueryDTO, queryResult, result);

        return result;
    }

    /**
     * 根据代办的数据来查询获取数据
     *
     * @param request
     * @param dataQueryDTO
     * @return
     */
    public Integer querySize(HttpServletRequest request, DataQueryDTO dataQueryDTO) {
        if (dataQueryDTO.getDataSourceSet() == null) {
            dataQueryDTO.setDataSourceSet(dataQueryDTO.getDataSourceSetDTO());
        }
        if (dataQueryDTO.getDataSource() == null) {
            dataQueryDTO.setDataSource(dataQueryDTO.getDataSourceDTO());
        }
        if (dataQueryDTO.getDataSource() == null && dataQueryDTO.getDataSourceSet() != null && CollectionUtils.isNotEmpty(dataQueryDTO.getDataSourceSet().getDataSourceList())) {
            dataQueryDTO.setDataSource(dataQueryDTO.getDataSourceSet().getDataSourceList().get(0));
        }

        if (dataQueryDTO.getDataSource() == null) {
            throw new IllegalArgumentException("queryDataByActivityDTO.getDataSource()");
        }
        if (dataQueryDTO.getExecuteContext() == null) {
            throw new IllegalArgumentException("queryDataByActivityDTO.getExecuteContext()");
        }

        dataQueryDTO.getExecuteContext().appendHttpRequest(request);
        return dataFetchingService.size(dataQueryDTO);
    }

    /**
     * 根据代办的数据来查询获取数据
     *
     * @param request
     * @param dataQueryDTO
     * @return
     */
    public Map querySizeByDataSize(HttpServletRequest request, DataQueryDTO dataQueryDTO) {
        AuthoredUser authoredUser = (AuthoredUser) request.getAttribute(com.digiwin.athena.appcore.auth.GlobalConstant.AUTH_USER);
        LogDto logDto = new LogDto("根据代办的数据来查询获取数据size开始:", authoredUser.getTenantId());
        log.info(logDto.toString());
        if (dataQueryDTO.getDataSourceSet() == null) {
            dataQueryDTO.setDataSourceSet(dataQueryDTO.getDataSourceSetDTO());
        }
        if (dataQueryDTO.getDataSource() == null) {
            dataQueryDTO.setDataSource(dataQueryDTO.getDataSourceDTO());
        }
        if (dataQueryDTO.getDataSource() == null && dataQueryDTO.getDataSourceSet() != null && CollectionUtils.isNotEmpty(dataQueryDTO.getDataSourceSet().getDataSourceList())) {
            dataQueryDTO.setDataSource(dataQueryDTO.getDataSourceSet().getDataSourceList().get(0));
        }

        if (dataQueryDTO.getDataSource() == null) {
            throw new IllegalArgumentException("queryDataByActivityDTO.getDataSource()");
        }
        if (dataQueryDTO.getExecuteContext() == null) {
            throw new IllegalArgumentException("queryDataByActivityDTO.getExecuteContext()");
        }

        dataQueryDTO.getExecuteContext().appendHttpRequest(request);

        //对于size直接使用union来做处理，union数据源内部会进行优化，合并api做查询
        if (dataQueryDTO.getDataSourceSet().getDataSourceList() != null && dataQueryDTO.getDataSourceSet().getDataSourceList().size() > 1) {
            Map<String, List<DataSourceDTO>> groupDs = dataQueryDTO.getDataSourceSet().getDataSourceList().stream().collect(Collectors.groupingBy(DataSourceDTO::getName));
            dataQueryDTO.getDataSourceSet().getDataSourceList().clear();
            for (List<DataSourceDTO> dataSourceDTOS : groupDs.values()) {
                if (dataSourceDTOS.size() > 1) {
                    DataSourceDTO allDs = new DataSourceDTO();
                    allDs.setType(DataSourceConstants.ACTION_CATEGORY_MIX_UNION);
                    allDs.setUnionItems(dataSourceDTOS);
                    allDs.setName(dataSourceDTOS.get(0).getName());
                    allDs.setDataKeys(dataSourceDTOS.get(0).getDataKeys());
                    dataQueryDTO.getDataSourceSet().getDataSourceList().add(allDs);
                } else {
                    dataQueryDTO.getDataSourceSet().getDataSourceList().add(dataSourceDTOS.get(0));
                }
            }
            if (dataQueryDTO.getDataSourceSet().getDataSourceList().size() == 1) {
                dataQueryDTO.setDataSource(dataQueryDTO.getDataSourceSet().getDataSourceList().get(0));
                dataQueryDTO.getDataSourceSet().getDataSourceList().clear();
            }
        }
        int size = 0;
        if (dataQueryDTO.getDataSourceSet().getDataSourceList() != null && dataQueryDTO.getDataSourceSet().getDataSourceList().size() > 1) {
            for (DataSourceDTO dataSource : dataQueryDTO.getDataSourceSet().getDataSourceList()) {
                if (dataQueryDTO.getDataSourceSet().getMainDatasource().equals(dataSource.getName())) {
                    size += dataFetchingService.size(dataQueryDTO, dataSource);
                }
            }
        } else {
            if (dataQueryDTO.getDataSourceSet().getMainDatasource().equals(dataQueryDTO.getDataSource().getName())) {
                size = dataFetchingService.size(dataQueryDTO);
            }
        }
        return MapUtil.of("size", size);
    }

    /**
     * 根据数据源的定义来获取数据
     *
     * @param queryDataByActionVO
     * @return
     */
    public QueryResult queryByDatasource(HttpServletRequest request, QueryDataByDataSourceDTO queryDataByActionVO, AuthoredUser user) {
        ExecuteContext executeContext = queryDataByActionVO.getExecuteContext();
        if (executeContext != null) {
            executeContext.appendHttpRequest(request);
        } else {
            executeContext = ExecuteContext.createByHttpRequest(request);
        }
        executeContext.setBusinessUnit(queryDataByActionVO.getBusinessUnit());

        if (queryDataByActionVO.getDataSource() == null) {
            throw new IllegalArgumentException("queryDataByActionVO.getDataSource()");
        }
        return dataFetchingService.querySingleResult(executeContext, queryDataByActionVO.getDataSource(), queryDataByActionVO.getParameter(), queryDataByActionVO.getPageInfo(), queryDataByActionVO.getSortInfo(), queryDataByActionVO.getSearchInfo());
    }

    /**
     * 根据tmACTION来获取数据
     *
     * @param queryDataByActionDTO
     * @return
     */
    public Map actionData(HttpServletRequest request, QueryDataByActionDTO queryDataByActionDTO, AuthoredUser user) {
        ExecuteContext executeContext = ExecuteContext.createByHttpRequest(request);
        executeContext.setBusinessUnit(queryDataByActionDTO.getBusinessUnit());

        //获取ESP 节点相关信息
        TmQueryAction tmAction = queryDataByActionDTO.getTmAction();
        DataSourceDTO dataSourceBase = null;
        if (queryDataByActionDTO.getBpmData() != null) {
            dataSourceBase = baseTmDataSourceService.analysis(executeContext, "data", tmAction, queryDataByActionDTO.getBpmData());
        } else if (queryDataByActionDTO.getParas() != null) {
            dataSourceBase = baseTmDataSourceService.analysisHasParas(executeContext, "data", tmAction, queryDataByActionDTO.getParas());
        }
        QueryResult queryResult = dataFetchingService.querySingleResult(executeContext, dataSourceBase, null, queryDataByActionDTO.getPageInfo(), queryDataByActionDTO.getSortInfo(), queryDataByActionDTO.getSearchInfo());

        return queryResult.getPageData();
    }

    /**
     * 根据tmACTION来获取数据
     *
     * @param queryDataByActionDTO
     * @return
     */
    public Map queryByActionId(HttpServletRequest request, QueryDataByActionIdDTO queryDataByActionDTO, AuthoredUser user) {
        ExecuteContext executeContext = ExecuteContext.createByHttpRequest(request);
        executeContext.setBusinessUnit(queryDataByActionDTO.getBusinessUnit());
        if (StringUtils.isEmpty(queryDataByActionDTO.getActionId())) {
            throw new IllegalArgumentException("actionId 为空");
        }
        if (Objects.nonNull(queryDataByActionDTO.getExecuteContext())) {
            String application = queryDataByActionDTO.getExecuteContext().getApplication();
            executeContext.setApplication(application);
        }
        return commonEspService.queryByApiName(executeContext, queryDataByActionDTO.getActionId(), queryDataByActionDTO.getParameter(), queryDataByActionDTO.getPageInfo(), queryDataByActionDTO.getSortInfo(), queryDataByActionDTO.getSearchInfo());
    }


    /**
     * 根据代办的数据来查询获取数据
     *
     * @param request
     * @param dataQueryDTO
     * @return
     */
    public Map querySizeByDataSizeAbnormal(HttpServletRequest request, DataQueryDTO dataQueryDTO) {
        if (dataQueryDTO.getDataSourceSet() == null) {
            dataQueryDTO.setDataSourceSet(dataQueryDTO.getDataSourceSetDTO());
        }
        if (dataQueryDTO.getDataSource() == null) {
            dataQueryDTO.setDataSource(dataQueryDTO.getDataSourceDTO());
        }
        if (dataQueryDTO.getDataSource() == null && dataQueryDTO.getDataSourceSet() != null && CollectionUtils.isNotEmpty(dataQueryDTO.getDataSourceSet().getDataSourceList())) {
            dataQueryDTO.setDataSource(dataQueryDTO.getDataSourceSet().getDataSourceList().get(0));
        }

        if (dataQueryDTO.getDataSource() == null) {
            throw new IllegalArgumentException("queryDataByActivityDTO.getDataSource()");
        }
        if (dataQueryDTO.getExecuteContext() == null) {
            throw new IllegalArgumentException("queryDataByActivityDTO.getExecuteContext()");
        }

        dataQueryDTO.getExecuteContext().appendHttpRequest(request);

        //对于size直接使用union来做处理，union数据源内部会进行优化，合并api做查询
        if (dataQueryDTO.getDataSourceSet().getDataSourceList() != null && dataQueryDTO.getDataSourceSet().getDataSourceList().size() > 1) {
            Map<String, List<DataSourceDTO>> groupDs = dataQueryDTO.getDataSourceSet().getDataSourceList().stream().collect(Collectors.groupingBy(DataSourceDTO::getName));
            dataQueryDTO.getDataSourceSet().getDataSourceList().clear();
            for (List<DataSourceDTO> dataSourceDTOS : groupDs.values()) {
                if (dataSourceDTOS.size() > 1) {
                    DataSourceDTO allDs = new DataSourceDTO();
                    allDs.setType(DataSourceConstants.ACTION_CATEGORY_MIX_UNION);
                    allDs.setUnionItems(dataSourceDTOS);
                    allDs.setName(dataSourceDTOS.get(0).getName());
                    allDs.setDataKeys(dataSourceDTOS.get(0).getDataKeys());
                    dataQueryDTO.getDataSourceSet().getDataSourceList().add(allDs);
                } else {
                    dataQueryDTO.getDataSourceSet().getDataSourceList().add(dataSourceDTOS.get(0));
                }
            }
            if (dataQueryDTO.getDataSourceSet().getDataSourceList().size() == 1) {
                dataQueryDTO.setDataSource(dataQueryDTO.getDataSourceSet().getDataSourceList().get(0));
                dataQueryDTO.getDataSourceSet().getDataSourceList().clear();
            }
        }
        Integer size = 0;
        Integer abnormalSize = 0;
        if (dataQueryDTO.getDataSourceSet().getDataSourceList() != null && dataQueryDTO.getDataSourceSet().getDataSourceList().size() > 1) {
            for (DataSourceDTO dataSource : dataQueryDTO.getDataSourceSet().getDataSourceList()) {
                if (dataQueryDTO.getDataSourceSet().getMainDatasource().equals(dataSource.getName())) {
                    Map<String, Integer> sizeMap = dataFetchingService.getDataSize(dataQueryDTO.getExecuteContext(), dataSource, dataQueryDTO.getParameter(), dataQueryDTO.getRowSizeType());
                    size += sizeMap.get("size");
                    abnormalSize += sizeMap.get("abnormalSize");
                }
            }
        } else {
            if (dataQueryDTO.getDataSourceSet().getMainDatasource().equals(dataQueryDTO.getDataSource().getName())) {
                Map<String, Integer> sizeMap = dataFetchingService.getDataSize(dataQueryDTO.getExecuteContext(), dataQueryDTO.getDataSource(), dataQueryDTO.getParameter(), dataQueryDTO.getRowSizeType());
                size = sizeMap.get("size");
                abnormalSize = sizeMap.get("abnormalSize");
            }
        }
        Map result = new HashMap();
        result.put("size", size);
        result.put("abnormalSize", abnormalSize);
        return result;
    }
}
