package com.digiwin.athena.atdm.datasource.datasource.process;

import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.atdm.UiBotConstants;
import com.digiwin.athena.atdm.aam.AttachmentEntity;
import com.digiwin.athena.atdm.aam.CommonAttachmentService;
import com.digiwin.athena.atdm.activity.domain.AttachmentConfig;
import com.digiwin.athena.atdm.activity.domain.AttachmentConfigInfo;
import com.digiwin.athena.atdm.constant.ErrorCodeEnum;
import com.digiwin.athena.atdm.datasource.BuildPageDataProcessService;
import com.digiwin.athena.atdm.datasource.datasource.DataSourceBase;
import com.digiwin.athena.atdm.datasource.datasource.DataSourceProcessService;
import com.digiwin.athena.atdm.datasource.domain.ApiMetadataCollection;
import com.digiwin.athena.atdm.datasource.domain.DataSourceProcessor;
import com.digiwin.athena.atdm.datasource.domain.ExecuteContext;
import com.digiwin.athena.atdm.datasource.domain.MetadataField;
import com.digiwin.athena.atdm.datasource.domain.QueryResult;
import com.digiwin.athena.atdm.datasource.domain.QueryResultSet;
import com.digiwin.athena.atdm.datasource.domain.TagConstant;
import com.digiwin.athena.atdm.datasource.domain.TagDefinition;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

@Slf4j
@Service(AttachmentDataSourceService.ATTACHMENT_DS_SERVICE_INSTANCE_NAME)
public class AttachmentDataSourceService implements BuildPageDataProcessService, DataSourceProcessService {
    public static final String ATTACHMENT_DS_SERVICE_INSTANCE_NAME = "attachmentDataService";

    @Autowired
    private CommonAttachmentService commonAttachmentService;

    @Autowired
    private MessageUtils messageUtils;

    private static final String SCHEMA = "attachment";

    @Override
    public void handelMetadata(DataSourceBase dataSourceBase, ExecuteContext executeContext, DataSourceProcessor dataSourceProcessor, QueryResult queryResult) {
        if (!AttachmentUtil.isPageCodeEnableAttachment(executeContext.getPageCode())) {
            return;
        }

        AttachmentConfigInfo attachmentConfigInfo = AttachmentUtil.toAttachment(dataSourceProcessor.getParas());
        handelMetadata(executeContext, attachmentConfigInfo, queryResult);
    }

    @Override
    public void handelData(DataSourceBase dataSourceBase, ExecuteContext executeContext, DataSourceProcessor dataSourceProcessor, QueryResult queryResult) {
        if (!AttachmentUtil.isPageCodeEnableAttachment(executeContext.getPageCode())) {
            return;
        }
        if (!AttachmentUtil.sourceTypeEnableAttachment(dataSourceBase.getType())) {
            return;
        }
        AttachmentConfigInfo attachmentConfigInfo = AttachmentUtil.toAttachment(dataSourceProcessor.getParas());
        handelData(executeContext, attachmentConfigInfo, queryResult);
    }

    @Override
    public void handelPageData(ExecuteContext executeContext, DataSourceProcessor dataSourceProcessor, QueryResultSet queryResultSet) {
        if (!AttachmentUtil.isPageCodeEnableAttachment(executeContext.getPageCode())) {
            return;
        }

        AttachmentConfigInfo attachmentConfigInfo = AttachmentUtil.toAttachment(dataSourceProcessor.getParas());
        handelMetadata(executeContext, attachmentConfigInfo, queryResultSet.getMainQueryResult());
        handelData(executeContext, attachmentConfigInfo, queryResultSet.getMainQueryResult());
    }

    private void handelMetadata(ExecuteContext executeContext, AttachmentConfigInfo attachmentConfigInfo, QueryResult queryResult) {
        addAttachmentFileMetadataField(queryResult.getApiMetadataCollection(), attachmentConfigInfo);
    }

    private void addAttachmentFileMetadataField(ApiMetadataCollection apiMetadataCollection, AttachmentConfigInfo attachmentConfigInfo) {
        // 添加多个Attachment MetadataField
        if (apiMetadataCollection == null) {
            return;
        }
        List<AttachmentConfig> attachmentConfigs = attachmentConfigInfo.getConfigs();
        for (int idx = 0; idx < attachmentConfigs.size(); idx++) {
            AttachmentConfig attachmentConfig = attachmentConfigs.get(idx);

            MetadataField attachmentField = new MetadataField();
            attachmentField.setUiBot("Y");
            /**
             * 如果指定了目标字段，就优先使用目标字段；否则自动生成schema
             * @author majianfu
             * @since 【迭代15 手动发起项目中控台】
             */
            if (StringUtils.isNotBlank(attachmentConfig.getTargetSchema())) {
                attachmentField.setName(attachmentConfig.getTargetSchema());
            } else {
                attachmentField.setName(SCHEMA + idx);
            }
            attachmentField.setDataType("object");

            String description = Optional.ofNullable(attachmentConfig.getDescription())
                    .map(AttachmentConfig.Description::getTitle)
                    .orElse(messageUtils.getMessage("uibot.activity.metadataField.attachment"));
            attachmentField.setDescription(description);
            attachmentField.setTagDefinitions(buildAttachmentFileTagDefs(description, description));

            //构造附件第三层元数据
            List<MetadataField> subFields = new ArrayList<>();
            MetadataField rowData = new MetadataField();
            rowData.setName("row_data");
            rowData.setDataType("string");
            rowData.setDataKey("false");
            rowData.setBusinessKey(false);
            rowData.setRequired(false);
            rowData.setArray(false);
            subFields.add(rowData);

            MetadataField data = new MetadataField();
            data.setName("data");
            data.setDataType("object");
            data.setDataKey("false");
            data.setBusinessKey(false);
            data.setRequired(false);
            data.setArray(true);
            data.setSubFields(new ArrayList<>());
            subFields.add(data);

            attachmentField.setSubFields(subFields);

            List<MetadataField> metadataFields = apiMetadataCollection.getMasterApiMetadata().getResponseFields().get(0).getSubFields();
            List<MetadataField> attachmentList = metadataFields.stream().filter(a -> attachmentField.getName().equals(a.getName())).collect(Collectors.toList());
            if (CollectionUtils.isNotEmpty(attachmentList)) {
                metadataFields.removeAll(attachmentList);
            }
            metadataFields.add(attachmentField);
        }
    }

    /**
     * 构建附件tag
     *
     * @return 附件tag列表
     */
    public static List<TagDefinition> buildAttachmentFileTagDefs(String tagName, String tagDescription) {
        List<TagDefinition> tagDefinitions = Lists.newArrayList();

        TagDefinition tagDefinition = new TagDefinition();
        tagDefinition.setCode(TagConstant.BUSINESS_ATTACHMENT_FILE);
        tagDefinition.setName(tagName);
        tagDefinition.setDescription(tagDescription);
        tagDefinition.setCategory(TagConstant.CATEGORY_BUSINESS);
        tagDefinition.setInterpreterServiceName("attachmentFileInterpreter");
        tagDefinition.setCustomize(false);

        tagDefinitions.add(tagDefinition);
        return tagDefinitions;
    }

    public void handelData(ExecuteContext executeContext, AttachmentConfigInfo attachmentConfigInfo, QueryResult queryResult) {
        List<Map<String, Object>> rowDataList = queryResult.getData();
        if (CollectionUtils.isEmpty(rowDataList)) {
            return;
        }

        List<String> rowDataKeyList = rowDataList.stream().map(rowData -> Optional.ofNullable(rowData.get(UiBotConstants.DATA_SOURCE_DATA_KEY)).map(Object::toString).orElse(""))
                .filter(StringUtils::isNotBlank)
                .collect(Collectors.toList());
        // 没有实际业务数据行 || attachment，也就没有rowDataKey || category，去捞数据了
        if (CollectionUtils.isEmpty(rowDataKeyList)) {
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0031.getErrCode(), messageUtils.getMessage("exception.api.config.lack.dataKey"));
        }

        // 构建查询条件
        String tenantId = executeContext.getTenantId();
        String projectId = executeContext.getTmProjectId();

        List<AttachmentConfig> attachmentConfigs = attachmentConfigInfo.getConfigs();
        for (int idx = 0; idx < attachmentConfigs.size(); idx++) {
            AttachmentConfig attachmentConfig = attachmentConfigs.get(idx);
            List<String> categories = AttachmentUtil.getQueryCategories(attachmentConfig);
            if (CollectionUtils.isEmpty(categories)) {
                continue;
            }

            // 启用aam
            if (null == attachmentConfig.getDisableAam() || BooleanUtils.isFalse(attachmentConfig.getDisableAam())) {
                List<AttachmentEntity> attachEntities = queryRowDataAttachmentList(tenantId, attachmentConfig.getTaskId(), projectId, categories, rowDataKeyList);
                // 为每行添加附件字段，附件字段名称：如果targetSchema值不为空，则使用targetSchema的值；否则使用（schema + 下标）
                String fieldName = StringUtils.isBlank(attachmentConfig.getTargetSchema()) ? SCHEMA + idx : attachmentConfig.getTargetSchema();
                putAttachmentToRowData(fieldName, rowDataList, attachEntities);
            }
            // 不启用aam
            else {
                String fieldName = StringUtils.isBlank(attachmentConfig.getTargetSchema()) ? SCHEMA + idx : attachmentConfig.getTargetSchema();
                for (Map<String, Object> rowData : rowDataList) {
                    // 附件字段值为null或者是个空Map，构造个空数据塞入进rowData
                    if (null == rowData.get(fieldName) ||
                            ((rowData.get(fieldName) instanceof Map) && MapUtils.isEmpty((Map) rowData.get(fieldName)))) {
                        String rowDataKey = Optional.ofNullable(rowData.get(UiBotConstants.DATA_SOURCE_DATA_KEY)).map(Object::toString).orElse("");

                        Map<String, Object> attachmentRowData = Maps.newHashMap();
                        attachmentRowData.put("row_data", rowDataKey);
                        attachmentRowData.put("data", Collections.emptyList());

                        rowData.put(fieldName, attachmentRowData);
                    }
                    // 附件字段值不为null且类型不是Map，则打印告警日志，不做任何操作
                    else if (null != rowData.get(fieldName) && !(rowData.get(fieldName) instanceof Map)) {
                        log.warn("附件日志{}作业的业务数据{}值不是Map类型，不符合附件数据格式规范", executeContext.getTmActivityId(), fieldName);
                    }
                }
            }
        }
    }

    private List<AttachmentEntity> queryRowDataAttachmentList(String tenantId, String taskId, String projectId, List<String> categories, List<String> rowDataKeyList) {
        return commonAttachmentService.queryRowDataAttachmentList(tenantId, taskId, projectId, categories, rowDataKeyList);
    }

    private void putAttachmentToRowData(String fieldName, List<Map<String, Object>> rowDataList, List<AttachmentEntity> attachEntities) {
        if (CollectionUtils.isEmpty(attachEntities)) {
            for (Map<String, Object> rowData : rowDataList) {
                String rowDataKey = Optional.ofNullable(rowData.get(UiBotConstants.DATA_SOURCE_DATA_KEY))
                        .map(Object::toString)
                        .orElse("");
                rowData.put(fieldName, buildAttachmentRowData(rowDataKey, null));
            }
        }

        // <rowDataKey, attachment entities>
        Map<String, List<AttachmentEntity>> rowDataKeyAttachEntitiesMap = Maps.newHashMapWithExpectedSize(attachEntities.size());
        for (AttachmentEntity attachEntity : attachEntities) {
            List<AttachmentEntity> rowAttachEntities = rowDataKeyAttachEntitiesMap.computeIfAbsent(attachEntity.getRowDataKey(), rowDataKey -> Lists.newArrayList());
            rowAttachEntities.add(attachEntity);
        }

        for (Map<String, Object> rowData : rowDataList) {
            String rowDataKey = Optional.ofNullable(rowData.get(UiBotConstants.DATA_SOURCE_DATA_KEY)).map(Object::toString).orElse("");
            List<AttachmentEntity> rowAttachEntities = rowDataKeyAttachEntitiesMap.get(rowDataKey);

            if (CollectionUtils.isNotEmpty(rowAttachEntities)) {
                List<AttachmentEntity> attachmentRowData = Lists.newArrayList();
                for (AttachmentEntity rowAttachEntity : rowAttachEntities) {
                    // 返回给前端的pageData中，只有id、name、category、categoryId、rowDataKey、uploadUserId、uploadUserName、size这7个属性
                    AttachmentEntity newAttachmentEntity = AttachmentEntity.builder()
                            .id(rowAttachEntity.getId())
                            .name(rowAttachEntity.getName())
                            .rowDataKey(rowAttachEntity.getRowDataKey())
                            .category(rowAttachEntity.getCategory())
                            .categoryId(rowAttachEntity.getCategoryId())
                            .uploadUserId(rowAttachEntity.getUploadUserId())
                            .uploadUserName(rowAttachEntity.getUploadUserName())
                            .defence(rowAttachEntity.getDefence())
                            .size(rowAttachEntity.getSize()).build();
                    newAttachmentEntity.setCreateDate(rowAttachEntity.getCreateDate());
                    attachmentRowData.add(newAttachmentEntity);
                }
                rowData.put(fieldName, buildAttachmentRowData(rowDataKey, attachmentRowData));
            } else {
                rowData.put(fieldName, buildAttachmentRowData(rowDataKey, null));
            }
        }
    }

    private Map<String, Object> buildAttachmentRowData(String rowDataKey, List<AttachmentEntity> data) {
        Map<String, Object> attachmentRowData = Maps.newHashMap();

        attachmentRowData.put("rowDataKey", rowDataKey);
        if (CollectionUtils.isNotEmpty(data)) {
            attachmentRowData.put("data", data);
        } else {
            attachmentRowData.put("data", Collections.emptyList());
        }

        return attachmentRowData;
    }
}
