package com.digiwin.athena.atdm.datasource;


import cn.hutool.core.util.NumberUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.JSONPath;
import com.alibaba.fastjson.TypeReference;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.atdm.datasource.datasource.DataQuery;
import com.digiwin.athena.atdm.datasource.datasource.DataSourceBase;
import com.digiwin.athena.atdm.datasource.datasource.DataSourceSet;
import com.digiwin.athena.atdm.datasource.datasource.converter.DataSourceConverter;
import com.digiwin.athena.atdm.dataUniformity.DataUniformityMongodbService;
import com.digiwin.athena.atdm.datasource.domain.DataSourceProcessor;
import com.digiwin.athena.atdm.datasource.domain.DetailPageInfo;
import com.digiwin.athena.atdm.datasource.domain.ExecuteContext;
import com.digiwin.athena.atdm.datasource.domain.QueryResult;
import com.digiwin.athena.atdm.datasource.domain.QueryResultSet;
import com.digiwin.athena.atdm.datasource.dto.DataQueryDTO;
import com.digiwin.athena.atdm.datasource.dto.DataSourceDTO;
import com.digiwin.athena.atdm.datasource.dto.DataSourceSetDTO;
import com.digiwin.athena.atdm.datasource.dto.PageInfo;
import com.digiwin.athena.atdm.datasource.datasource.process.dataUniformity.DataUnfiormityProcessor;
import com.digiwin.athena.atdm.datasource.datasource.process.dataUniformity.RetainAbnormalAndAddColumnService;
import com.digiwin.athena.atdm.datasource.datasource.process.dataUniformity.RetainAbnormalDataService;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

@Service
public class DataFetchingServiceImpl implements DataFetchingService {

    protected static final Map<String, Object> ROW_APPEND_FIELDS = new HashMap<>();

    static {
        ROW_APPEND_FIELDS.put("edit_type", 1);
    }

    ;

    @Autowired
    DataUniformityMongodbService dataUniformityMongodbService;

    @Autowired
    RetainAbnormalDataService retainAbnormalDataService;

    @Autowired
    DataUnfiormityProcessor dataUnfiormityProcessor;

    @Autowired
    private MessageUtils messageUtils;

    @Autowired
    RetainAbnormalAndAddColumnService retainAbnormalAndAddColumnService;

    @Override
    public QueryResultSet queryWithMetaData(ExecuteContext executeContext, DataSourceSetDTO dataSourceSetDTO, Map<String, Object> parameter, PageInfo pageInfo, List<Map> sortInfo, List<Map> searchInfo) {

        if (dataSourceSetDTO == null) {
            throw new IllegalArgumentException("dataSourceSetDTO");
        }
        DataSourceSet dataSourceSet = DataSourceConverter.convert(dataSourceSetDTO);
        if (dataSourceSet == null) {
            return QueryResultSet.empty();
        }

        return dataSourceSet.queryWithMetaData(executeContext, parameter, pageInfo, sortInfo, searchInfo);
    }


    @Override
    @Deprecated
    public int size(ExecuteContext executeContext, DataSourceDTO dataSourceDTO, Map<String, Object> parameter, String sizeType) {

        if (dataSourceDTO == null) {
            return 0;
        }
        DataSourceBase dataSourceBase = DataSourceConverter.convert(dataSourceDTO, true);
        return dataSourceBase.size(executeContext, parameter, sizeType);

    }

    @Override
    public int size(DataQueryDTO dataQuery) {
        return this.size(dataQuery, dataQuery.getDataSource());
    }

    @Override
    public int size(DataQueryDTO dataQueryDto, DataSourceDTO dataSource) {
        if (dataSource == null) {
            return 0;
        }
        DataSourceBase dataSourceBase = DataSourceConverter.convert(dataSource, true);
        DataQuery dataQuery = DataQuery.builder().executeContext(dataQueryDto.getExecuteContext()).parameter(dataQueryDto.getParameter()).pageInfo(dataQueryDto.getPageInfo()).sortInfo(dataQueryDto.getSortInfo()).searchInfo(dataQueryDto.getSearchInfo()).rowSizeType(dataQueryDto.getRowSizeType()).build();
        return dataSourceBase.size(dataQuery);
    }

    @Override
    public QueryResultSet query(ExecuteContext executeContext, DataSourceSetDTO dataSourceSetDTO, Map<String, Object> parameter, PageInfo pageInfo, List<Map> sortInfo, List<Map> searchInfo) {

        if (dataSourceSetDTO == null) {
            throw new IllegalArgumentException("dataSourceSetDTO");
        }
        DataSourceSet dataSourceSet = DataSourceConverter.convert(dataSourceSetDTO);
        if (dataSourceSet == null) {
            return QueryResultSet.empty();
        }

        QueryResultSet queryResultSet = dataSourceSet.query(executeContext, parameter, pageInfo, sortInfo, searchInfo);
        if (pageInfo != null && pageInfo.isNextAllData() && pageInfo.getPageSize() != null) {
            queryResultSet.getMainQueryResult().removeTop(pageInfo.getPageSize());
        }

        //基础料需要构建edit_type
        if ("DATA_ENTRY".equals(executeContext.getPattern())) {
            appendField(queryResultSet.getPageData());
        }
        return queryResultSet;
    }

    public void queryInfo(DataQueryDTO dataQueryDTO, QueryResultSet queryResultSet, Map result) {
        if (queryResultSet != null && queryResultSet.getMainQueryResult() != null) {
            if (dataQueryDTO.getPageInfo() != null) {
                PageInfo pageInfo = new PageInfo();
                pageInfo.setHasNext(queryResultSet.getMainQueryResult().isHasNext());
                pageInfo.setPageNo(dataQueryDTO.getPageInfo().getPageNo());
                pageInfo.setPageSize(dataQueryDTO.getPageInfo().getPageSize());
                pageInfo.setDetailName(dataQueryDTO.getPageInfo().getDetailName());
                pageInfo.setDetailPath(dataQueryDTO.getPageInfo().getDetailPath());
                Collection<Map<String, Object>> returnPageInfo = queryResultSet.getMainQueryResult().getReturnPageInfo();
                // 单身/子单身分页
                if (returnPageInfo != null) {
                    for (Map<String, Object> info : returnPageInfo) {
                        boolean o1 = Objects.equals(dataQueryDTO.getTempPageInfo().getDetailName(), info.get("detail_name"));
                        boolean o2 = Objects.equals(dataQueryDTO.getTempPageInfo().getDetailPath(), info.get("detail_path"));
                        if (o1 && o2) {
                            Object totalResults = info.get("total_results");
                            Object hasNext = info.get("has_next");
                            if (hasNext instanceof Boolean) {
                                pageInfo.setHasNext((Boolean) hasNext);
                            } else if (hasNext instanceof String) {
                                pageInfo.setHasNext(Boolean.getBoolean((String) hasNext));
                            }
                            if (totalResults instanceof Number) {
                                pageInfo.setTotalResults(NumberUtil.toBigDecimal((Number) totalResults).intValue());
                            } else if (totalResults instanceof String) {
                                pageInfo.setTotalResults(NumberUtil.toBigDecimal((String) totalResults).intValue());
                            } else {
                                pageInfo.setTotalResults(0);
                            }
                        }
                    }

                } else {
                    if (queryResultSet.getMainQueryResult().getTotalResults() != null) {
                        double totalResults = Math.round(queryResultSet.getMainQueryResult().getTotalResults());
                        pageInfo.setTotalResults((int) Math.floor(totalResults));
                    } else {
                        pageInfo.setTotalResults(0);
                    }
                }
                result.put("pageInfo", pageInfo);
            }
            if (CollectionUtils.isNotEmpty(dataQueryDTO.getSortInfo())) {
                result.put("sortInfo", dataQueryDTO.getSortInfo());
            }
            //返回搜索条件，用于前端展示
            if (CollectionUtils.isNotEmpty(dataQueryDTO.getSearchInfo())) {
                result.put("searchInfo", dataQueryDTO.getSearchInfo());
            }
        }
    }


    private void appendField(Map<String, Object> pageData) {
        if (MapUtils.isEmpty(pageData)) {
            return;
        }
        for (Object dataList : pageData.values()) {
            if (dataList instanceof List) {
                List rowDataList = (List) dataList;
                for (Object rowData : rowDataList) {
                    Map rowDataMap = (Map) rowData;
                    rowDataMap.putAll(ROW_APPEND_FIELDS);
                    if (rowDataMap.containsKey("manage_status") && rowDataMap.get("manage_status") == null) {
                        rowDataMap.put("manage_status", "N");
                    }
                }
            } else if (dataList instanceof Map) {
                Map rowDataMap = (Map) dataList;
                rowDataMap.putAll(ROW_APPEND_FIELDS);
            }
        }
    }

    @Override
    public QueryResult querySingleResultWithMetaData(ExecuteContext executeContext, DataSourceDTO dataSourceDTO, Map<String, Object> parameter, PageInfo pageInfo, List<Map> sortInfo, List<Map> searchInfo) {

        if (dataSourceDTO == null) {
            throw new IllegalArgumentException("dataSourceDTO");
        }
        DataSourceBase dataSourceBase = DataSourceConverter.convert(dataSourceDTO, true);

        return dataSourceBase.queryWithMetaData(executeContext, parameter, pageInfo, sortInfo, searchInfo);
    }

    @Override
    public QueryResult querySingleResult(ExecuteContext executeContext, DataSourceDTO dataSourceDTO, Map<String, Object> parameter, PageInfo pageInfo, List<Map> sortInfo, List<Map> searchInfo) {

        if (dataSourceDTO == null) {
            throw new IllegalArgumentException("dataSourceDTO");
        }
        DataSourceBase dataSourceBase = DataSourceConverter.convert(dataSourceDTO, true);

        return dataSourceBase.query(executeContext, parameter, pageInfo, sortInfo, searchInfo);
    }

    @Override
    public Map<String, Integer> getDataSize(ExecuteContext executeContext, DataSourceDTO dataSourceDTO, Map<String, Object> parameter, String sizeType) {
        if (dataSourceDTO == null) {
            Map<String, Integer> map = new HashMap<>();
            map.put("size", 0);
            map.put("abnormalSize", 0);
            return map;
        }
        DataSourceBase dataSourceBase = DataSourceConverter.convert(dataSourceDTO, true);
        return dataSourceBase.getDataSize(executeContext, parameter, sizeType);

    }

    @Override
    public void dealQueryResult(QueryResultSet queryResultSet, DataSourceProcessor dataSourceProcessor) {
        if (null == dataSourceProcessor) {
            return;
        }
        //合并数据
        groupQueryResult(queryResultSet);
        //处理异动数据
        QueryResult queryResult = queryResultSet.getQueryResults().get(0);
        if ("removeAbnormalDataService".equals(dataSourceProcessor.getServiceName())) {
            removeAbnormalData(queryResult, dataSourceProcessor);
        } else {
            retainAbnormalData(queryResult, dataSourceProcessor);
        }
    }

    /**
     * 合并数据
     *
     * @Author：SYQ
     * @Date：2022/5/25 14:08
     */
    private void groupQueryResult(QueryResultSet queryResultSet) {
        Map<String, List<QueryResult>> group = queryResultSet.getQueryResults().stream().collect(Collectors.groupingBy(QueryResult::getDataSourceName));
        for (Map.Entry<String, List<QueryResult>> groupItem : group.entrySet()) {
            List<QueryResult> resultList = groupItem.getValue();
            if (resultList.size() > 1) {
                QueryResult qr = resultList.get(0);
                for (int i = 1; i < resultList.size(); i++) {
                    qr.getData().addAll(resultList.get(i).getData());
                    qr.getDataKeyIndex().putAll(resultList.get(i).getDataKeyIndex());
                    //有可能第一个 QueryResult 没有查到数据，所以需要根据条件更新 DataKeys 和 ApiMetadataCollection
                    if (CollectionUtils.isEmpty(qr.getDataKeys()) && CollectionUtils.isNotEmpty(resultList.get(i).getDataKeys())) {
                        qr.setDataKeys(resultList.get(i).getDataKeys());
                    }
                    if (qr.getApiMetadataCollection() == null && resultList.get(i).getApiMetadataCollection() != null) {
                        qr.withMetaData(resultList.get(i).getApiMetadataCollection());
                    }
                    queryResultSet.getQueryResults().remove(resultList.get(i));
                }
                queryResultSet.setMainQueryResult(qr);
            }
        }
    }

    /**
     * 保留异动数据
     *
     * @Author：SYQ
     * @Date：2022/5/25 14:08
     */
    private void retainAbnormalData(QueryResult queryResult, DataSourceProcessor dataSourceProcessor) {
        List<Map<String, Object>> dataList = queryResult.getData();
        if (CollectionUtils.isEmpty(dataList) || null == dataSourceProcessor.getParas()) {
            return;
        }
        //获取异动数据
        Set<Map<String, Object>> abnormalDataSet = dataUnfiormityProcessor.comparisonData(dataList, dataSourceProcessor);
        List<Map<String, Object>> abnormalDataList = new ArrayList<>();
        //去除处理方案为结束的
        for (Map<String, Object> dataMap : abnormalDataSet) {
            //操作方式
            String optTypeName = "0".equals(dataMap.get("optType")) ? messageUtils.getMessage("data_uniformity_optType_update") : messageUtils.getMessage("data_uniformity_optType_remove");
            dataMap.put("data_uniformity_optType", optTypeName);
            dataMap.put("data_uniformity_lastModifyTime", dataMap.get("lastModifyTime"));
            dataMap.put("data_uniformity_dealType", "");
            abnormalDataList.add(dataMap);
        }
        queryResult.setData(abnormalDataList);
        queryResult.initializeDataKey();
    }

    @Override
    public DataSourceProcessor judgeAbnormalProcessor(DataQueryDTO dataQueryDTO) {
        DataSourceProcessor dataSourceProcessor = null;
        if (dataQueryDTO == null || dataQueryDTO.getDataSourceSetDTO() == null || CollectionUtils.isEmpty(dataQueryDTO.getDataSourceSetDTO().getDataSourceList())) {
            return null;
        }
        DataSourceDTO dataSourceDTO = dataQueryDTO.getDataSourceSetDTO().getDataSourceList().get(0);
        List<DataSourceProcessor> dataSourceProcessorList = dataSourceDTO.getDataSourceProcessors();
        if (CollectionUtils.isNotEmpty(dataSourceProcessorList)) {
            boolean isRemoveProcessor = false;
            for (DataSourceProcessor processor : dataSourceProcessorList) {
                if ("removeAbnormalDataService".equals(processor.getServiceName()) || "retainAbnormalDataService".equals(processor.getServiceName())) {
                    isRemoveProcessor = true;
                    dataSourceProcessor = processor;
                    break;
                } else if ("retainAbnormalAndAddColumnService".equals(processor.getServiceName())) {
                    dataSourceProcessor = processor;
                    break;
                }
            }
            if (isRemoveProcessor) {
                dataSourceProcessorList.remove(dataSourceProcessor);
            }
        }
        return dataSourceProcessor;
    }

    /**
     * 删除异动数据
     *
     * @Author：SYQ
     * @Date：2022/5/26 12:04
     */
    private void removeAbnormalData(QueryResult queryResult, DataSourceProcessor dataSourceProcessor) {
        List<Map<String, Object>> dataList = queryResult.getData();
        if (CollectionUtils.isEmpty(dataList) || null == dataSourceProcessor.getParas()) {
            return;
        }
        //获取异动数据
        Set<Map<String, Object>> abnormalDataSet = dataUnfiormityProcessor.comparisonData(dataList, dataSourceProcessor);
        //删除异动数据
        List<Map<String, Object>> resultList = new ArrayList<>();

        for (Map<String, Object> dataMap : dataList) {
            if (!abnormalDataSet.contains(dataMap)) {
                resultList.add(dataMap);
            }
        }

        int totalKeySize = queryResult.keySize();

        queryResult.setData(resultList);
        queryResult.initializeDataKey();
        queryResult.setAbnormalDataSize(abnormalDataSet.size());
        queryResult.setAbnormalDataKeyIndexSize(totalKeySize - queryResult.keySize());
    }

    @Override
    public void dealDataQueryInfoBeforeQuery(DataQueryDTO dataQuery) {
        Map<String, Object> returnRequestData = dataQuery.getReturnRequestData();
        if (MapUtils.isNotEmpty(returnRequestData)) {
            dataQuery.setTempPageInfo(dataQuery.getPageInfo());
            // 替换分页信息
            PageInfo previousPageInfo = JSON.parseObject(JSON.toJSONString(returnRequestData.get("pageInfo")), PageInfo.class);
            List<DetailPageInfo> detailPageInfos = previousPageInfo.getDetailPageInfo();
            if (detailPageInfos == null) {
                detailPageInfos = new ArrayList<>();
            }
            PageInfo tempPageInfo = dataQuery.getTempPageInfo();

            String detailPath = StringUtils.defaultIfBlank(tempPageInfo.getDetailPath(), "");
            String detailName = StringUtils.defaultIfBlank(tempPageInfo.getDetailName(), "");
            Optional<DetailPageInfo> optionalDetailPageInfo = detailPageInfos.stream().filter(o -> Objects.equals(StringUtils.defaultIfBlank(o.getDetailPath(), ""), detailPath) && Objects.equals(detailName, StringUtils.defaultIfBlank(o.getDetailName(), ""))).findFirst();
            List<String> sonPath = new ArrayList<>();
            if (optionalDetailPageInfo.isPresent()) {
                DetailPageInfo detailPageInfo = optionalDetailPageInfo.get();
                sonPath.add(detailPageInfo.getDetailName());
                // sorted.add(0, detailPageInfo.getDetailPath());

                while (true) {
                    Optional<DetailPageInfo> parent = detailPageInfos.stream().filter(o -> Objects.equals(StringUtils.defaultIfBlank(o.getDetailName(), ""), sonPath.get(sonPath.size() - 1))).findFirst();
                    if (parent.isPresent()) {
                        DetailPageInfo detailPageInfo1 = parent.get();
                        String detailPath1 = detailPageInfo1.getDetailPath();
                        if (StringUtils.isBlank(detailPath1)) {
                            break;
                        } else {
                            sonPath.add(detailPath1);
                        }
                    } else {
                        break;
                    }
                }
            }
            if (sonPath.size() > 0) {
                sonPath.remove(0);
            }

            replacePageInfo(previousPageInfo, dataQuery.getPageInfo(), sonPath);
            dataQuery.setPageInfo(previousPageInfo);
            // 替换排序信息
            List<Map> previousSortInfo = JSON.parseObject(JSON.toJSONString(returnRequestData.get("sortInfo")), new TypeReference<List<Map>>() {
            });
            previousSortInfo = replaceSortInfo(previousSortInfo, dataQuery.getSortInfo(), sonPath);
            dataQuery.setSortInfo(previousSortInfo);
            // 替换搜索信息
            List<Map> previousSearchInfo = JSON.parseObject(JSON.toJSONString(returnRequestData.get("searchInfo")), new TypeReference<List<Map>>() {
            });
            previousSearchInfo = replaceSearchInfo(previousSearchInfo, dataQuery.getSearchInfo(), sonPath);
            dataQuery.setSearchInfo(previousSearchInfo);
        }

    }

    private List<Map> replaceSearchInfo(List<Map> previousSortInfo, List<Map> searchInfo, List<String> sonPath) {
        if (searchInfo == null) {
            searchInfo = new ArrayList<>();
        }
        if (previousSortInfo == null) {
            previousSortInfo = new ArrayList<>();
        }
        previousSortInfo.removeIf(next -> sonPath.contains(StringUtils.defaultIfBlank(String.valueOf(next.get("searchFieldPath")), "")));
        previousSortInfo.addAll(searchInfo);
        return previousSortInfo;
    }

    private List<Map> replaceSortInfo(List<Map> previousSortInfo, List<Map> sortInfo, List<String> sonPath) {
        if (sortInfo == null) {
            sortInfo = new ArrayList<>();
        }
        if (previousSortInfo == null) {
            previousSortInfo = new ArrayList<>();
        }
        previousSortInfo.removeIf(next -> sonPath.contains(StringUtils.defaultIfBlank(String.valueOf(next.get("sortFieldPath")), "")));
        previousSortInfo.addAll(sortInfo);
        return previousSortInfo;
    }

    private void replacePageInfo(PageInfo previousPageInfo, PageInfo paramPageInfo, List<String> sonPath) {
        // 兼容之前页面开启分页。但是表格上未配置分页信息。导致 return_request_date 中没有detailName 信息
        if (StringUtils.isBlank(previousPageInfo.getDetailName()) && StringUtils.isBlank(previousPageInfo.getDetailPath())) {
            previousPageInfo.setPageNo(paramPageInfo.getPageNo());
            previousPageInfo.setPageSize(paramPageInfo.getPageSize());
        }
        // 找到上次的 单头 分页信息
        if (Objects.equals(StringUtils.defaultIfBlank(paramPageInfo.getDetailPath(), ""), StringUtils.defaultIfBlank(previousPageInfo.getDetailPath(), "")) && Objects.equals(StringUtils.defaultIfBlank(paramPageInfo.getDetailName(), ""), StringUtils.defaultIfBlank(previousPageInfo.getDetailName(), ""))) {
            previousPageInfo.setPageNo(paramPageInfo.getPageNo());
            previousPageInfo.setPageSize(paramPageInfo.getPageSize());
        }
        // 单身、子单身
        List<DetailPageInfo> detailPageInfo = previousPageInfo.getDetailPageInfo();
        if (CollectionUtils.isNotEmpty(detailPageInfo)) {
            paramPageInfo.setDetailPageInfo(detailPageInfo);
            for (DetailPageInfo tempPageInfo : detailPageInfo) {
                if (Objects.equals(paramPageInfo.getDetailPath(), tempPageInfo.getDetailPath()) && Objects.equals(paramPageInfo.getDetailName(), tempPageInfo.getDetailName())) {
                    tempPageInfo.setPageNo(paramPageInfo.getPageNo());
                    tempPageInfo.setPageSize(paramPageInfo.getPageSize());
                }
                // 查询单身时，子单身查询第一页
                if (sonPath.contains(StringUtils.defaultIfBlank(tempPageInfo.getDetailPath(), "")) && sonPath.contains(StringUtils.defaultIfBlank(tempPageInfo.getDetailName(), ""))) {
                    tempPageInfo.setPageNo(1);
                }
            }
        }

    }

    @Override
    public void dealPageDataAfterQuery(DataQueryDTO dataQuery, QueryResultSet queryResultSet, Map result) {
        Map<String, Object> returnRequestData = dataQuery.getReturnRequestData();
        if (returnRequestData != null && queryResultSet.getMainQueryResult() != null) {

            JSONObject jsonObject = new JSONObject(returnRequestData);
            List<DetailPageInfo> detailPageInfos = JSON.parseObject(JSON.toJSONString(JSONPath.eval(jsonObject, "$.pageInfo.detailPageInfo")), new TypeReference<List<DetailPageInfo>>() {
            });
            if (CollectionUtils.isNotEmpty(detailPageInfos)) {
                PageInfo tempPageInfo = dataQuery.getTempPageInfo();
                String detailPath = StringUtils.defaultIfBlank(tempPageInfo.getDetailPath(), "");
                String detailName = StringUtils.defaultIfBlank(tempPageInfo.getDetailName(), "");
                Optional<DetailPageInfo> optionalDetailPageInfo = detailPageInfos.stream().filter(o -> Objects.equals(StringUtils.defaultIfBlank(o.getDetailPath(), ""), detailPath) && Objects.equals(detailName, StringUtils.defaultIfBlank(o.getDetailName(), ""))).findFirst();
                if (optionalDetailPageInfo.isPresent()) {
                    DetailPageInfo detailPageInfo = optionalDetailPageInfo.get();
                    List<String> sorted = new ArrayList<>();
                    sorted.add(detailPageInfo.getDetailName());
                    sorted.add(0, detailPageInfo.getDetailPath());

                    while (true) {
                        Optional<DetailPageInfo> parent = detailPageInfos.stream().filter(o -> Objects.equals(StringUtils.defaultIfBlank(o.getDetailName(), ""), sorted.get(0))).findFirst();
                        if (parent.isPresent()) {
                            DetailPageInfo detailPageInfo1 = parent.get();
                            String detailPath1 = detailPageInfo1.getDetailPath();
                            if (StringUtils.isBlank(detailPath1)) {
                                break;
                            } else {
                                sorted.add(0, detailPath1);
                            }
                        } else {
                            break;
                        }
                    }
                    Map<String, Object> pageData = (Map<String, Object>) result.get("pageData");
                    List<String> pathList = sorted.stream().filter(StringUtils::isNotBlank).collect(Collectors.toList());
                    if (pathList.size() > 0 && pageData != null) {

                        Object o = pageData.get(pathList.get(0));
                        for (int i = 1; i < pathList.size(); i++) {
                            if (o instanceof Collection) {
                                o = getObject(o, pathList.get(i));

                            } else if (o instanceof Map) {
                                o = ((Map) o).get(pathList.get(i));
                            }
                        }
                        pageData.put(pathList.get(pathList.size() - 1), o);

                    }
                }

            }
        }


    }

    private Object getObject(Object o, String path) {
        while (o instanceof Collection) {
            o = ((Collection) o).toArray()[0];
        }
        if (o instanceof Map) {
            o = ((Map) o).get(path);
        }
        return o;
    }
}
