package com.digiwin.athena.atdm.datasource.datasource;

import cn.hutool.core.map.MapUtil;
import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.SpringUtil;
import com.digiwin.athena.atdm.UiBotConstants;
import com.digiwin.athena.atdm.constant.ErrorCodeEnum;
import com.digiwin.athena.atdm.datasource.BuildPageDataProcessService;
import com.digiwin.athena.atdm.datasource.domain.*;
import com.digiwin.athena.atdm.datasource.dto.PageInfo;
import com.fasterxml.jackson.annotation.JsonIgnore;
import lombok.Data;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;

@Data
public class DataSourceSet {
    private Logger logger = LoggerFactory.getLogger(DataSourceSet.class);

    public static final Map<String, Object> COMMON_FIELD_MAP = MapUtil
            .builder(new HashMap<String, Object>())
            .put("uibot_data_status", "not-changed") //标记数据状态  未修改not-changed   已修改changed   删除delete； 新增数据new-data。
            .build();
    /**
     * 查询的数据源
     */
    private List<DataSourceBase> dataSourceList;

    private String mainDatasource;

    /**
     * 全局数据处理器
     */
    private List<DataSourceProcessor> dataProcess;

    public static DataSourceSet create(DataSourceBase dataSourceBase) {
        DataSourceSet dataSourceSet = new DataSourceSet();
        List<DataSourceBase> dataSourceList = new ArrayList<>();
        dataSourceList.add(dataSourceBase);
        dataSourceSet.setDataSourceList(dataSourceList);
        return dataSourceSet;
    }

    public DataSourceSet append(DataSourceBase dataSourceBase) {
        if (dataSourceList == null) {
            this.dataSourceList = new ArrayList<>();
        }
        dataSourceList.add(dataSourceBase);
        return this;
    }

    /**
     * 查询数据和元数据
     *
     * @param executeContext 执行上下文
     * @param parameter      执行参数
     * @return
     */
    public QueryResultSet queryWithMetaData(ExecuteContext executeContext, Map<String, Object> parameter, PageInfo pageInfo, List<Map> sortInfo, List<Map> searchInfo) {
        if (CollectionUtils.isEmpty(dataSourceList)) {
            return QueryResultSet.empty();
        }
        List<QueryResult> queryResults = new ArrayList<>();
        if (dataSourceList.size() == 1) {
            queryResults.add(dataSourceList.get(0).queryWithMetaData(executeContext, parameter, pageInfo, sortInfo, searchInfo));
        } else {
            try {
                queryResults = ParallelQueryTaskUtils.queryWithMetaData(executeContext, parameter, dataSourceList, pageInfo, sortInfo, searchInfo);
            } catch (ExecutionException | TimeoutException | InterruptedException e) {//NOSONAR

                if (e.getCause() instanceof BusinessException) {
                    throw (BusinessException) e.getCause();
                }
                logger.error("多线程处理出问题:", e);
                throw BusinessException.create(ErrorCodeEnum.NUM_500_0047.getErrCode(), "多线程处理出问题", e);
            }
        }
        QueryResultSet queryResultSet = getQueryResultSet(executeContext, queryResults);
        return queryResultSet;
    }

    /**
     * 查询数据
     *
     * @param executeContext 执行上下文
     * @param parameter      参数
     * @return
     */
    public QueryResultSet query(ExecuteContext executeContext, Map<String, Object> parameter, PageInfo pageInfo, List<Map> sortInfo, List<Map> searchInfo) {
        if (CollectionUtils.isEmpty(dataSourceList)) {
            return QueryResultSet.empty();
        }
        List<QueryResult> queryResults = new ArrayList<>();

        if (dataSourceList.size() == 1) {
            queryResults.add(dataSourceList.get(0).query(executeContext, parameter, pageInfo, sortInfo, searchInfo));
        } else {
            try {
                queryResults = ParallelQueryTaskUtils.query(executeContext, parameter, dataSourceList, pageInfo, sortInfo, searchInfo);
            } catch (ExecutionException | TimeoutException | InterruptedException e) {//NOSONAR
                if (e.getCause() instanceof BusinessException) {
                    throw (BusinessException) e.getCause();
                }
                logger.error("多线程处理出问题:", e);
                throw BusinessException.create(ErrorCodeEnum.NUM_500_0046.getErrCode(), "多线程处理出问题", e);
            }
        }

        QueryResultSet queryResultSet = getQueryResultSet(executeContext, queryResults);

        return queryResultSet;
    }


    private QueryResultSet getQueryResultSet(ExecuteContext executeContext, List<QueryResult> queryResults) {
        QueryResultSet queryResultSet = QueryResultSet.withData(this.mainDatasource, queryResults);

        if (CollectionUtils.isNotEmpty(this.dataProcess)) {
            for (DataSourceProcessor dataSourceProcessor : dataProcess) {
                if (UiBotConstants.DATA_PROCESS_ACTIVE_POINT_EXECUTE_COMPLETED.equals(dataSourceProcessor.getActivePoint())) {
                    BuildPageDataProcessService buildPageDataProcessService = (BuildPageDataProcessService) SpringUtil.tryGetBean(dataSourceProcessor.getServiceName());
                    buildPageDataProcessService.handelPageData(executeContext, dataSourceProcessor, queryResultSet);
                }
            }
        }

        // pageData 新增通用字段
        appendCommonField(queryResultSet.getPageData());
        return queryResultSet;
    }

    /**
     * pageData 新增通用字段
     * @param pageData
     */
    private void appendCommonField(Map<String, Object> pageData) {
        if (MapUtils.isEmpty(pageData)) {
            return;
        }
        for (Object dataList : pageData.values()) {
            if (dataList instanceof List) {
                List rowDataList = (List) dataList;
                for (Object rowData : rowDataList) {
                    Map rowDataMap = (Map) rowData;
                    rowDataMap.putAll(COMMON_FIELD_MAP);
                }
            } else if(dataList instanceof Map)  {
                Map rowDataMap =  (Map)dataList;
                rowDataMap.putAll(COMMON_FIELD_MAP);
            }
        }
    }

    @JsonIgnore
    public String getFirstActionId() {
        if (CollectionUtils.isEmpty(dataSourceList)) {
            return null;
        }
        return this.dataSourceList.get(0).getActionId();
    }

    @JsonIgnore
    public boolean isEmpty() {
        return CollectionUtils.isEmpty(dataSourceList);
    }

    @JsonIgnore
    public DataSourceBase getFirstDataQuery() {
        if (CollectionUtils.isEmpty(dataSourceList)) {
            return null;
        }
        return this.dataSourceList.get(0);
    }

    @JsonIgnore
    public QueryAction getFirstAction() {
        if (CollectionUtils.isEmpty(dataSourceList)) {
            return null;
        }
        return this.dataSourceList.get(0).getAction();
    }
}
