package com.digiwin.athena.atdm.datasource.datasource.process;

import com.digiwin.athena.appcore.exception.BusinessException;
import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.appcore.util.SpringUtil;
import com.digiwin.athena.atdm.UiBotConstants;
import com.digiwin.athena.atdm.activity.ActivityConstants;
import com.digiwin.athena.atdm.atmc.CommonAtmcService;
import com.digiwin.athena.atdm.constant.ErrorCodeEnum;
import com.digiwin.athena.atdm.datasource.datasource.DataSourceProcessService;
import com.digiwin.athena.atdm.datasource.datasource.DataSourceBase;
import com.digiwin.athena.atdm.datasource.domain.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;

@Service("atmcBatchDataService")
@Slf4j
public class AtmcBatchDataSourceService implements DataSourceProcessService {

    @Autowired
    private MessageUtils messageUtils;
    @Autowired
    private CommonAtmcService atmcService;

    @Value("${process.atmcBatch.dealAsync:true}")
    private String dealAsync;

    /**
     * 对元数据处理
     *
     * @param dataSourceBase      查询的结果的数据
     * @param executeContext      执行上下文
     * @param dataSourceProcessor 处理器
     * @param queryResult         查询结果
     */
    @Override
    public void handelMetadata(DataSourceBase dataSourceBase, ExecuteContext executeContext,
                               DataSourceProcessor dataSourceProcessor, QueryResult queryResult) {
        if (!ActivityConstants.TASK_DETAIL.equals(executeContext.getPageCode())
                && !ActivityConstants.PROJECT_DETAIL.equals(executeContext.getPageCode())) {
            return;
        }
        addAtmcFiled(dataSourceBase, executeContext, dataSourceProcessor, queryResult.getApiMetadataCollection());
    }


    /**
     * @param dataSourceProcessor DataSourceProcessor
     * @param executeContext      ExecuteContext
     * @return void
     * @description 开启缩减报文的配置后，对非移动端的请求，补充
     */
    private void addBatchProcessorDataForDatasource(DataSourceProcessor dataSourceProcessor, ExecuteContext executeContext) {
        try {
            //开启了简化报文：executeContext中需传入isSimplified=true
            boolean isSimplified = Boolean.TRUE.equals(executeContext.getEnableSimplifyPack());
            //标准前端判断条件：executeContext中需传入agentType=standardweb
            boolean isStandardWeb = UiBotConstants.STANDARD_WEB.equals(executeContext.getAgentType());
            //标准前端，开启了缩减大报文的情况下，如果是atmcBatchDataService类型的processor，需要查一下流程数据
            if (!CollectionUtils.isEmpty(dataSourceProcessor.getTaskWithBacklogDataList())) {
                return;
            }
            if ((isSimplified && isStandardWeb) || Boolean.TRUE.equals(dataSourceProcessor.getHasTaskData())) {
                List<TaskWithBacklogData> taskWithBacklogDataList = atmcService.getTaskData(executeContext.getPtmBacklogId(), executeContext).getTasks();
                dataSourceProcessor.setTaskWithBacklogDataList(taskWithBacklogDataList);
            }
        } catch (Exception e) {
            log.warn("addBatchProcessorDataForDatasource exception {}", e.getMessage());
        }
    }


    /**
     * 对数据处理
     *
     * @param dataSourceBase
     * @param executeContext
     * @param dataSourceProcessor 处理器
     * @param queryResult
     */
    @Override
    public void handelData(DataSourceBase dataSourceBase, ExecuteContext executeContext,
                           DataSourceProcessor dataSourceProcessor, QueryResult queryResult) {
        if (!ActivityConstants.TASK_DETAIL.equals(executeContext.getPageCode())
                && !ActivityConstants.PROJECT_DETAIL.equals(executeContext.getPageCode())) {
            return;
        }
        if (CollectionUtils.isEmpty(queryResult.getData())) {
            return;
        }

        //add by gonghongxing at 2022/03/26  for 数据源合并处理优化 start
        //如果没有定义mapKeys则直接返回
        if (Objects.isNull(dataSourceProcessor.getMappingKeys())) {
            return;
        }


        addBatchProcessorDataForDatasource(dataSourceProcessor, executeContext);

        //如果没有定义任务数组则直接返回
        if (CollectionUtils.isEmpty(dataSourceProcessor.getTaskWithBacklogDataList())) {
            return;
        }
        //add by gonghongxing at 2022/03/26  for 数据源合并处理优化 end

        //创建结果和查询数据的索引
        String mainDataName = dataSourceProcessor.getMappingKeys().getDataName();
        List<String> keyList = parseMappingKeys(dataSourceProcessor.getMappingKeys());
        String detailField = null;
        if (dataSourceProcessor.getParas() != null) {
            Map<String, String> processParas = (Map<String, String>) dataSourceProcessor.getParas();
            if (processParas.containsKey("detailField")) {
                detailField = processParas.get("detailField");
            }
        }
        //modify by gonghongxing at 2022/03/26  for 数据源合并处理优化 start
        //获取批量的任务数据
        List<TaskWithBacklogData> taskWithBacklogDataLists = dataSourceProcessor.getTaskWithBacklogDataList();
        Map<String, TaskWithBacklogData> taskWithBacklogIndexMap = dealBpmDataByMapKeys(keyList, mainDataName, taskWithBacklogDataLists);

        final String detailFieldFinal = detailField;
        if (Boolean.TRUE.equals(Boolean.valueOf(dealAsync))) {
            dealMultiPartAsync(executeContext, queryResult.getData(), taskWithBacklogIndexMap, keyList, detailFieldFinal);
        } else {
            dealResultData(executeContext, queryResult.getData(), taskWithBacklogIndexMap, keyList, detailFieldFinal);
        }


        for (TaskWithBacklogData taskWithBacklogData : taskWithBacklogDataLists) {
            if ("SOLVE".equals(executeContext.getCategory())
                    && "waitting".equals(executeContext.getDataStateCode())
                    && taskWithBacklogData.getBpmData() != null
                    && taskWithBacklogData.getBpmData().containsKey("checkResult")
                    && Objects.nonNull(dataSourceBase)) {
                addCheckResult(dataSourceBase, queryResult, taskWithBacklogData);
            }
        }
    }

    /**
     * @param executeContext          ExecuteContext
     * @param data                    List<Map<Object>>
     * @param taskWithBacklogIndexMap Map<TaskWithBacklogData>
     * @param keyList                 List<String>
     * @param detailFieldFinal        String
     * @return void
     * @description 分批异步处理
     */
    private void dealMultiPartAsync(ExecuteContext executeContext, List<Map<String, Object>> data, Map<String, TaskWithBacklogData> taskWithBacklogIndexMap, List<String> keyList, String detailFieldFinal) {
        ExecutorService executorService = (ExecutorService) SpringUtil.getBean("asyncServiceExecutor");
        //批量大小 500
        int batchsize = 500;
        int maxBatchNum = 50;
        int totalDataSize = data.size();
        //最多50个线程
        if (totalDataSize / batchsize > maxBatchNum) {
            batchsize = totalDataSize / maxBatchNum;
        }
        int mod = totalDataSize % batchsize;
        //批次
        int batchNum = totalDataSize / batchsize + (mod > 0 ? 1 : 0);
        CompletableFuture[] completableFutureList = new CompletableFuture[batchNum];
        for (int i = 0; i < batchNum; i++) {
            int fromIdx = i * batchsize;
            int toIdx = fromIdx + ((i == batchNum - 1 && mod > 0) ? mod : batchsize);
            CompletableFuture completableFuture = CompletableFuture.runAsync(() -> {
                this.dealResultData(executeContext, data.subList(fromIdx, toIdx), taskWithBacklogIndexMap, keyList, detailFieldFinal);
            }, executorService);
            completableFutureList[i] = completableFuture;
        }
        log.info("[AtmcBatchDataSourceService]批量异步处理，总数[{}],批数[{}],每批大小[{}]", totalDataSize, batchNum, batchsize);
        CompletableFuture<Void> allFuture = CompletableFuture.allOf(completableFutureList);
        try {
            allFuture.get();
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        } catch (ExecutionException e) {
            log.error("异步执行futureDealResultData操作异常[{}]", e);
            throw BusinessException.create(ErrorCodeEnum.NUM_500_0118.getErrCode(), messageUtils.getMessage("delivery.AtmcBatchDataSourceService.dealAsyncError"));
        }
    }

    /**
     * 添加异常排除的数据到结果中
     */
    private void addCheckResult(DataSourceBase dataSourceBase, QueryResult queryResult,
                                TaskWithBacklogData taskWithBacklogData) {
        // 如果有推送异常原因（checkResult），则添加到数据源里
        List<Map> checkResult = (List) taskWithBacklogData.getBpmData().get("checkResult");
        // 如果合并的数据源有dataKey，则通过这个key去主数据源中找数据
        if (!CollectionUtils.isEmpty(dataSourceBase.getDataKeys())) {
            StringBuilder stringBuilder = new StringBuilder();
            // 如果不一致，则需要遍历
            for (Map<String, Object> datum : checkResult) {
                stringBuilder.setLength(0);
                for (String key : dataSourceBase.getDataKeys()) {
                    if (datum.containsKey(key)) {
                        stringBuilder.append(StringUtils.isEmpty(datum.get(key)) ? "NAN" : datum.get(key).toString()).append(";");
                    } else {
                        stringBuilder.append("NAN;");
                    }
                }
                if (stringBuilder.length() > 0) {
                    String key = stringBuilder.toString();
                    Map<String, Object> findItem = queryResult.findByKey(key);
                    if (findItem != null) {
                        findItem.put("check_result", datum.get("result"));
                    }
                }
            }
        }
    }


    /**
     * 只有当责者的时候才出现ATMC数据 添加Activity字段
     *
     * @param apiMetadataCollection
     */
    private void addAtmcFiled(DataSourceBase dataSourceBase, ExecuteContext executeContext, DataSourceProcessor dataSourceProcessor,
                              ApiMetadataCollection apiMetadataCollection) {
        if (apiMetadataCollection == null
                || apiMetadataCollection.getMasterApiMetadata() == null
                || apiMetadataCollection.getMasterApiMetadata().getResponseFields() == null
                || apiMetadataCollection.getMasterApiMetadata().getResponseFields().size() == 0) {
            return;
        }

        // 当前关的处理人
        MetadataField performerNameMetadataField = new MetadataField();
        performerNameMetadataField.setUiBot("Y");
        performerNameMetadataField.setName("activity__performerName");
        performerNameMetadataField.setDataType("string");
        performerNameMetadataField
                .setDescription(messageUtils.getMessage("uibot.activity.metadataField.performerName"));

        MetadataField planEndTimeMetadataField = new MetadataField();
        planEndTimeMetadataField.setUiBot("Y");
        planEndTimeMetadataField.setName("activity__planEndTime");
        planEndTimeMetadataField.setDataType("string");
        planEndTimeMetadataField.setDescription(messageUtils.getMessage("uibot.activity.metadataField.planEndTime"));

        MetadataField readCountMetadataField = new MetadataField();
        readCountMetadataField.setUiBot("Y");
        readCountMetadataField.setName("activity__readCount");
        readCountMetadataField.setDataType("string");
        readCountMetadataField.setDescription(messageUtils.getMessage("uibot.activity.metadataField.readCount"));

        MetadataField dataStatusMetadataField = new MetadataField();
        dataStatusMetadataField.setUiBot("Y");
        dataStatusMetadataField.setName("activity__backLog__data");
        dataStatusMetadataField.setDataType("object");
        dataStatusMetadataField.setDescription(messageUtils.getMessage("uibot.activity.metadataField.dataStatus"));
        List<MetadataField> metadataFields = apiMetadataCollection.getMasterApiMetadata().getResponseFields().get(0)
                .getSubFields();

        metadataFields.add(performerNameMetadataField);
        metadataFields.add(planEndTimeMetadataField);
        metadataFields.add(readCountMetadataField);
        metadataFields.add(dataStatusMetadataField);

        // 添加任务的元数据字段
        ActivityUtils.addActivityMetadataField(metadataFields, messageUtils);

        // p5s1 9100 添加“数据流转状态”列
        if (!UiBotConstants.ACTION_CATEGORY_SHELVED_DATA.equals(dataSourceBase.getType())) {
            metadataFields.add(createDataFlowStatusField());
        }

        //修改逻辑,控制元数据的处理，只有一份，通过每次调用setter来覆盖原来的，保证只有一份最新
        MetadataField prePerformerNameMetadataField = null;

        MetadataField prePerformerIdMetadataField = null;

        MetadataField dataTranceStatusMetadataField = null;

        MetadataField checkResultMetadataField = null;

        if (CollectionUtils.isEmpty(dataSourceProcessor.getTaskWithBacklogDataList())) {
            return;
        }
        List<TaskWithBacklogData> taskWithBacklogDataList = dataSourceProcessor.getTaskWithBacklogDataList();
        for (TaskWithBacklogData taskWithBacklogData : taskWithBacklogDataList) {
            if ("SOLVE".equals(executeContext.getCategory()) || ("APPROVAL".equals(executeContext.getCategory()) && executeContext.getIdentity().equals("charge"))) {
                if (!CollectionUtils.isEmpty(taskWithBacklogData.getBacklog())) {
                    BacklogData firstBacklogData = taskWithBacklogData.getBacklog().get(0);
                    if ((firstBacklogData.getPrePerformerId() != null
                            || firstBacklogData.getPreAgentPerformerId() != null)) {
                        // 上一个的处理人id和name
                        prePerformerNameMetadataField = new MetadataField();
                        prePerformerNameMetadataField.setUiBot("Y");
                        prePerformerNameMetadataField.setName("activity__prePerformer_Name");
                        prePerformerNameMetadataField.setDataType("string");
                        prePerformerNameMetadataField
                                .setDescription(messageUtils.getMessage("uibot.activity.metadataField.performerName"));
                        prePerformerIdMetadataField = new MetadataField();
                        prePerformerIdMetadataField.setUiBot("Y");
                        prePerformerIdMetadataField.setName("activity__prePerformer_Id");
                        prePerformerIdMetadataField.setDataType("string");
                        prePerformerIdMetadataField
                                .setDescription(messageUtils.getMessage("uibot.activity.metadataField.performerName"));

                    }
                }
                dataTranceStatusMetadataField = new MetadataField();
                dataTranceStatusMetadataField.setUiBot("Y");
                dataTranceStatusMetadataField.setName("activity__approval__state");
                dataTranceStatusMetadataField.setDataType("string");
                dataTranceStatusMetadataField
                        .setDescription(messageUtils.getMessage("uibot.activity.metadataField.approvalStatus"));

                if (taskWithBacklogData.getBpmData() != null
                        && "waitting".equals(executeContext.getDataStateCode())
                        && taskWithBacklogData.getBpmData().containsKey("checkResult")) {

                    checkResultMetadataField = new MetadataField();
                    checkResultMetadataField.setArray(true);
                    checkResultMetadataField.setUiBot("Y");
                    checkResultMetadataField.setName("check_result");
                    if (ActivityConstants.TASK_DETAIL.equals(executeContext.getPageCode())
                            || ActivityConstants.PROJECT_DETAIL.equals(executeContext.getPageCode())) {
                        checkResultMetadataField.setTagDefinitions(CustomizeTagUtils.createOrderTagDefinitions("999"));
                    }
                    checkResultMetadataField.setDescription(messageUtils.getMessage("uibot.acitivity.check.title"));
                    checkResultMetadataField.setDataType("object");
                    List<MetadataField> subFields = new ArrayList<>();
                    MetadataField keyMetadataField = new MetadataField();
                    keyMetadataField.setUiBot("Y");
                    keyMetadataField.setArray(false);
                    keyMetadataField.setName("key");
                    keyMetadataField.setDataType("string");
                    subFields.add(keyMetadataField);
                    MetadataField nameMetadataField = new MetadataField();
                    nameMetadataField.setArray(false);
                    nameMetadataField.setUiBot("Y");
                    nameMetadataField.setName("name");
                    nameMetadataField.setDataType("string");
                    subFields.add(nameMetadataField);
                    MetadataField standardValueMetadataField = new MetadataField();
                    standardValueMetadataField.setArray(false);
                    standardValueMetadataField.setUiBot("Y");
                    standardValueMetadataField.setName("standardValue");
                    standardValueMetadataField.setDataType("string");
                    subFields.add(standardValueMetadataField);
                    MetadataField valueMetadataField = new MetadataField();
                    valueMetadataField.setArray(false);
                    valueMetadataField.setUiBot("Y");
                    valueMetadataField.setName("value");
                    valueMetadataField.setDataType("string");
                    subFields.add(valueMetadataField);
                    MetadataField dropValueMetadataField = new MetadataField();
                    dropValueMetadataField.setArray(false);
                    dropValueMetadataField.setUiBot("Y");
                    dropValueMetadataField.setName("dropValue");
                    dropValueMetadataField.setDataType("string");
                    subFields.add(dropValueMetadataField);
                    MetadataField yesOrNoMetadataField = new MetadataField();
                    yesOrNoMetadataField.setArray(false);
                    yesOrNoMetadataField.setUiBot("Y");
                    yesOrNoMetadataField.setName("yesOrNo");
                    yesOrNoMetadataField.setDataType("boolean");
                    subFields.add(yesOrNoMetadataField);
                    checkResultMetadataField.setSubFields(subFields);

                }
                break;
            }
            //如果是母子任务，则在每行数据上附加子任务的id和状态
            else if ("DERIVE".equals(executeContext.getPattern())
                    && taskWithBacklogData.getBpmData() != null
                    && taskWithBacklogData.getBpmData().containsKey("__DERIVE_TASK_RELATION")) {
                Map task = (HashMap) taskWithBacklogData.getBpmData().get("__DERIVE_TASK_RELATION");
                if (task != null && task.size() > 0) {
                    for (Object key : task.keySet()) {
                        MetadataField subTaskMetadataField = new MetadataField();
                        subTaskMetadataField.setUiBot("Y");
                        subTaskMetadataField.setName("uibot__" + key + "__subTask");
                        subTaskMetadataField.setDataType("object");
                        subTaskMetadataField.setDescription(key + "关联的子任务");
                        metadataFields.add(subTaskMetadataField);
                    }
                }
                break;
            }


        }
        if (null != prePerformerIdMetadataField) {
            metadataFields.add(prePerformerIdMetadataField);
        }
        if (null != prePerformerNameMetadataField) {
            metadataFields.add(prePerformerNameMetadataField);
        }
        if (null != dataTranceStatusMetadataField) {
            metadataFields.add(dataTranceStatusMetadataField);
        }
        if (null != checkResultMetadataField) {
            metadataFields.add(checkResultMetadataField);
        }
    }

    /**
     * 根据任务定义的mapKeys将对应的bpmData进行规整
     *
     * @param mapKeys                 任务定义的mapKeys
     * @param mainDataName            bpmData中表示查询入参的 keyName
     * @param taskWithBacklogDataList 从api查询出来的数据
     * @return 处理的集合
     */
    private Map<String, TaskWithBacklogData> dealBpmDataByMapKeys(List<String> mapKeys, String mainDataName, List<TaskWithBacklogData> taskWithBacklogDataList) {
        //返回的结果的索引对象
        Map<String, TaskWithBacklogData> backlogIndexMap = new HashMap<>(20);
        //循环处理data数据,执行该代码前，有前置条件对mapKeys以及data进行判空
        for (TaskWithBacklogData tmpData : taskWithBacklogDataList) {
            if (tmpData.getBpmData() != null && tmpData.getBpmData().containsKey(mainDataName) && !CollectionUtils.isEmpty(tmpData.getBacklog())) {
                List<Map> bpmDataList = (List<Map>) MapUtils.getObject(tmpData.getBpmData(), mainDataName);
                if (!CollectionUtils.isEmpty(bpmDataList)) {
                    bpmDataList.forEach(bpm -> {
                        String result = generateIndexKey(mapKeys, bpm);
                        //如果生成了indexKey，则将数据加入索引中
                        if (!StringUtils.isEmpty(result)) {
                            backlogIndexMap.put(result, tmpData);
                        }
                    });
                }
            }
        }
        return backlogIndexMap;
    }


    /**
     * 判断map集合中是否包含maoKeys的所有key，并且生成新的indexKey
     * add by gonghongxing at 2022/03/26  for 数据源合并处理优化
     *
     * @param mapKeys 条件key集合
     * @param map     元数据
     * @param <T>     类型
     * @return 索引值
     */
    private <T> String generateIndexKey(List<String> mapKeys, Map<String, T> map) {
        //用于计数，判断返回的数据是否都包含对应的mapKey
        int i = 0;
        for (String tempKey : mapKeys) {
            //如果包含，则计数+1
            if (map.containsKey(tempKey)) i++;
        }
        //如果计数值等于mapKeys长度，说明该返回数据符合条件
        StringBuilder indexKey = new StringBuilder();
        if (i == mapKeys.size()) {
            for (String tempKey : mapKeys) {
                indexKey.append(MapUtils.getString(map, tempKey));
            }
        }
        return indexKey.toString();
    }


    /**
     * 根据mappingKeys获取keys的相关定义
     *
     * @param mappingDTO 需要解析的bk定义对象
     * @return dataNameList mapping中定义的具体dataname
     */
    private List<String> parseMappingKeys(BusinessKeyMappingDTO mappingDTO) {
        List<String> dataNameList = new ArrayList<>();
        for (BusinessKeyMappingDTO childMapping : mappingDTO.getField()) {
            dataNameList.add(childMapping.getDataName());
        }
        return dataNameList;

    }

    /**
     * 创建数据流转状态列
     *
     * @return
     */
    private MetadataField createDataFlowStatusField() {
        MetadataField dataFlowStatusField = new MetadataField();
        dataFlowStatusField.setUiBot("Y");
        dataFlowStatusField.setName("activity__data__status");
        dataFlowStatusField.setDataType("object");
        dataFlowStatusField.setDescription(messageUtils.getMessage("uibot.activity.metadataField.dataFlowStatus"));
        return dataFlowStatusField;
    }

    /**
     * @param executeContext          ExecuteContext
     * @param dataList                List<Map<Object>>
     * @param taskWithBacklogIndexMap Map<TaskWithBacklogData>
     * @param keyList                 List<String>
     * @param detailFieldFinal        String
     * @return void
     * @description 处理流程数据，增加流程待办信息
     */
    public void dealResultData(ExecuteContext executeContext, List<Map<String, Object>> dataList, Map<String,
            TaskWithBacklogData> taskWithBacklogIndexMap, List<String> keyList, String detailFieldFinal) {
        //当前配置为非执行人，去除用户状态
        //perf:优化从根据ExecuteContext中，减少循环的次数，对数据对象的遍历以及比较操作
        boolean isPerformer = AtmcDataSourceService.isPerformer(executeContext.getSettings());

        final boolean IS_SOLVE_CATEGORY = "SOLVE".equals(executeContext.getCategory());

        final boolean IS_APPROVAL_CATEGORY = "APPROVAL".equals(executeContext.getCategory());

        final boolean IS_DERIVE_PATTERN = "DERIVE".equals(executeContext.getPattern());

        final boolean IS_IDENTITY_CHARGE = executeContext.getIdentity().equals("charge");

        final boolean IS_IDENTITY_PERFORMER = executeContext.getIdentity().equals("performer");
        //perf:优化从根据ExecuteContext中，减少循环的次数，对数据对象的遍历以及比较操作 end

        //循环任务，如果与查询结果的数据匹配，则给查询结果添加附加属性
        for (Map<String, Object> data : dataList) {
            String taskIndexKey = generateIndexKey(keyList, data);

            TaskWithBacklogData taskWithBacklogData = taskWithBacklogIndexMap.get(taskIndexKey);
            if (taskWithBacklogData == null) {
                continue;
            }
            //modify by gonghongxing at 2022/03/26  for 数据源合并处理优化 end
            if (!CollectionUtils.isEmpty(taskWithBacklogData.getBacklog())) {
                BacklogData firstBacklogData = taskWithBacklogData.getBacklog().get(0);

                if (!isPerformer) {
                    firstBacklogData.setPerformerState(null);
                }
                boolean hasPrePerformerId = firstBacklogData.getPrePerformerId() != null
                        || firstBacklogData.getPreAgentPerformerId() != null;

                //modify by gonghongxing at 2022/03/26  for 数据源合并处理优化 end
                firstBacklogData.setProcessSerialNumber(taskWithBacklogData.getProcessSerialNumber());
                firstBacklogData.setTaskUid(taskWithBacklogData.getTaskUid());

                // 数据流转状态data
                Map<String, Object> activityDataStatus = new HashMap<>();
                // 是否为已撤回数据
                activityDataStatus.put("withdraw", AtmcDataSourceService.isTaskWithdraw(firstBacklogData));
                activityDataStatus.put("returnInfo", ApprovalUtil.addReturnComments(firstBacklogData, data));
                data.put("activity__data__status", activityDataStatus);
                data.put("activity__backLog__data", firstBacklogData);
                data.put("activity__remainTime", firstBacklogData.getRemainTime());
                data.put("activity__performerName", firstBacklogData.getPerformerName());
                if (hasPrePerformerId) {
                    //region 执行人
                    if (firstBacklogData.getPrePerformerType() != null && firstBacklogData.getPrePerformerType() > 0) {
                        data.put("activity__prePerformer_Id", firstBacklogData.getPreAgentPerformerId());
                        data.put("activity__prePerformer_Name", firstBacklogData.getPreAgentPerformerName());
                    } else {
                        data.put("activity__prePerformer_Id", firstBacklogData.getPrePerformerId());
                        data.put("activity__prePerformer_Name", firstBacklogData.getPrePerformerName());
                    }
                    //endregion
                }
                //region 添加运营单元
                boolean hasBusinessUnit = Objects.nonNull(taskWithBacklogData.getBusinessUnit());
                //region 添加运营单元
                data.put("businessUnit", hasBusinessUnit ? taskWithBacklogData.getBusinessUnit() : executeContext.getBusinessUnit());
                //endregion
                data.put("activity__planEndTime", firstBacklogData.getPlanEndTime());
                data.put("activity__readCount", firstBacklogData.getReadCount());
                // 增加任务的字段到数据中(82848)
                ActivityUtils.addActivityData(data, firstBacklogData);
                if (IS_SOLVE_CATEGORY) {
                    SolveUtil.addTraceData(executeContext, detailFieldFinal, taskWithBacklogData, firstBacklogData, data);

                    //endregion
                } else if (IS_APPROVAL_CATEGORY && IS_IDENTITY_CHARGE) {
                    ApprovalUtil.addTaskData(executeContext, taskWithBacklogData, firstBacklogData, data);
                }
                //如果是母子任务，则在每行数据上附加子任务的id和状态
                else if (IS_DERIVE_PATTERN
                        && taskWithBacklogData.getBpmData() != null
                        && taskWithBacklogData.getBpmData().containsKey("__DERIVE_TASK_RELATION_UNIKEYS")
                        && taskWithBacklogData.getBpmData().containsKey("__DERIVE_TASK_RELATION")) {
                    DeriveUtil.addSubTaskData(taskWithBacklogData, data);
                }

                // 人工任务关联的签核信息
                if (!IS_APPROVAL_CATEGORY && IS_IDENTITY_PERFORMER && null != taskWithBacklogData.getRelationApproval()) {
                    ApprovalUtil.addTaskData(taskWithBacklogData, firstBacklogData, data);
                }

            }
        }
    }


}
