package com.digiwin.athena.atdm.datasource.datasource.process;

import com.digiwin.athena.appcore.util.MessageUtils;
import com.digiwin.athena.atdm.UiBotConstants;
import com.digiwin.athena.atdm.activity.ActivityConstants;
import com.digiwin.athena.atdm.datasource.datasource.DataSourceProcessService;
import com.digiwin.athena.atdm.datasource.datasource.DataSourceBase;
import com.digiwin.athena.atdm.datasource.domain.ApiMetadataCollection;
import com.digiwin.athena.atdm.datasource.domain.BacklogData;
import com.digiwin.athena.atdm.datasource.domain.DataSourceProcessor;
import com.digiwin.athena.atdm.datasource.domain.ExecuteContext;
import com.digiwin.athena.atdm.datasource.domain.MetadataField;
import com.digiwin.athena.atdm.datasource.domain.QueryResult;
import com.digiwin.athena.atdm.datasource.domain.TaskWithBacklogData;
import net.sf.json.JSONArray;
import net.sf.json.JSONObject;
import org.apache.commons.collections.MapUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

@Service("atmcDataService")
public class AtmcDataSourceService implements DataSourceProcessService {
    // 任务撤回createType
    private static final Integer CREATE_TYPE_WITHDRAW = 9;
    // 需要展示数据流转状态的createType集合
    private static final List<Integer> DATA_FLOW_STATUS_LIST = Arrays.asList(CREATE_TYPE_WITHDRAW);

    public static final String GROUP_ORDER_SETTING_KEY = "project-settings";

    @Autowired
    private MessageUtils messageUtils;

    /**
     * 对元数据处理
     *
     * @param dataSourceBase      查询的结果的数据
     * @param executeContext      执行上下文
     * @param dataSourceProcessor 处理器
     * @param queryResult         查询结果
     */
    @Override
    public void handelMetadata(DataSourceBase dataSourceBase, ExecuteContext executeContext,
                               DataSourceProcessor dataSourceProcessor, QueryResult queryResult) {
        if (!ActivityConstants.TASK_DETAIL.equals(executeContext.getPageCode())
                && !ActivityConstants.PROJECT_DETAIL.equals(executeContext.getPageCode())) {
            return;
        }
        addAtmcFiled(dataSourceBase, executeContext, dataSourceProcessor, queryResult.getApiMetadataCollection());
    }

    /**
     * 对数据处理
     *
     * @param dataSourceBase
     * @param executeContext
     * @param dataSourceProcessor 处理器
     * @param queryResult
     */
    @Override
    public void handelData(DataSourceBase dataSourceBase, ExecuteContext executeContext,
                           DataSourceProcessor dataSourceProcessor, QueryResult queryResult) {
        if (!ActivityConstants.TASK_DETAIL.equals(executeContext.getPageCode())
                && !ActivityConstants.PROJECT_DETAIL.equals(executeContext.getPageCode())) {
            return;
        }
        if (CollectionUtils.isEmpty(queryResult.getData())) {
            return;
        }

        if (Objects.isNull(dataSourceProcessor.getTaskWithBacklogData())) {
            return;
        }
        String detailField = null;
        if (dataSourceProcessor.getParas() != null) {
            Map<String, String> processParas = (Map<String, String>) dataSourceProcessor.getParas();
            if (processParas.containsKey("detailField")) {
                detailField = processParas.get("detailField");
            }
        }
        TaskWithBacklogData taskWithBacklogData = dataSourceProcessor.getTaskWithBacklogData();
        if (!CollectionUtils.isEmpty(taskWithBacklogData.getBacklog())) {
            BacklogData firstBacklogData = taskWithBacklogData.getBacklog().get(0);
            //当前配置为非执行人，去除用户状态
            boolean isPerformer =  AtmcDataSourceService.isPerformer(executeContext.getSettings());
            if(!isPerformer){
                firstBacklogData.setPerformerState(null);
            }

            boolean hasPrePerformerId = firstBacklogData.getPrePerformerId() != null
                    || firstBacklogData.getPreAgentPerformerId() != null;
            // 数据流转状态data
            Map<String, Object> activityDataStatus = new HashMap<>();
            // 是否为已撤回数据
            activityDataStatus.put("withdraw", AtmcDataSourceService.isTaskWithdraw(firstBacklogData));

            // 目前 把activity 相关数据统一存放
            for (Map<String, Object> data : queryResult.getData()) {
                firstBacklogData.setProcessSerialNumber(taskWithBacklogData.getProcessSerialNumber());
                firstBacklogData.setTaskUid(taskWithBacklogData.getTaskUid());
                data.put("activity__backLog__data", firstBacklogData);
                data.put("activity__remainTime", firstBacklogData.getRemainTime());
                data.put("activity__performerName", firstBacklogData.getPerformerName());
                if (hasPrePerformerId) {
                    if (firstBacklogData.getPrePerformerType() != null && firstBacklogData.getPrePerformerType() > 0) {
                        data.put("activity__prePerformer_Id", firstBacklogData.getPreAgentPerformerId());
                        data.put("activity__prePerformer_Name", firstBacklogData.getPreAgentPerformerName());
                    } else {
                        data.put("activity__prePerformer_Id", firstBacklogData.getPrePerformerId());
                        data.put("activity__prePerformer_Name", firstBacklogData.getPrePerformerName());
                    }
                }
                //添加运营单元
                if (taskWithBacklogData.getBusinessUnit() != null) {
                    data.put("businessUnit", taskWithBacklogData.getBusinessUnit());
                } else {
                    data.put("businessUnit", executeContext.getBusinessUnit());
                }
                data.put("activity__planEndTime", firstBacklogData.getPlanEndTime());
                data.put("activity__readCount", firstBacklogData.getReadCount());
                // 增加任务的字段到数据中(82848)
                ActivityUtils.addActivityData(data,firstBacklogData);
                if ("SOLVE".equals(executeContext.getCategory())) {
                    SolveUtil.addTraceData(executeContext, detailField, taskWithBacklogData, firstBacklogData, data);
                } else if ("APPROVAL".equals(executeContext.getCategory()) && executeContext.getIdentity().equals("charge")) {
                    ApprovalUtil.addTaskData(executeContext, taskWithBacklogData, firstBacklogData, data);
                }
                //如果是母子任务，则在每行数据上附加子任务的id和状态
                else if ("DERIVE".equals(executeContext.getPattern())
                        && taskWithBacklogData.getBpmData() != null
                        && taskWithBacklogData.getBpmData().containsKey("__DERIVE_TASK_RELATION_UNIKEYS")
                        && taskWithBacklogData.getBpmData().containsKey("__DERIVE_TASK_RELATION")) {
                    DeriveUtil.addSubTaskData(taskWithBacklogData, data);
                }
                // 添加数据流转状态data
                activityDataStatus.put("returnInfo", ApprovalUtil.addReturnComments(firstBacklogData,data));
                data.put("activity__data__status", activityDataStatus);

                // 人工任务关联的签核信息
                if (!"APPROVAL".equals(executeContext.getCategory()) && executeContext.getIdentity().equals("performer")
                        && null != taskWithBacklogData.getRelationApproval()) {
                    ApprovalUtil.addTaskData(taskWithBacklogData, firstBacklogData, data);
                }

            }
            if ("SOLVE".equals(executeContext.getCategory())
                    && "waitting".equals(executeContext.getDataStateCode())
                    && taskWithBacklogData.getBpmData() != null
                    && taskWithBacklogData.getBpmData().containsKey("checkResult")) {
                addCheckResult(dataSourceBase, queryResult, taskWithBacklogData);
            }
        }

    }



    /**
     * 添加异常排除的数据到结果中
     */
    private void addCheckResult(DataSourceBase dataSourceBase, QueryResult queryResult,
                                TaskWithBacklogData taskWithBacklogData) {
        // 如果有推送异常原因（checkResult），则添加到数据源里
        List<Map> checkResult = (List) taskWithBacklogData.getBpmData().get("checkResult");
        // 如果合并的数据源有dataKey，则通过这个key去主数据源中找数据
        if (!CollectionUtils.isEmpty(dataSourceBase.getDataKeys())) {
            StringBuilder stringBuilder = new StringBuilder();
            // 如果不一致，则需要遍历
            for (Map<String, Object> datum : checkResult) {
                stringBuilder.setLength(0);
                for (String key : dataSourceBase.getDataKeys()) {
                    if (datum.containsKey(key)) {
                        stringBuilder.append(StringUtils.isEmpty(datum.get(key)) ? "NAN" : datum.get(key).toString()).append(";");
                    } else {
                        stringBuilder.append("NAN;");
                    }
                }
                if (stringBuilder.length() > 0) {
                    String key = stringBuilder.toString();
                    Map<String, Object> findItem = queryResult.findByKey(key);
                    if (findItem != null) {
                        findItem.put("check_result", datum.get("result"));
                    }
                }
            }
        }
    }


    /**
     * 只有当责者的时候才出现ATMC数据 添加Activity字段
     *
     * @param apiMetadataCollection
     */
    private void addAtmcFiled(DataSourceBase dataSourceBase, ExecuteContext executeContext, DataSourceProcessor dataSourceProcessor,
                              ApiMetadataCollection apiMetadataCollection) {
        if (apiMetadataCollection == null
                || apiMetadataCollection.getMasterApiMetadata().getResponseFields().size() == 0) {
            return;
        }

        // 当前关的处理人
        MetadataField performerNameMetadataField = new MetadataField();
        performerNameMetadataField.setUiBot("Y");
        performerNameMetadataField.setName("activity__performerName");
        performerNameMetadataField.setDataType("string");
        performerNameMetadataField.setDescription(messageUtils.getMessage("uibot.activity.metadataField.performerName"));

        MetadataField planEndTimeMetadataField = new MetadataField();
        planEndTimeMetadataField.setUiBot("Y");
        planEndTimeMetadataField.setName("activity__planEndTime");
        planEndTimeMetadataField.setDataType("string");
        planEndTimeMetadataField.setDescription(messageUtils.getMessage("uibot.activity.metadataField.planEndTime"));

        MetadataField readCountMetadataField = new MetadataField();
        readCountMetadataField.setUiBot("Y");
        readCountMetadataField.setName("activity__readCount");
        readCountMetadataField.setDataType("string");
        readCountMetadataField.setDescription(messageUtils.getMessage("uibot.activity.metadataField.readCount"));

        MetadataField dataStatusMetadataField = new MetadataField();
        dataStatusMetadataField.setUiBot("Y");
        dataStatusMetadataField.setName("activity__backLog__data");
        dataStatusMetadataField.setDataType("object");
        dataStatusMetadataField.setDescription(messageUtils.getMessage("uibot.activity.metadataField.dataStatus"));
        List<MetadataField> metadataFields = apiMetadataCollection.getMasterApiMetadata().getResponseFields().get(0)
                .getSubFields();

        metadataFields.add(performerNameMetadataField);
        metadataFields.add(planEndTimeMetadataField);
        metadataFields.add(readCountMetadataField);
        metadataFields.add(dataStatusMetadataField);
        // 添加任务的元数据字段
        ActivityUtils.addActivityMetadataField(metadataFields,messageUtils);

        /**
         * p5s1 9100 添加“数据流转状态”列
         * 如果已经有“”
         */
        if (!UiBotConstants.ACTION_CATEGORY_SHELVED_DATA.equals(dataSourceBase.getType())) {
            metadataFields.add(createDataFlowStatusField());
        }

        TaskWithBacklogData taskWithBacklogData = dataSourceProcessor.getTaskWithBacklogData();
        if (taskWithBacklogData != null) {
            if ("SOLVE".equals(executeContext.getCategory()) || ("APPROVAL".equals(executeContext.getCategory()) && executeContext.getIdentity().equals("charge"))) {
                if (!CollectionUtils.isEmpty(taskWithBacklogData.getBacklog())) {
                    BacklogData firstBacklogData = taskWithBacklogData.getBacklog().get(0);
                    if (firstBacklogData.getPrePerformerId() != null
                            || firstBacklogData.getPreAgentPerformerId() != null) {
                        // 上一个的处理人id和name
                        MetadataField prePerformerNameMetadataField = new MetadataField();
                        prePerformerNameMetadataField.setUiBot("Y");
                        prePerformerNameMetadataField.setName("activity__prePerformer_Name");
                        prePerformerNameMetadataField.setDataType("string");
                        prePerformerNameMetadataField
                                .setDescription(messageUtils.getMessage("uibot.activity.metadataField.performerName"));
                        MetadataField prePerformerIdMetadataField = new MetadataField();
                        prePerformerIdMetadataField.setUiBot("Y");
                        prePerformerIdMetadataField.setName("activity__prePerformer_Id");
                        prePerformerIdMetadataField.setDataType("string");
                        prePerformerIdMetadataField
                                .setDescription(messageUtils.getMessage("uibot.activity.metadataField.performerName"));

                        metadataFields.add(prePerformerIdMetadataField);
                        metadataFields.add(prePerformerNameMetadataField);
                    }
                }
                MetadataField dataTranceStatusMetadataField = new MetadataField();
                dataTranceStatusMetadataField.setUiBot("Y");
                dataTranceStatusMetadataField.setName("activity__approval__state");
                dataTranceStatusMetadataField.setDataType("string");
                dataTranceStatusMetadataField
                        .setDescription(messageUtils.getMessage("uibot.activity.metadataField.approvalStatus"));
                metadataFields.add(dataTranceStatusMetadataField);
                if (taskWithBacklogData.getBpmData() != null
                        && "waitting".equals(executeContext.getDataStateCode())
                        && taskWithBacklogData.getBpmData().containsKey("checkResult")) {
                    MetadataField checkResultMetadataField = new MetadataField();
                    checkResultMetadataField.setArray(true);
                    checkResultMetadataField.setUiBot("Y");
                    checkResultMetadataField.setName("check_result");
                    if (ActivityConstants.TASK_DETAIL.equals(executeContext.getPageCode())
                            || ActivityConstants.PROJECT_DETAIL.equals(executeContext.getPageCode())) {
                        checkResultMetadataField.setTagDefinitions(CustomizeTagUtils.createOrderTagDefinitions("999"));
                    }
                    checkResultMetadataField.setDescription(messageUtils.getMessage("uibot.acitivity.check.title"));
                    checkResultMetadataField.setDataType("object");
                    List<MetadataField> subFields = new ArrayList<>();
                    MetadataField keyMetadataField = new MetadataField();
                    keyMetadataField.setUiBot("Y");
                    keyMetadataField.setArray(false);
                    keyMetadataField.setName("key");
                    keyMetadataField.setDataType("string");
                    subFields.add(keyMetadataField);
                    MetadataField nameMetadataField = new MetadataField();
                    nameMetadataField.setArray(false);
                    nameMetadataField.setUiBot("Y");
                    nameMetadataField.setName("name");
                    nameMetadataField.setDataType("string");
                    subFields.add(nameMetadataField);
                    MetadataField standardValueMetadataField = new MetadataField();
                    standardValueMetadataField.setArray(false);
                    standardValueMetadataField.setUiBot("Y");
                    standardValueMetadataField.setName("standardValue");
                    standardValueMetadataField.setDataType("string");
                    subFields.add(standardValueMetadataField);
                    MetadataField valueMetadataField = new MetadataField();
                    valueMetadataField.setArray(false);
                    valueMetadataField.setUiBot("Y");
                    valueMetadataField.setName("value");
                    valueMetadataField.setDataType("string");
                    subFields.add(valueMetadataField);
                    MetadataField dropValueMetadataField = new MetadataField();
                    dropValueMetadataField.setArray(false);
                    dropValueMetadataField.setUiBot("Y");
                    dropValueMetadataField.setName("dropValue");
                    dropValueMetadataField.setDataType("string");
                    subFields.add(dropValueMetadataField);
                    MetadataField yesOrNoMetadataField = new MetadataField();
                    yesOrNoMetadataField.setArray(false);
                    yesOrNoMetadataField.setUiBot("Y");
                    yesOrNoMetadataField.setName("yesOrNo");
                    yesOrNoMetadataField.setDataType("boolean");
                    subFields.add(yesOrNoMetadataField);
                    checkResultMetadataField.setSubFields(subFields);
                    metadataFields.add(checkResultMetadataField);
                }
            }
            //如果是母子任务，则在每行数据上附加子任务的id和状态
            else if ("DERIVE".equals(executeContext.getPattern())
                    && taskWithBacklogData.getBpmData() != null
                    && taskWithBacklogData.getBpmData().containsKey("__DERIVE_TASK_RELATION")
                    && taskWithBacklogData.getBpmData().containsKey("__DERIVE_TASK_RELATION_UNIKEYS")) {
                Map task = (HashMap) taskWithBacklogData.getBpmData().get("__DERIVE_TASK_RELATION");
                if (task != null && task.size() > 0) {
                    for (Object key : task.keySet()) {
                        MetadataField subTaskMetadataField = new MetadataField();
                        subTaskMetadataField.setUiBot("Y");
                        subTaskMetadataField.setName("uibot__" + key + "__subTask");
                        subTaskMetadataField.setDataType("object");
                        subTaskMetadataField.setDescription(key.toString());

                        metadataFields.add(subTaskMetadataField);
                    }
                }
            }
        }
    }

    /**
     * 创建数据流转状态列
     *
     * @return
     */
    private MetadataField createDataFlowStatusField() {
        MetadataField dataFlowStatusField = new MetadataField();
        dataFlowStatusField.setUiBot("Y");
        dataFlowStatusField.setName("activity__data__status");
        dataFlowStatusField.setDataType("object");
        dataFlowStatusField.setDescription(messageUtils.getMessage("uibot.activity.metadataField.dataFlowStatus"));
        return dataFlowStatusField;
    }

    /**
     * 是否为任务撤回
     *
     * @param backlogData
     * @return
     */
    public static int isTaskWithdraw(BacklogData backlogData) {
        if (!AtmcDataSourceService.isDataFlowStatusCreateType(backlogData)) {
            return 0;
        }

        Integer createType = (Integer) (backlogData.getWorkitemList().get(0).get("createType"));
        return CREATE_TYPE_WITHDRAW.equals(createType) ? 1 : 0;
    }

    /**
     * 处理退回意见
     * @param backlogData ptm返回数据
     * @param data  应用返回的业务数据
     * @return
     */
    public static Map<String, Object> dealReturnComment(BacklogData backlogData, List<Map<String, Object>> data) {
        List<Map<String,Object>> workitemList = backlogData.getWorkitemList();
        for(Map workitem : workitemList){
            List<Map<String,Object>> returnWorkitem = (List<Map<String,Object>>) workitem.get("returnWorkitem");
        }
        return null;
    }

    /**
     * 是否为当责者分组
     * @param settings
     * @return
     */
    public static boolean   isPerformer(JSONObject settings) {
        boolean isPerformer = false;
        if (null == settings || !settings.containsKey(GROUP_ORDER_SETTING_KEY)) {
            return true;
        }
        JSONObject setObject = settings.getJSONObject(GROUP_ORDER_SETTING_KEY);
        JSONArray operations = JSONArray.fromObject(setObject.get("options"));
        if(operations.size() > 0){
            for (Object obj : operations){
                JSONObject groupObj = JSONObject.fromObject(obj);
                if(groupObj.containsKey("value") && Objects.equals("activity__performerName",groupObj.getString("value"))){
                    isPerformer = true;
                    break;
                }
            }
        }
        return isPerformer;
    }

    /**
     * 是否为经过数据流转（比如：终止、撤回、退回等）的数据
     *
     * @param backlogData
     * @return
     */
    public static boolean isDataFlowStatusCreateType(BacklogData backlogData) {
        if (null == backlogData || CollectionUtils.isEmpty(backlogData.getWorkitemList())) {
            return false;
        }

        Map<String, Object> workItem = backlogData.getWorkitemList().get(0);
        if (MapUtils.isEmpty(workItem) || null == workItem.get("createType")) {
            return false;
        }

        Integer createType = (Integer) workItem.get("createType");
        return DATA_FLOW_STATUS_LIST.contains(createType);
    }

}
